// ----------------------------------------------------------------------------
// Read processed Bruker data and write UCSF format.
//

#include <iostream>		// Use cerr
#include <stdlib.h>		// Use exit()

#include "memcache.h"		// use Memory_Cache
#include "nmrdata.h"		// Use standard_nucleus_name()
#include "stringc.h"		// Use Stringy
#include "ucsffile.h"		// Use write_ucsf_nmr_data()
#include "brukerfile.h"		// Use varian_nmr_data()

#define MEMORY_CACHE_SIZE 8000000

// ----------------------------------------------------------------------------
//
static char *usage = 
"Syntax: bruk2ucsf <bruker-file> <ucsf-file>\n"
"\n"
"Here are examples used for converting 2D or 3D data:\n"
"\n"
"	% bruk2ucsf 1/pdata/1/2rr noesy150.ucsf\n"
"	% bruk2ucsf 1/pdata/1/3rrr hnca.ucsf\n"
"\n"
"Only the real component of the processed data is used by Sparky.\n"
"The dimension of the data is ascertained by the number in the Bruker\n"
"file name.  So that name should not be changed.  Also parameters for\n"
"each spectrum axis are extracted from the proc, proc2s, proc3s files\n"
"in the same directory, and from the acqus, acqu2s, acqu3s files in\n"
"in the directory above.  The path to the Bruker data file should not\n"
"be a symbolic link since paths to the acqus and procs parameter files\n"
"like symbolic-link/../procs will not be correct.  For more information\n"
"see the Sparky documentation manual/files.html.\n";

// ----------------------------------------------------------------------------
//
int main(int argc, char **argv)
{
  if (argc < 3)
    { std::cerr << usage; exit(1); }

  Stringy bruker_data_path = argv[1];
  Stringy ucsf_data_path = argv[2];

  Stringy error_msg;
  Memory_Cache mcache(MEMORY_CACHE_SIZE);
  NMR_Data *bruker_data = bruker_nmr_data(bruker_data_path, &error_msg,
					  &mcache);
  if (bruker_data == NULL)
    { std::cerr << "bruk2ucsf: " << error_msg << std::endl; exit(1); }

  if (!write_ucsf_nmr_data(bruker_data, ucsf_data_path, &error_msg, &mcache))
    { std::cerr << "bruk2ucsf: " << error_msg << std::endl; exit(1); }

  return 0;
}
