// ----------------------------------------------------------------------------
// Read NMR data processed using Bruker software.
//
//
// Read the acqus, acqu2s, acqu3s, ... files from the experiment directory
// Read the procs, proc2s, proc3s, ... files from the processing directory
// There can be many processing directories.
// Work from the data path (eg. 1/pdata/1/2rr) to find other files.
// This won't work with symbolic link to data file.  But can use
// symbolic link to experiment directory.
//
// procNs variables:
//
// OFFSET = ppm value of first data point (8.5923)
// SF = spectrometer freq in MHz (600.137904543)
// SI = size of full matrix (1024)
// XDIM = size of a block (128)
// SW_p = (processed?) sweep width in Hz (4587.155342)
// BYTORDP = byte order for processed data? (0 or 1?)
//
// acquNs variables:
//
// BYTORDA = little or big endian byte order for fid? (0 or 1)
// NUCLEUS = nucleus name (1H)
// BF1 = base frequency in Mhz (600.13)
// O1 = frequency offset (10780.088514025)
// SFO1 = spectrometer frequency = BF1 + O1 (600.140780088514)
// Not clear when I might need to use SFO2 or SFO3 or SFO4.
// SW = sweep width in ppm (7.63434535)
// SW_h = sweep width in Hz = SFO1 * SW (4587.155342)
//

#include <fstream>		// Use ifstream
#include <stdio.h>		// Use FILE *
#include <stdlib.h>		// Use atoi()
#include <string.h>		// Use strchr()

#include "binaryIO.h"		// Use read_s32(), read_f32(), ...
#include "blockfile.h"		// Use Block_File
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "memcache.h"		// use Memory_Cache
#include "nmrdata.h"		// Use NMR_Data
#include "spoint.h"		// Use IPoint, SPoint
#include "subarray.h"		// use position_index()
#include "stringc.h"		// Use Stringy
#include "system.h"		// Use file_exists(), file_path()
#include "utility.h"		// Use fatal_error()

// ----------------------------------------------------------------------------
//
class Bruker_Block_File : public Block_File
{
public:
  Bruker_Block_File(const Stringy &path, bool little_endian,
		    const IPoint &size, const IPoint &block_size,
		    Memory_Cache *);
  virtual ~Bruker_Block_File();

  virtual bool read_block(const IPoint &block, float *values);
  virtual bool write_block(const IPoint &block, float *values);
private:
  FILE *fp;
  bool little_endian;

  offset_type block_location(const IPoint &block);
};

// ----------------------------------------------------------------------------
//
class Bruker_Params
{
public:
  Bruker_Params(const Stringy &path);
  ~Bruker_Params();

  bool int_value(const Stringy &tag, int *i) const;
  bool double_value(const Stringy &tag, double *d) const;
  bool string_value(const Stringy &tag, Stringy *s) const;

private:
  List tags, values;

  bool find(const Stringy &tag, Stringy *value) const;
};

// ----------------------------------------------------------------------------
//
static bool read_bruker_parameters(const Stringy &data_path,
				   bool *little_endian,
				   IPoint *size, IPoint *block_size,
				   SPoint *spectrometer_freq,
				   SPoint *spectral_width,
				   SPoint *origin_ppm,
				   List *nucleus_names,
				   Stringy *error_msg);
static void bruker_paths(const Stringy &data_path,
			 Stringy *experiment_directory,
			 Stringy *processing_directory,
			 int *dim);
static bool unquote(const Stringy &quoted, Stringy *unquoted);
static bool read_parameter_entry(std::istream &strm,
				 Stringy *tag, Stringy *value);
static bool bruker_byte_order(const Stringy &processing_directory,
			      bool *little_endian, Stringy *error_msg);
static bool read_axis_parameters(const Stringy &experiment_directory,
				 const Stringy &processing_directory,
				 int dim, int axis,
				 int *size, int *block_size,
				 double *spectrometer_freq,
				 double *spectral_width,
				 double *origin_ppm,
				 Stringy *nucleus_name,
				 Stringy *error_msg);

// ----------------------------------------------------------------------------
//
bool is_bruker_nmr_data(const Stringy &data_path)
{
  int dim;
  Stringy experiment_directory, processing_directory;

  bruker_paths(data_path, &experiment_directory, &processing_directory, &dim);

  Stringy acqus = file_path(experiment_directory, "acqus");
  Stringy procs = file_path(processing_directory, "procs");

  return file_exists(data_path) && file_exists(acqus) && file_exists(procs);
}

// ----------------------------------------------------------------------------
//
NMR_Data *bruker_nmr_data(const Stringy &data_path, Stringy *error_msg,
			  Memory_Cache *mcache)
{
  bool little_endian;
  IPoint size, block_size;
  SPoint spectrometer_freq, spectral_width, origin_ppm;
  List nucleus_names;
  NMR_Data *nmrdata;

  if (read_bruker_parameters(data_path,
			     &little_endian, &size, &block_size,
			     &spectrometer_freq, &spectral_width,
			     &origin_ppm, &nucleus_names, error_msg))
    {
      Block_File *bf = new Bruker_Block_File(data_path, little_endian,
					     size, block_size, mcache);
      
      nmrdata = block_file_nmr_data(bf, spectrometer_freq,
				    spectral_width, origin_ppm, nucleus_names);
    }
  else
    nmrdata = NULL;

  return nmrdata;
}

// ----------------------------------------------------------------------------
//
static bool read_bruker_parameters(const Stringy &data_path,
				   bool *little_endian,
				   IPoint *size, IPoint *block_size,
				   SPoint *spectrometer_freq,
				   SPoint *spectral_width,
				   SPoint *origin_ppm,
				   List *nucleus_names,
				   Stringy *error_msg)
{
  int dim;
  Stringy experiment_directory, processing_directory;
  bruker_paths(data_path, &experiment_directory, &processing_directory, &dim);

  if (!bruker_byte_order(processing_directory, little_endian, error_msg))
    return false;

  IPoint sz(dim), bsize(dim);
  SPoint sfreq(dim), swidth(dim), orig(dim);
  List nucnames;

  for (int a = 0 ; a < dim ; ++a)
    {
      nucnames.append(new Stringy());
      if (!read_axis_parameters(experiment_directory, processing_directory,
				dim, a,
				&sz[a], &bsize[a], &sfreq[a], &swidth[a],
				&orig[a], (Stringy *) nucnames[a],
				error_msg))
	{
	  free_string_list_entries(nucnames);
	  return false;
	}
    }

  *size = sz;
  *block_size = bsize;
  *spectrometer_freq = sfreq;
  *spectral_width = swidth;
  *origin_ppm = orig;
  *nucleus_names = nucnames;

  return true;
}

// ----------------------------------------------------------------------------
//
static void bruker_paths(const Stringy &data_path,
			 Stringy *experiment_directory,
			 Stringy *processing_directory,
			 int *dim)
{
  *dim = atoi(file_name(data_path).cstring());
  *processing_directory = file_directory(data_path);
  Stringy processing_directories = parent_directory(*processing_directory);
  *experiment_directory = parent_directory(processing_directories);
}

// ----------------------------------------------------------------------------
//
Bruker_Params::Bruker_Params(const Stringy &path)
{
  std::ifstream bp(path.cstring());

  Stringy tag, value;
  while (read_parameter_entry(bp, &tag, &value))
    {
      tags.append(new Stringy(tag));
      values.append(new Stringy(value));
    }
}

// ----------------------------------------------------------------------------
//
Bruker_Params::~Bruker_Params()
{
  free_string_list_entries(tags);
  free_string_list_entries(values);
}

// ----------------------------------------------------------------------------
//
bool Bruker_Params::int_value(const Stringy &tag, int *i) const
  { Stringy v; return find(tag, &v) && sscanf(v.cstring(), "%d", i) == 1; }
bool Bruker_Params::double_value(const Stringy &tag, double *d) const
  { Stringy v; return find(tag, &v) && sscanf(v.cstring(), "%lf", d) == 1; }
bool Bruker_Params::string_value(const Stringy &tag, Stringy *s) const
  { Stringy v; return find(tag, &v) && unquote(v, s); }

// ----------------------------------------------------------------------------
//
bool Bruker_Params::find(const Stringy &tag, Stringy *value) const
{
  for (int t = 0 ; t < tags.size() ; ++t)
    if (*(Stringy *) tags[t] == tag)
      {
	*value = *(Stringy *) values[t];
	return true;
      }
  return false;
}

// ----------------------------------------------------------------------------
//
static bool unquote(const Stringy &quoted, Stringy *unquoted)
{
  const char *q = quoted.cstring();
  const char *q1 = strchr(q, '<');
  const char *q2 = strrchr(q, '>');
  if (q1 && q2 && q1 != q2)
    *unquoted = substring(q1 + 1, q2);

  return q1 && q2 && q1 != q2;
}

// ----------------------------------------------------------------------------
// Parameter lines look like:
//
//	##$FREQ= 600.123
//	##$NUCLEUS= <1H>
//
// Values can be on multiple lines but I don't handle this.
// They can be arrays and I don't handle that either.
//
static bool read_parameter_entry(std::istream &strm,
				 Stringy *tag, Stringy *value)
{
  Stringy line;
  while (stream_line(strm, &line))
    if (line.length() >= 3 &&
	line[0] == '#' && line[1] == '#' && line[2] == '$')
      {
	const char *cline = line.cstring();
	const char *end_of_tag = strchr(cline, '=');
	*tag = substring(cline+3, end_of_tag);
	*value = end_of_tag+1;
	return true;
      }

  return false;
}

// ----------------------------------------------------------------------------
//
static bool bruker_byte_order(const Stringy &processing_directory,
			      bool *little_endian, Stringy *error_msg)
{
  Stringy procs = file_path(processing_directory, "procs");
  Bruker_Params proc_params(procs);

  int order;
  if (!proc_params.int_value("BYTORDP", &order))
    {
      *error_msg = Stringy("Couldn't find parameter BYTORDP in " + procs);
      return false;
    }
  *little_endian = !order;

  return true;
}

// ----------------------------------------------------------------------------
//
static bool read_axis_parameters(const Stringy &experiment_directory,
				 const Stringy &processing_directory,
				 int dim, int axis,
				 int *size, int *block_size,
				 double *spectrometer_freq,
				 double *spectral_width,
				 double *origin_ppm,
				 Stringy *nucleus_name,
				 Stringy *error_msg)
{
  Stringy axis_digit =
    (axis == dim-1 ? Stringy("") : formatted_string("%d", dim-axis));
  Stringy acqus = formatted_string("%s/acqu%ss",
				   experiment_directory.cstring(),
				   axis_digit.cstring());
  Stringy procs = formatted_string("%s/proc%ss",
				   processing_directory.cstring(),
				   axis_digit.cstring());
  Bruker_Params acqu_params(acqus);
  Bruker_Params proc_params(procs);

  if (!acqu_params.string_value("NUC1", nucleus_name) &&
      !acqu_params.string_value("NUCLEUS", nucleus_name))
    *error_msg = Stringy("Couldn't find parameters NUC1, NUCLEUS in ") + acqus;
  else if (!proc_params.int_value("SI", size))
    *error_msg = Stringy("Couldn't find parameter SI in ") + procs;
  else if (!proc_params.int_value("XDIM", block_size))
    *error_msg = Stringy("Couldn't find parameter XDIM in ") + procs;
  else if (!acqu_params.double_value("SFO1", spectrometer_freq))
    *error_msg = Stringy("Couldn't find parameter SFO1 in ") + acqus;
  else if (!proc_params.double_value("SW_p", spectral_width))
    *error_msg = Stringy("Couldn't find parameter SW_p in ") + procs;
  else if (!proc_params.double_value("OFFSET", origin_ppm))
    *error_msg = Stringy("Couldn't find parameter OFFSET in ") + procs;
  else
    {
      *nucleus_name = standard_nucleus_name(*nucleus_name);
      return true;
    }

  return false;
}

// ----------------------------------------------------------------------------
//
Bruker_Block_File::Bruker_Block_File(const Stringy &path,
				     bool little_endian,
				     const IPoint &size,
				     const IPoint &block_size,
				     Memory_Cache *mcache) :
  Block_File(path, size, block_size, mcache)
{
  fp = fopen(path.cstring(), "rb");
  if (!fp)
    fatal_error("Bruker_Block_File::Failed opening: %s", path.cstring());

  this->fp = fp;
  this->little_endian = little_endian;
}

// ----------------------------------------------------------------------------
//
Bruker_Block_File::~Bruker_Block_File()
{
  flush_cache();
  fclose(fp);
  fp = NULL;
}

// ----------------------------------------------------------------------------
// Need to convert Bruker integer data to floats and reverse byte order
// if file is stored in little endian format.
//
bool Bruker_Block_File::read_block(const IPoint &block, float *values)
{
  offset_type size = block_volume();
  offset_type block_num = position_index(block, block_space(), true);
  offset_type position = block_num * size * sizeof(int);

  int *ints = new int [size];
  bool read = (seek_position(fp, position, 0) &&
	       read_s32_array(ints, size, !little_endian, fp));

  if (read)
    for (int k = 0 ; k < size ; ++k)
      values[k] = (float) ints[k];

  delete [] ints;

  return read;
}

// ----------------------------------------------------------------------------
//
bool Bruker_Block_File::write_block(const IPoint &, float *)
{
  return false;		// Not implemented.
}
