// ----------------------------------------------------------------------------
//
#include <ctype.h>		// use isdigit()
#include <math.h>		// Use fabs()
#include <stdlib.h>		// Use atof()
#include <string.h>		// use strlen(), strcpy()

#include "atom.h"		// use Atom
#include "condition.h"		// Use Condition
#include "crosspeak.h"
#include "group.h"
#include "label.h"
#include "mathconst.h"		// use MC_SQRT2
#include "molecule.h"
#include "notifier.h"		// use Notifier
#include "num.h"
#include "peak.h"
#include "peakgp.h"
#include "resonance.h"
#include "spectrum.h"		// use Spectrum
#include "utility.h"		// use fatal_error()

static int resonance_compare(const CrossPeak *xp1, const CrossPeak *xp2,
			     int axis);

// ----------------------------------------------------------------------------
//
CrossPeak::CrossPeak(Spectrum *sp, const SPoint &pos) : Ornament(sp)
{
	int dim = sp->dimension();

	mPosition		= pos;
	mFrequency		= pos;
	mAlias			= SPoint(dim);		// no alias

	for (int i = 0; i < dim; i++)
		mResonances[i] = NULL;

	mLabel = NULL;
	mVolumeError		= 0;
}

// ----------------------------------------------------------------------------
//
CrossPeak::~CrossPeak(void)
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_delete_crosspeak, this);

	//
	// If I'm linked to a Label, tell it it no longer has a parent
	//
	if (label()) {
		label()->attach(NULL);
	}

	//
	// If I've been linked to a Resonance, I have to remove the link
	//
	for (int i = 0; i < dimension(); i++)
	  SetResonance(i, NULL);
}

// ----------------------------------------------------------------------------
//
Ornament_Type CrossPeak::type() const
  { return (Ornament_Type) -1; }
const char *CrossPeak::type_name() const { return "crosspeak"; }

// ----------------------------------------------------------------------------
//
Label *CrossPeak::label() const { return mLabel; }
void CrossPeak::label(Label *lbl)
{
  if (lbl != NULL && mLabel != NULL)
    fatal_error("CrossPeak::label(): Second crosspeak label.\n");
  mLabel = lbl;
}

// ----------------------------------------------------------------------------
//
SPoint CrossPeak::position() const
  { return mPosition; }

void CrossPeak::set_position(const SPoint &r)
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_change_ornament, (Ornament *) this);
  if (label())	// Erase pointer to peak.
    n.send_notice(nt_will_change_ornament, (Ornament *) label());

  mPosition = r;
  update_frequency();

  n.send_notice(nt_changed_ornament, (Ornament *) this);
  if (label())
    n.send_notice(nt_changed_ornament, (Ornament *) label());

  spectrum()->save_file().NeedsSaving(true);
}

// ----------------------------------------------------------------------------
//
void CrossPeak::update_frequency()
{
  mFrequency = position() + alias();
  update_resonances();
}

/*
 * Let the resonances know their cached frequencies are bad.
 */
void CrossPeak::update_resonances() const
{
	for (int i = 0; i < dimension(); i++)
		if (resonance(i))
			resonance(i)->peak_moved(this);
}

SPoint CrossPeak::frequency() const
  { return mFrequency; }

double CrossPeak::frequency(int a) const
{
	return mFrequency[a];
}

Resonance *CrossPeak::resonance(int a) const
{
	return mResonances[a];
}

void CrossPeak::SetResonance(int a, Resonance *newrp)
{
	Resonance	*oldrp = resonance(a);

	if (newrp != oldrp) {

		mResonances[a] = newrp;

		if (oldrp) 
			oldrp->remove_assignment(this);
		if (newrp)
			newrp->add_assignment(this);

		Notifier &n = spectrum()->notifier();
		n.send_notice(nt_changed_ornament, (Ornament *) this);

		Label	*lp = label();
		if (lp)
			lp->AssignmentChanged();

		spectrum()->save_file().NeedsSaving(true);
	}
}

Resonance *CrossPeak::ChangeAssignment(int a,
				       const Stringy &atom,
				       const Stringy &group)
{
  Spectrum *sp = spectrum();
  Condition *c = sp->condition();
  Resonance *newrp = c->define_resonance(group, atom, sp->nucleus_type(a));

  SetResonance(a, newrp);

  return newrp;
}

bool CrossPeak::IsAssigned() const
{
	for (int i = 0; i < dimension(); i++)
		if (resonance(i))
			return true;
	return false;
}

// ----------------------------------------------------------------------------
//
bool CrossPeak::is_fully_assigned() const
{
  for (int a = 0 ; a < dimension(); ++a)
    if (resonance(a) == NULL)
      return false;
  return true;
}

//
// Delete all assignments from the crosspeak. 
//
void CrossPeak::DeleteAssignment()
{
	for (int i = 0; i < dimension(); i++)
		SetResonance(i, NULL);
}


/*
 * Return the assignment name of this CrossPeak.
 */
Stringy CrossPeak::assignment_name() const
  { return assignment_name(spectrum()->GetDefaultAssignmentFormat()); }
#define MAX_LINE_LENGTH 4096
Stringy CrossPeak::assignment_name(const Stringy &format) const
{
	/*
	 * Format the assignment.
	 */
	const char	*fmt = format.cstring();
	char		buf[MAX_LINE_LENGTH];
	Group		*lastGroup = NULL;
	char	*bp;
	int	i = -1;

	bp = buf;
	while (fmt && *fmt) {

		//
		// Normal characters get pushed through
		//
		if (fmt[0] != '%') {
			*bp++ = *fmt++;
			continue;
		}

		//
		// Skip the '%'
		//
		fmt++;

		//
		// If there is '%%', put one '%' in the output
		// and continue.
		//
		if (fmt[0] == '%') {
			*bp++ = *fmt++;
			continue;
		}

		/*
		 * '%A<n>' is the full atom name and '%a<n>' is the
		 * smart name (the group is only given if it differs
		 * from the previous group). <n> the frequency axis.
		 */
		if (fmt[0] == 'A' || fmt[0] == 'a') {
			int		longAtom = (fmt[0] == 'A');

			//
			// Get the resonance number. If the resonace
			// number isn't given, go to the next number
   			// after the current one.
			//
			fmt++;
			if (isdigit(fmt[0]))
				i = *fmt++ - '1';
			else
				i++;

			if (0 <= i && i < dimension()) {
			    if (resonance(i)) {
				Atom *ap = resonance(i)->atom();

				//
				// Must use long atom names if the
				// group has changed.
				//
				if (lastGroup != ap->group())
					longAtom = true;

				strcpy(bp, longAtom ?
				       ap->LongName().cstring() :
				       ap->name().cstring());
				bp = bp + strlen(bp);
				lastGroup = ap->group();
			    }
			    else {
				*bp++ = '?';
				lastGroup = NULL;
			    }
			}
		}

		/*
		 * '%G' and '%g' are both the group name.
		 */
		else if (fmt[0] == 'G' || fmt[0] == 'g') {
			//
			// Get the resonance number. If the resonance
			// number isn't given, go to the next number
			// after the current one.
			//
			fmt++;
			if (isdigit(fmt[0]))
				i = *fmt++ - '1';
			else
				i++;

			if (0 <= i && i < dimension()) {
			    if (resonance(i)) {
				lastGroup = resonance(i)->group();
				if (lastGroup) {
				  strcpy(bp, lastGroup->name().cstring());
				  bp = bp + strlen(bp);
				}
			    }
			    else {
				*bp++ = '?';
				lastGroup = NULL;
			    }
			}
		}
	}
	*bp = '\0';

	return buf;
}

/*
 * Return the user name of this CrossPeak.
 */
Stringy CrossPeak::user_name() const
{
  Label *lbl = label();

  return (lbl && !lbl->is_assignment()) ? lbl->GetString() : Stringy("");
}

////////////////////////////////////////////////////////////
//
//		Miscellaneous
//
////////////////////////////////////////////////////////////

SPoint CrossPeak::alias() const
  { return mAlias; }

void CrossPeak::set_alias(const SPoint &alias)
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_change_ornament, (Ornament *) this);
  mAlias = alias;
  update_frequency();
  n.send_notice(nt_changed_ornament, (Ornament *) this);
}
#define ALIAS_NEAR_ZERO 1e-4		// ppm
bool CrossPeak::IsAliased() const
{
  for (int a = 0 ; a < mAlias.dimension() ; ++a)
    if (fabs(mAlias[a]) > ALIAS_NEAR_ZERO)
      return true;

  return false;
}

bool CrossPeak::IsAliased(int a) const
{
  return fabs(mAlias[a]) > ALIAS_NEAR_ZERO;
}

bool CrossPeak::volume(double *, Stringy *) const
  { return false; }

// ----------------------------------------------------------------------------
//
bool CrossPeak::volume_error(double *fraction, Stringy *method) const
{
  if (mVolumeError > 0)
    {
      if (fraction)
	*fraction = mVolumeError;
      if (method)
	*method = mVolumeErrorMethod;
      return true;
    }
  return false;
}

// ----------------------------------------------------------------------------
//
void CrossPeak::set_volume_error(double fraction, const Stringy &method)
{
  mVolumeError = fraction;
  mVolumeErrorMethod = method;
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_changed_ornament, (Ornament *) this);
  if (is_peaklet(this))
    ((Peak *) this)->peakgp()->set_volume_error(0, "");
}

/*
 * Return < 0, 0 or > 0 if the assigned name of xp1 is less than,
 * equal to or greater than the assigned name of xp2, using the
 * give axis as a primary sort field.followed by the other axes in
 * order from w1 to wN.
 */
int compare_assignments(const CrossPeak *xp1, const CrossPeak *xp2, int axis)
{
	int compare = resonance_compare(xp1, xp2, axis);
	if (compare)
	  return compare;

	int dim = xp1->dimension();
	for (int a = 0; a < dim; a++)
	  if (a != axis)
	    {
	      compare = resonance_compare(xp1, xp2, a);

	      if (compare != 0)
		return compare;
	    }

	return 0;
}

// ----------------------------------------------------------------------------
//
static int resonance_compare(const CrossPeak *xp1, const CrossPeak *xp2,
			     int axis)
{
  Resonance *rp1 = (axis < xp1->dimension() ?
		    xp1->resonance(axis) : (Resonance *) NULL);
  Resonance *rp2 = (axis < xp2->dimension() ?
		    xp2->resonance(axis) : (Resonance *) NULL);

  if (rp1 == NULL && rp2 != NULL)
    return -1;
  if (rp1 != NULL && rp2 == NULL)
    return 1;
  if (rp1 != NULL && rp2 != NULL)
    return compare_resonance_names(rp1, rp2);
  
  return 0;
}

// ----------------------------------------------------------------------------
//
Rectangle CrossPeak::marker_square(ODraw &dr) const
{
  double x, y, w, h;

  dr.xy_point(position(), &x, &y);
  w = (type() == -1 ? 0 : dr.ornament_size(type(), dr.axis(X)));
  h = (type() == -1 ? 0 : dr.ornament_size(type(), dr.axis(Y)));

  return Rectangle(x - w/2, y - h/2, x + w/2, y + h/2);
}

// ----------------------------------------------------------------------------
//
SRegion CrossPeak::selection_region(ODraw &dr) const
{
  Rectangle m = marker_square(dr);

  if (volume())
    m.scale(MC_SQRT2);	// Make room for cirle drawn around X

  SRegion r(dr.axis(X), m.min(X), m.max(X),
	    dr.axis(Y), m.min(Y), m.max(Y),
	    position());

  add_selection_padding(&r, dr);

  return r;
}

// ----------------------------------------------------------------------------
//
SRegion CrossPeak::erasure_region(ODraw &dr) const
  { return selection_region(dr); }

// ----------------------------------------------------------------------------
// Draw a peak.
//
void CrossPeak::display(ODraw &dr) const
{
  Rectangle r = marker_square(dr);

  dr.set_drawing_color(GetColor());

  int plane = dr.in_plane(position());
  if (plane == 0)			// Peak in view plane
    {
      dr.draw_line(r.min(X), r.min(Y), r.max(X), r.max(Y));
      dr.draw_line(r.min(X), r.max(Y), r.max(X), r.min(Y));
    }
  else if (plane == 1)			// Peak above view plane
    {
      dr.draw_line(r.min(X), r.min(Y), r.center(X), r.center(Y));
      dr.draw_line(r.center(X), r.center(Y), r.max(X), r.min(Y));
    }
  else if (plane == -1)			// Peak above view plane
    {
      dr.draw_line(r.center(X), r.center(Y), r.max(X), r.max(Y));
      dr.draw_line(r.min(X), r.max(Y), r.center(X), r.center(Y));
    }

  if (volume())
    {
      r.scale(MC_SQRT2);
      dr.draw_ellipse(r);
    }

  Ornament::display(dr);	// Handle selection highlighting.
}

// ---------------------------------------------------------------------------
//
bool is_crosspeak(Ornament *op)
  { return op->type() == peak || op->type() == peakgroup; }

// ---------------------------------------------------------------------------
//
bool is_peaklet(Ornament *op)
  { return op->type() == peak && ((Peak *)op)->peakgp() != NULL; }

