// ----------------------------------------------------------------------------
// Maintain mapping between C objects and id numbers.
//
// This is used by undo eg. to restore a deleted label's connection to
// an existing peak.  And it is used by Python.  Python calls use id
// numbers to reference Sparky objects.
//

#include "list.h"		// use List
#include "memalloc.h"		// use new()
#include "objectid.h"		// use Object_Table
#include "stringc.h"		// Use Stringy
#include "table.h"		// Use Table
#include "utility.h"		// Use fatal_error()

// ----------------------------------------------------------------------------
//
class typed_pointer
{
public:
  typed_pointer(void *pointer, const Stringy &type)
    { this->pointer = pointer; this->type = type; }
  void *pointer;
  Stringy type;
};

static bool equal_objects(TableKey k1, TableKey k2);
unsigned long object_hash(TableKey key);

// ----------------------------------------------------------------------------
//
Object_Table::Object_Table()
  : id_to_object_table(),
    object_to_id_table(equal_objects, object_hash)
{
  next_object_id = 0;
}

// ----------------------------------------------------------------------------
//
int Object_Table::id(void *pointer, const Stringy &type)
{
  int id;
  TableData table_id;
  typed_pointer j(pointer, type);

  if (object_to_id_table.find((TableKey) &j, &table_id))
      id = (int) (long) table_id;
  else
    {
      id = next_object_id;
      next_object_id += 1;
      typed_pointer *nj = new typed_pointer(pointer, type);
      object_to_id_table.insert((TableKey) nj, (TableData) id);
      id_to_object_table.insert((TableKey) id, (TableData) nj);
    }

  return id;
}

// ----------------------------------------------------------------------------
//
void *Object_Table::object(int id, Stringy *type)
{
  TableData table_object;

  if (id_to_object_table.find((TableKey) id, &table_object))
    {
      typed_pointer *j = (typed_pointer *) table_object;
      if (type)
	*type = j->type;

      return j->pointer;
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
List Object_Table::delete_ids(void *pointer)
{
  List idlist;
  TableData table_id;
  typed_pointer j(pointer, "any");
  while (object_to_id_table.find((TableKey) &j, &table_id))
    {
      TableData tj;
      if (!id_to_object_table.find((TableKey) table_id, &tj))
	fatal_error("Object_Table::delete_ids(): ID %d missing.\n",
		    (int) (long) table_id);
      object_to_id_table.remove((TableKey) tj);
      id_to_object_table.remove((TableKey) table_id);
      delete (typed_pointer *) tj;
      idlist.append(table_id);
    }
  return idlist;
}

// ----------------------------------------------------------------------------
//
void Object_Table::set_id(void *pointer, const Stringy &type, int id)
{
  TableData table_object;
  if (!id_to_object_table.find((TableKey) id, &table_object))
    {
      typed_pointer *nj = new typed_pointer(pointer, type);
      object_to_id_table.insert((TableKey) nj, (TableData) id);
      id_to_object_table.insert((TableKey) id, (TableData) nj);
    }
  else if (((typed_pointer *) table_object)->pointer != pointer ||
	   ((typed_pointer *) table_object)->type != type)
    fatal_error("Object_Table::set_id(): id %d already in use\n", id);
}

// ----------------------------------------------------------------------------
//
static bool equal_objects(TableKey k1, TableKey k2)
{
  typed_pointer *j1 = (typed_pointer *) k1;
  typed_pointer *j2 = (typed_pointer *) k2;
  return (j1->pointer == j2->pointer &&
	  (j1->type == j2->type || j1->type == "any" || j2->type == "any"));
}

// ----------------------------------------------------------------------------
//
unsigned long object_hash(TableKey key)
  { return (unsigned long) ((typed_pointer *) key)->pointer; }
