/*
 * Ornament.C:		Implementation of the Ornament base class
 *
 * The Ornament class is the base class for all ornaments. Each view maintains
 * a list of Ornaments that are to be displayed, and each ornament is a
 * specialization of this base class. In fact, this is a virtual base class so
 * cannot be instantiated.
 *
 */
#include <stdlib.h>		// Use atoi()
#include <math.h>		// Use sqrt(), fabs()

#include "axismap.h"
#include "color.h"
#include "condition.h"
#include "label.h"
#include "molecule.h"
#include "notifier.h"		// use Notifier
#include "ornament.h"
#include "peak.h"
#include "peakgp.h"
#include "spectrum.h"
#include "utility.h"		// Use message()

static bool has_owner(Ornament *op, const List &ornaments);

// ----------------------------------------------------------------------------
//
const char *Ornament_Type_Names[] =
  {"label", "line", "peak", "grid", "peakgroup", NULL};

// ----------------------------------------------------------------------------
//
Ornament::Ornament(Spectrum *sp)
  : mColor("white")
{
	mLocked		= false;
	mSelected	= false;
	mSpectrum	= sp;
	mSpectrum->AddOrnament(this);
}

// ----------------------------------------------------------------------------
//
Ornament::~Ornament()
{
  spectrum()->RemoveOrnament(this);	// Remove from spectrum ornament list.

  mSpectrum = NULL;
}

// ----------------------------------------------------------------------------
//
Ornament *Ornament::copy(Spectrum *, const Axis_Map &)
{
  fatal_error("Ornament::copy(): Can't copy Ornament.\n");

  return NULL;
}

// ----------------------------------------------------------------------------
//
Ornament_Type Ornament::type() const { return (Ornament_Type) -1; }
const char *Ornament::type_name() const { return "ornament"; }
bool valid_otype(int t) { return (t >= 0 && t < ORNAMENT_TYPE_COUNT); }

// ----------------------------------------------------------------------------
//
Spectrum *Ornament::spectrum() const
  { return mSpectrum; }
int Ornament::dimension() const
  { return spectrum()->dimension(); }

// ----------------------------------------------------------------------------
//
Stringy Ornament::GetNote() const
  { return mNote; }

// ----------------------------------------------------------------------------
//
void Ornament::SetNote(const Stringy &note)
{
  /*
   * If there is no current note, or if the current note does
   * not match the new note, replace the current note.
   */
  if (note != mNote)
    {
      mNote = note;
      Notifier &n = spectrum()->notifier();
      n.send_notice(nt_changed_ornament, this);
    }
}

// ----------------------------------------------------------------------------
//
bool Ornament::IsSelected() const
  { return mSelected; }

// ----------------------------------------------------------------------------
//
void Ornament::select(bool selected)
{
  if (selected != mSelected)
    {
      Notifier &n = spectrum()->notifier();
      n.send_notice(nt_will_change_ornament, this);
      mSelected = selected;
      if (selected)
	{
	  n.send_notice(nt_changed_ornament, this);
	  n.send_notice(nt_selected_ornament, this);
	}
      else
	{
	  n.send_notice(nt_changed_ornament, this);
	  n.send_notice(nt_unselected_ornament, this);
	}
    }
}

// ---------------------------------------------------------------------------
//
bool Ornament::IsLocked() const
  { return mLocked; }

// ---------------------------------------------------------------------------
//
void Ornament::lock(bool state)
{
	mLocked = state;
}

// ---------------------------------------------------------------------------
//
Color Ornament::GetColor() const
  { return mColor; }

/*
 * If the color <c> is different than the ornament's current color, set
 * it to the new color and mark that it has changed.
 */
void Ornament::SetColor(const Color &c)
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_change_ornament, this);
  mColor = c;
  n.send_notice(nt_changed_ornament, this);
}

/*
 * Routines that need knowledge about the windowing system
 */

// ----------------------------------------------------------------------------
// If ornament selected display selection box rectangle.
//
void Ornament::display(ODraw &dr) const
{
  if (IsSelected())
    {
      dr.set_drawing_color("white");
      SRegion region = selection_region(dr);
      if (region.intersect(dr.clip_region()))
	{
	  Rectangle r(region.min[dr.axis(X)], region.min[dr.axis(Y)],
		      region.max[dr.axis(X)], region.max[dr.axis(Y)]);
	  dr.draw_rectangle(r);
	}
    }
}

// ----------------------------------------------------------------------------
// Rectangle for selecting with mouse.
//
SRegion Ornament::selection_region(ODraw &dr) const
{
  return erasure_region(dr);
}

// ----------------------------------------------------------------------------
// Rectangle for erasing ornament.
//
void add_selection_padding(SRegion *r, ODraw &dr)
{
  int xaxis = dr.axis(X);
  int yaxis = dr.axis(Y);
  double ex = dr.select_size(xaxis);
  double ey = dr.select_size(yaxis);

  r->min[xaxis] -= ex;
  r->max[xaxis] += ex;
  r->min[yaxis] -= ey;
  r->max[yaxis] += ey;
}

// ----------------------------------------------------------------------------
//
void Ornament::print(ODraw &, FILE *, double, double, Rectangle) const
{
}

// ----------------------------------------------------------------------------
//
Spectrum *spectrum_of_ornaments(const List &ornaments)
{
  Spectrum *sp = NULL;

  for (int oi = 0 ; oi < ornaments.size() ; ++oi)
    {
      Ornament *op = (Ornament *) ornaments[oi];

      if (sp && op->spectrum() != sp)
	return NULL;			// More than one spectrum

      sp = op->spectrum();
    }

  return sp;
}

// ----------------------------------------------------------------------------
//
List ornament_copy(const List &olist, Spectrum *sp, const Axis_Map &axismap)
{
  List copies;
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (!has_owner(op, olist))
	copies.append(op->copy(sp, axismap));
    }
  return copies;
}

// ----------------------------------------------------------------------------
//
static bool has_owner(Ornament *op, const List &ornaments)
{
  Ornament *owner = NULL;

  if (op->type() == label)
    owner = ((Label *)op)->attached();
  else if (op->type() == peak)
    owner = ((Peak *)op)->peakgp();

  return owner && ornaments.contains(owner);
}

// ----------------------------------------------------------------------------
//
void select_ornaments(const List &olist)
{
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      op->select(true);
    }
}

// ----------------------------------------------------------------------------
//
void ODraw::set_drawing_color(const Color &) {}
void ODraw::draw_line(double, double, double, double) {}
void ODraw::draw_text(double, double, const Stringy &) {}
void ODraw::text_size(const Stringy &, double *, double *, double *) {}
void ODraw::draw_rectangle(Rectangle) {}
void ODraw::draw_triangle(double, double, double, double, double, double) {}
void ODraw::draw_ellipse(Rectangle) {}
int ODraw::axis(Axis) { return 0; }
double ODraw::aspect() { return 1.0; }
SRegion ODraw::clip_region() { return SRegion(); }
double ODraw::ornament_size(Ornament_Type, int) { return 0; }
double ODraw::select_size(int) { return 0; }
void ODraw::xy_point(const SPoint &, double *, double *) {}
bool ODraw::is_visible(const Ornament *, Rectangle) { return false; }
int ODraw::in_plane(const SPoint &) { return 0; }
bool ODraw::black_labels() { return false; }
bool ODraw::subtract_fit_peaks() { return false; }
