/*
 * ornament.h:		Interface to the Ornament base class
 *
 * The Ornament class is the base class for all ornaments. Each view maintains
 * a list of Ornaments that are to be displayed, and each ornament is a
 * specialization of this base class. In fact, this is a virtual base class so
 * cannot be instantiated.
 *
 */
#ifndef _Ornament_h
#define _Ornament_h

#include <stdio.h>		// Use FILE

#include "color.h"		// Use Color
#include "rectangle.h"		// Use Axis
#include "stringc.h"		// Use Stringy

class Axis_Map;
class List;
class ODraw;
class Rectangle;
class Spectrum;
class SPoint;
class SRegion;

enum Ornament_Type {label, line, peak, grid, peakgroup};
extern const char *Ornament_Type_Names[];
#define ORNAMENT_TYPE_COUNT 5

//
// The Ornament class:
//
class Ornament
{
public:

  Ornament(Spectrum *sp);
  virtual ~Ornament();

  virtual Ornament *copy(Spectrum *to, const Axis_Map &axismap);

  //
  // Routines for ornament display.
  //
  virtual void		display(ODraw &) const;
  virtual SRegion	erasure_region(ODraw &) const = 0;
  virtual SRegion	selection_region(ODraw &) const;
  virtual void		print(ODraw &, FILE *,
			      double xsc, double ysc, // dots per PPM
			      Rectangle r // PPM
			      ) const;

  virtual void	IncrementLocation(const SPoint &) = 0;

  virtual Ornament_Type		type() const;
  virtual const char	*type_name() const;

  Spectrum		*spectrum() const;
  int			dimension() const;

  bool		IsSelected() const;
  void		select(bool state);

  Color		GetColor() const;
  void		SetColor(const Color &);

  Stringy 	GetNote() const;
  void		SetNote(const Stringy &str);

  bool		IsLocked() const;
  void		lock(bool state);

private:
  Spectrum	*mSpectrum;	// the containing spectrum
  Color		mColor;		// this ornament's color
  Stringy		mNote;
  bool		mSelected;	// if selected
  bool		mLocked;	// if cannot move from this location
};
 
// ----------------------------------------------------------------------------
//
class ODraw
{
public:
  virtual void set_drawing_color(const Color &color);
  virtual void draw_line(double x1, double y1, double x2, double y2);
  virtual void draw_text(double cx, double cy, const Stringy &text);
  virtual void text_size(const Stringy &text, double *width,
			 double *ascent, double *descent);
  virtual void draw_rectangle(Rectangle r);
  virtual void draw_triangle(double x1, double y1,
			     double x2, double y2,
			     double x3, double y3);
  virtual void draw_ellipse(Rectangle r);

  virtual int axis(Axis a);
  virtual double aspect();
  virtual SRegion clip_region();
  virtual double ornament_size(Ornament_Type type, int axis);
  virtual double select_size(int axis);
  virtual void xy_point(const SPoint &p, double *x, double *y);
  virtual bool is_visible(const Ornament *op, Rectangle r);
  virtual int in_plane(const SPoint &p);
  virtual bool black_labels();
  virtual bool subtract_fit_peaks();
};

void add_selection_padding(SRegion *r, ODraw &);

bool valid_otype(int t);
Spectrum *spectrum_of_ornaments(const List &ornaments);
List ornament_copy(const List &olist, Spectrum *sp, const Axis_Map &axismap);
void select_ornaments(const List &olist);

#endif
