/*
 * peak.h - Interface for the Peak class.
 *
 * A Peak object is a single extremum in an NMR spectrum.
 * Peaks and PeakGps are both specializations of CrossPeaks.
 *
 */
#ifndef _Peak_h
#define _Peak_h

class Peak;

#include "crosspeak.h"
#include "integrate.h"			// Use Integration_Method
#include "spoint.h"			// Use SPoint

class Axis_Map;
class PeakGp;

// ----------------------------------------------------------------------------
//
class Peak : public CrossPeak {

public:

  Peak(Spectrum *sp, const SPoint &position);
  ~Peak();

  virtual Ornament	*copy(Spectrum *to, const Axis_Map &axismap);

  virtual Ornament_Type	type() const;
  virtual const char	*type_name() const;

  void			peakgp(PeakGp *);
  PeakGp *		peakgp() const;

  virtual Resonance *	resonance(int a) const;
  virtual void		SetResonance(int a, Resonance *);
  virtual Stringy	assignment_name(const Stringy &format) const;

  double		DataHeight() const;
  void			DataHeight(double h);
  virtual double	SignalToNoise() const;

  //
  // Integration
  //
  virtual bool		volume(double *vol = 0, Stringy *method = 0) const;
  bool			IntegrateByBox() const;
  bool			IntegrateByFit() const;
  Integration_Method	GetIntegrateMethod() const;

  void			no_volume();
  void			manual_volume(double);
  void			fit_volume(Integration_Method, double vol,
				   double fit_error);
  void			box_volume(Integration_Method, double vol);

  double		FitHeight() const;
  void			FitHeight(double h);
  double		fit_height(const SPoint &p);
  double		fit_residual() const;

  //
  // Linewidth
  //
  bool			HasLinewidth() const;
  SPoint		linewidth() const;
  double		linewidth(int a) const;
  Stringy		GetLinewidthMethod() const;
  double		linewidth_cutoff_range() const;

  void			linewidth(const Stringy &method, const SPoint &);
  void			estimate_linewidth();

  //
  // Ornament Section
  //
  virtual SRegion	erasure_region(ODraw &) const;
  virtual void		print(ODraw &dr, FILE *fp,
			      double xsc, double ysc,
			      Rectangle r) const;

  virtual void		IncrementLocation(const SPoint &dr);
  void			SetLocation(const SPoint &r); // unintegrates peak

  virtual void		set_position(const SPoint &);// moves peak group marker
  virtual void		set_alias(const SPoint &);   // moves peak group marker

  bool			MoveToMaximum();  // Move to data maximum

private:

  PeakGp		*mPeakGroup;		// parent crosspeak

  double		mDataHeight;		// cached data height
  bool			mDataHeightValid;

  Integration_Method	mIntegrateMethod;
  double		mVolume;
  double		mFitHeight;
  double		mFitResidual;		// RMS error fraction

  Stringy		mLinewidthMethod;
  SPoint		mLinewidth;		// the peak linewidth

  void			set_volume(double vol, Integration_Method t);
  void			SetIntegrateMethod(Integration_Method t);
};

#endif
