/*
 * PeakGp.C:		Implementation of the PeakGp class.
 *
 * A PeakGp object is a collection of Peaks in an NMR Spectrum.
 */

#include <ctype.h>
#include <math.h>		// use fabs()

#include "atom.h"		// use Atom
#include "axismap.h"		// Use Axis_Map
#include "color.h"
#include "group.h"		// Use Group
#include "label.h"
#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "num.h"
#include "ornament.h"
#include "peak.h"
#include "peakgp.h"
#include "print.h"		// use color_index()
#include "resonance.h"
#include "spectrum.h"
#include "utility.h"		// Use fatal_error()

// ----------------------------------------------------------------------------
//
PeakGp::PeakGp(Spectrum *sp, const SPoint &pos) : CrossPeak(sp, pos)
{
  spectrum()->add(*this);
  sp->notifier().send_notice(nt_created_ornament, (Ornament *) this);
}

// ----------------------------------------------------------------------------
//
PeakGp::PeakGp(Spectrum *sp, const List &peaks)
  : CrossPeak(sp, SPoint(sp->dimension()))
{
  spectrum()->add(*this);

  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    add(*(Peak *) peaks[pi]);
  match_peaklet_alias();

  sp->notifier().send_notice(nt_created_ornament, (Ornament *) this);
}

// ----------------------------------------------------------------------------
//
PeakGp::~PeakGp()
{
  Notifier &n = spectrum()->notifier();
  n.send_notice(nt_will_delete_ornament, (Ornament *) this);

	removeAllPeaks();
	spectrum()->remove(*this);
}

// ----------------------------------------------------------------------------
//
Ornament *PeakGp::copy(Spectrum *to, const Axis_Map &axismap)
{
  SPoint pos = axismap.map(position());
  PeakGp *pg = new PeakGp(to, pos);
  pg->set_alias(alias());
  pg->lock(IsLocked());
  pg->SetNote(GetNote());
  pg->SetColor(GetColor());

  for (int a = 0 ; a < dimension() ; ++a)
    if (resonance(a))
      {
	int to_axis = axismap.map(a);
	Resonance *r = resonance(a);
	pg->ChangeAssignment(to_axis,
			     r->atom()->name().cstring(),
			     r->group()->name().cstring());
      }

  const List &peaks = peaklets();
  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    pg->add(*(Peak *)((Peak *) peaks[pi])->copy(to, axismap));

  if (label())
    {
      Label *lbl = (Label *) label()->copy(to, axismap);
      lbl->attach(pg);
      lbl->assignment(label()->is_assignment());
    }

  return pg;
}

// ----------------------------------------------------------------------------
//
Ornament_Type PeakGp::type() const { return ::peakgroup; }
const char *PeakGp::type_name() const { return "peakgroup"; }

//
// Remove all Peaks from the PeakGp. This method must call the member remove()
// function to ensure the Peak is properly disconnected from the PeakGp.
//
void PeakGp::removeAllPeaks()
{
	List plist = mPeaks;
	for (int pk = 0 ; pk < plist.size() ; ++pk)
		remove(*(Peak *)plist[pk]);
}

void PeakGp::add(Peak &x)
{
	//
	// If the peak belongs to a group other than this one, remove it
	// from that group.
	//
	if (x.peakgp() && x.peakgp() != this)
		x.peakgp()->remove(x);

	if (x.peakgp() != this) {

		//
		// Remove the peaklets resonance information, which it will
		// now get from this PeakGroup.
		//
		for (int a = 0; a < x.dimension(); a++) {
			Resonance	*rp = x.resonance(a);

			if (rp) {
				//
				// If the PeakGroup doen't have an assignment
				// in this direction yet, add this assignment.
				//
				if (resonance(a) == NULL)
					SetResonance(a, rp);

				//
				// Remove the Peak's assignment.
				//
				x.SetResonance(a, NULL);
			}
		}


		//
		// Move Peak off of the spectrum list, onto this list.
		//
		spectrum()->remove(x);
		mPeaks.append(&x);

		//
		// Inform peak of its peak group
		//
		x.peakgp(this);

		recompute_position();
	}
}

void PeakGp::remove(Peak &x)
{
	//
	// Move Peak off of this list, onto the spectrum list.
	//
	int pk = mPeaks.find(&x);
	if (pk == mPeaks.size())
		fatal_error("PeakGp::remove(): Not a peaklet.\n");

	mPeaks.erase(pk);
	x.peakgp((PeakGp *) NULL);
	spectrum()->add(x);
	recompute_position();
}

// ----------------------------------------------------------------------------
//
const List &PeakGp::peaklets() const
  { return mPeaks; }

///////////////////////////////////////////////////////////
//
//			Frequency setting and getting
//
///////////////////////////////////////////////////////////

SPoint PeakGp::average_position() const
{
  if (volume())
    {
      // All peaklets integrated.  Compute the volume weighted average.

      SPoint freq(spectrum()->dimension());
      double volume_sum = 0;
      const List &peaks = peaklets();
      for (int pi = 0 ; pi < peaks.size() ; ++pi)
	{
	  Peak *pk = (Peak *) peaks[pi];
	  double vol;
	  if (pk->volume(&vol))
	    {
	      vol = fabs(vol);
	      volume_sum += vol;
	      freq += pk->frequency() * vol;
	    }
	}
      if (volume_sum > 0)
	{
	  freq = freq * (1.0 / volume_sum);
	  return freq - alias();
	}
    }

  // Compute average frequency giving peaklets equal weights

  const List &peaks = peaklets();
  if (peaks.size() > 0)
    {
      SPoint freq(spectrum()->dimension());
      for (int pi = 0 ; pi < peaks.size() ; ++pi)
	freq += ((Peak *) peaks[pi])->frequency();
      freq = freq * (1.0 / peaks.size());
      return freq - alias();
    }

  return SPoint(spectrum()->dimension());
}

void PeakGp::recompute_position()
{
  set_position(average_position());
}

// ----------------------------------------------------------------------------
//
void PeakGp::set_alias(const SPoint &a)
{
  CrossPeak::set_alias(a);
  recompute_position();
}

///////////////////////////////////////////////////////////
//
// 			VOLUME determination
//
///////////////////////////////////////////////////////////

//
// Return the crosspeak volume. If the volume has been overridden by
// the user, return the overridden value. Otherwise, return the
// sum of the volumes of the children.
//
bool PeakGp::volume(double *vol, Stringy *method) const
{
  const List &peaks = peaklets();
  if (peaks.size() == 0)
    return false;

  double sumvol = 0.0, v;
  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    if (((Peak *) peaks[pi])->volume(&v))
      sumvol += v;
    else
      return false;

  if (method)
    *method = "peak-group";
  if (vol)
    *vol = sumvol;

  return true;
}

/*
 * Return the *average* signal-to-noise ratio for a peak group.
 */
double PeakGp::SignalToNoise(void) const
{
	if (volume()) {
	  double	ave = 0.0;
	  const List &peaks = peaklets();
	  for (int pi = 0 ; pi < peaks.size() ; ++pi)
	    ave += ((Peak *) peaks[pi])->SignalToNoise();
	  return ave / peaks.size();
	}
	return 0.0;
}

/*
 * Add peaks to peakgroup setting alias of peakgroup if all peaks
 * have same alias.
 */
void PeakGp::match_peaklet_alias()
{
  const List &peaks = peaklets();
  if (peaks.size() > 0)
    {
      SPoint alias = ((Peak *)peaks[0])->alias();
      for (int k = 1 ; k < peaks.size() ; ++k)
	if (((Peak *)peaks[k])->alias() != alias)
	  return;

      set_alias(alias);
    }
}

//
// Move a PeakGp manually to a new location. Since the PeakGp
// normally is located at the mass center of the Peaks, this movement
// causes the frequency() of the PeakGp to be overridden.
//
void PeakGp::IncrementLocation(const SPoint &)
{
  // Peak group position determined by peaklets.
}

// ----------------------------------------------------------------------------
// Send a Postscript command to draw a peak.
//
void PeakGp::print(ODraw &dr, FILE *fp,
		   double xsc, double ysc, Rectangle r) const
{
	if (!dr.is_visible(this, r))
		return;

	Rectangle sq = marker_square(dr);
	double x = (r.max(X) - sq.max(X)) * xsc;
	double y = (r.max(Y) - sq.max(Y)) * ysc;
	double w = sq.size(X) * xsc;
	double h = sq.size(Y) * ysc;

	fprintf(fp, "%d %d %.0f %.0f %.0f %.0f PX\n",
		color_index(GetColor()),	// edge color
		color_index(GetColor()),	// fill color
		x, y, w, h);
}
