// ----------------------------------------------------------------------------
//
#include <stdio.h>		// Use sprintf()
#include <math.h>		// Use fabs()

#include "spoint.h"
#include "utility.h"		// Use fatal_error()
#include "num.h"		// Use nearest_int()

static bool has_element(const IPoint &p, int e);

// ----------------------------------------------------------------------------
//
SPoint::SPoint()
{
  dim = 0;
  for (int d = 0 ; d < DIM ; ++d)
    c[d] = 0;				// Debugging aid.
}

// ----------------------------------------------------------------------------
//
SPoint::SPoint(int dim)
{
  this->dim = dim;
  for (int d = 0 ; d < dim ; ++d)
    c[d] = 0;
  for (int d = dim ; d < DIM ; ++d)
    c[d] = 0;				// Debugging aid.
}

// ----------------------------------------------------------------------------
//
SPoint::SPoint(const IPoint &p)
{
  dim = p.dimension();
  for (int d = 0 ; d < dim ; ++d)
    c[d] = p[d];
  for (int d = dim ; d < DIM ; ++d)
    c[d] = 0;				// Debugging aid.
}

// ----------------------------------------------------------------------------
//
SPoint::SPoint(int dim, const double *coords)
{
  this->dim = dim;
  for (int d = 0 ; d < dim ; ++d)
    c[d] = coords[d];
  for (int d = dim ; d < DIM ; ++d)
    c[d] = 0;				// Debugging aid.
}

// ----------------------------------------------------------------------------
//
IPoint SPoint::rounded() const
{
  int e[DIM];
  for (int d = 0 ; d < dimension() ; ++d)
    e[d] = nearest_int((*this)[d]);
  return IPoint(dimension(), e);
}

// ----------------------------------------------------------------------------
//
int SPoint::dimension() const
{
  return dim;
}

// ----------------------------------------------------------------------------
//
bool SPoint::operator==(const SPoint &p) const
{
  if (p.dimension() != dimension())
    return false;

  for (int a = 0 ; a < dimension() ; ++a)
    if (p[a] != (*this)[a])
      return false;

  return true;
}

// ----------------------------------------------------------------------------
//
bool SPoint::operator!=(const SPoint &p) const
  { return !(*this == p); }

// ----------------------------------------------------------------------------
//
double SPoint::operator[](int axis) const
  { return c[axis]; }
double &SPoint::operator[](int axis)
{
//  if (axis < 0 || axis >= dimension() || dimension() > 4)
//    fatal_error("SPoint::operator[](): Index out of range.\n");

  return c[axis];
}

// ----------------------------------------------------------------------------
//
SPoint &SPoint::operator=(const SPoint &p)
{
  dim = p.dimension();
  for (int d = 0 ; d < dim ; ++d)
    c[d] = p[d];

  return *this;
}

// ----------------------------------------------------------------------------
//
SPoint &SPoint::operator+=(const SPoint &p)
{
  for (int d = 0 ; d < dimension() ; ++d)
    c[d] += p[d];

  return *this;
}

// ----------------------------------------------------------------------------
//
SPoint &SPoint::operator*=(const SPoint &p)
{
  for (int d = 0 ; d < dimension() ; ++d)
    c[d] *= p[d];

  return *this;
}

// ----------------------------------------------------------------------------
//
SPoint &SPoint::operator*=(double scale)
{
  for (int d = 0 ; d < dimension() ; ++d)
    c[d] *= scale;

  return *this;
}

// ----------------------------------------------------------------------------
//
SPoint SPoint::operator-(const SPoint &p) const
{
  SPoint diff(dimension());

  for (int d = 0 ; d < dimension() ; ++d)
    diff[d] = (*this)[d] - p[d];

  return diff;
}

// ----------------------------------------------------------------------------
//
SPoint SPoint::operator-() const
{
  SPoint neg(dimension());

  for (int d = 0 ; d < dimension() ; ++d)
    neg[d] = -(*this)[d];

  return neg;
}

// ----------------------------------------------------------------------------
//
SPoint SPoint::operator+(const SPoint &p) const
{
  SPoint sum(dimension());

  for (int d = 0 ; d < dimension() ; ++d)
    sum[d] = (*this)[d] + p[d];

  return sum;
}

// ----------------------------------------------------------------------------
//
SPoint SPoint::operator*(const SPoint &p) const
{
  SPoint prod(dimension());

  for (int d = 0 ; d < dimension() ; ++d)
    prod[d] = (*this)[d] * p[d];

  return prod;
}

// ----------------------------------------------------------------------------
//
SPoint SPoint::operator*(double scale) const
{
  SPoint prod(dimension());

  for (int d = 0 ; d < dimension() ; ++d)
    prod[d] = (*this)[d] * scale;

  return prod;
}

// ----------------------------------------------------------------------------
//
SPoint SPoint::normal() const
{
  SPoint n(dimension());

  if (dimension() >= 2)
    {
      double norm = sqrt((*this)[0] * (*this)[0] + (*this)[1] * (*this)[1]);
      if (norm == 0)
	n[0] = 1;
      else
	{
	  n[0] = -(*this)[1] / norm;
	  n[1] = (*this)[0] / norm;
	}
    }

  return n;
}

// ----------------------------------------------------------------------------
//
SPoint SPoint::permute(const IPoint &new_order) const
{
  SPoint permuted(dimension());
  for (int a = 0 ; a < dimension() ; ++a)
    permuted[a] = (*this)[new_order[a]];
  return permuted;
}

// ----------------------------------------------------------------------------
//
SPoint SPoint::unpermute(const IPoint &new_order) const
{
  SPoint unpermuted(dimension());
  for (int a = 0 ; a < dimension() ; ++a)
    unpermuted[new_order[a]] = (*this)[a];
  return unpermuted;
}

// ----------------------------------------------------------------------------
//
SPoint SPoint::remove_component(int axis) const
{
  SPoint p(dimension() - 1);
  for (int a = 0 ; a < p.dimension() ; ++a)
    p[a] = (a < axis ? (*this)[a] : (*this)[a+1]);
  return p;
}

// ----------------------------------------------------------------------------
//
double SPoint::norm() const
{
  double n2 = 0;
  for (int a = 0 ; a < dimension() ; ++a)
    n2 += (*this)[a] * (*this)[a];
  return sqrt(n2);
}

// ----------------------------------------------------------------------------
//
bool points_are_close(const SPoint &p1, const SPoint &p2, const SPoint &limit)
{
  for (int a = 0 ; a < limit.dimension() ; ++a)
    if (fabs(p1[a] - p2[a]) > limit[a])
      return false;

  return true;
}

// ----------------------------------------------------------------------------
//
SRegion::SRegion() {}

// ----------------------------------------------------------------------------
//
SRegion::SRegion(int dim) : min(dim), max(dim)
{
  for (int a = 0 ; a < dim ; ++a)
    max[a] = -1;			// empty region
}

// ----------------------------------------------------------------------------
//
SRegion::SRegion(const SPoint &p1, const SPoint &p2) : min(p1), max(p2) {}

// ----------------------------------------------------------------------------
//
SRegion::SRegion(int xaxis, double xmin, double xmax,
		 int yaxis, double ymin, double ymax, const SPoint &p)
{
  min = p;
  max = p;
  min[xaxis] = xmin;
  max[xaxis] = xmax;
  min[yaxis] = ymin;
  max[yaxis] = ymax;
}

// ----------------------------------------------------------------------------
//
SRegion::SRegion(const IRegion &p) : min(p.min), max(p.max) {}

// ----------------------------------------------------------------------------
//
IRegion SRegion::rounded() const
  { return IRegion(min.rounded(), max.rounded()); }

// ----------------------------------------------------------------------------
//
SRegion &SRegion::operator=(const SRegion &r)
{
  min = r.min;
  max = r.max;

  return *this;
}

// ----------------------------------------------------------------------------
//
int SRegion::dimension() const
{
  return min.dimension();
}

// ----------------------------------------------------------------------------
//
double SRegion::size(int axis) const
{
  return max[axis] - min[axis];
}

// ----------------------------------------------------------------------------
//
SPoint SRegion::size() const
{
  return max - min;
}

// ----------------------------------------------------------------------------
//
double SRegion::center(int axis) const
{
  return (max[axis] + min[axis]) / 2;
}

// ----------------------------------------------------------------------------
//
SPoint SRegion::center() const
{
  SPoint c(dimension());

  for (int a = 0 ; a < dimension() ; ++a)
    c[a] = center(a);

  return c;
}

// ----------------------------------------------------------------------------
//
bool SRegion::no_interior() const
{
  for (int d = 0 ; d < dimension() ; ++d)
    if (min[d] >= max[d])
      return true;

  return dimension() == 0;
}

// ----------------------------------------------------------------------------
//
bool SRegion::contains(const SPoint &p) const
{
  for (int d = 0 ; d < dimension() ; ++d)
    if (min[d] > p[d] || max[d] < p[d])
      return false;

  return true;
}

// ----------------------------------------------------------------------------
//
bool SRegion::intersects(const SRegion &region) const
{
  if (region.dimension() != dimension())
    fatal_error("SRegion::intersects(): Dimension mismatch.\n");

  for (int d = 0 ; d < dimension() ; ++d)
    if (min[d] >= region.max[d] || max[d] <= region.min[d])
      return false;

  return true;
}

// ----------------------------------------------------------------------------
//
bool SRegion::intersect(const SRegion &region)
{
  if (region.dimension() != dimension())
    fatal_error("SRegion::intersect(): Dimension mismatch.\n");

  bool empty = false;
  for (int d = 0 ; d < dimension() ; ++d)
    {
      if (min[d] < region.min[d]) min[d] = region.min[d];
      if (max[d] > region.max[d]) max[d] = region.max[d];
      if (min[d] > max[d])
	empty = true;
    }

  return dimension() > 0 && !empty;
}

// ----------------------------------------------------------------------------
//
bool SRegion::operator==(const SRegion &r) const
  { return (r.min == min && r.max == max); }

// ----------------------------------------------------------------------------
//
bool SRegion::operator!=(const SRegion &r) const
  { return !(*this == r); }

// ----------------------------------------------------------------------------
//
SRegion &SRegion::operator+=(const SPoint &p)
{
  min += p;
  max += p;

  return *this;
}

// ----------------------------------------------------------------------------
//
bool SRegion::plane_axes(int *a1, int *a2) const
{
  int d1, d2, d3;

  for (d1 = 0 ; d1 < dimension() && min[d1] == max[d1] ; ++d1) ;
  for (d2 = d1+1 ; d2 < dimension() && min[d2] == max[d2] ; ++d2) ;
  for (d3 = d2+1 ; d3 < dimension() && min[d3] == max[d3] ; ++d3) ;

  bool two_dimensional = (d2 < dimension() && d3 == dimension());
  if (two_dimensional)
    {
      *a1 = d1;
      *a2 = d2;
    }

  return two_dimensional;
}

// ----------------------------------------------------------------------------
//
void SRegion::clip(const SRegion &r)
{
  for (int a = 0 ; a < dimension() ; ++a)
    {
      if (r.min[a] > min[a])
	min[a] = r.min[a];
      if (r.max[a] < max[a])
	max[a] = r.max[a];
    }
}

// ----------------------------------------------------------------------------
//
void SRegion::translate(double delta, int axis)
{
  min[axis] += delta;
  max[axis] += delta;
}

// ----------------------------------------------------------------------------
//
void SRegion::scale(double factor)
{
  SPoint c = center();
  min *= factor;
  max *= factor;
  recenter(c);
}

// ----------------------------------------------------------------------------
//
void SRegion::encompass(const SPoint &p)
{
  for (int a = 0 ; a < dimension() ; ++a)
    {
      if (p[a] < min[a])
	min[a] = p[a];
      if (p[a] > max[a])
	max[a] = p[a];
    }
}

// ----------------------------------------------------------------------------
//
void SRegion::encompass(const SRegion &r)
{
  for (int a = 0 ; a < dimension() ; ++a)
    {
      if (r.min[a] < min[a])
	min[a] = r.min[a];
      if (r.max[a] > max[a])
	max[a] = r.max[a];
    }
}

// ----------------------------------------------------------------------------
//
void SRegion::recenter(int axis, double position)
{
  translate(position - center(axis), axis);
}

// ----------------------------------------------------------------------------
//
void SRegion::recenter(SPoint center)
{
  for (int a = 0 ; a < dimension() ; ++a)
    translate(center[a] - this->center(a), a);
}

// ----------------------------------------------------------------------------
//
IPoint::IPoint()
{
  dim = 0;
  for (int d = 0 ; d < DIM ; ++d)
    c[d] = 0;
}

// ----------------------------------------------------------------------------
//
IPoint::IPoint(int dim)
{
  this->dim = dim;
  for (int d = 0 ; d < DIM ; ++d)
    c[d] = 0;
}

// ----------------------------------------------------------------------------
//
IPoint::IPoint(int dim, const int *coords)
{
  this->dim = dim;
  for (int d = 0 ; d < dim ; ++d)
    c[d] = coords[d];
  for (int d = dim ; d < DIM ; ++d)
    c[d] = 0;
}

// ----------------------------------------------------------------------------
//
int IPoint::dimension() const
{
  return dim;
}

// ----------------------------------------------------------------------------
//
bool IPoint::operator==(const IPoint &p) const
{
  if (p.dimension() != dimension())
    return false;

  for (int a = 0 ; a < dimension() ; ++a)
    if (p[a] != (*this)[a])
      return false;

  return true;
}

// ----------------------------------------------------------------------------
//
bool IPoint::operator!=(const IPoint &p) const
  { return !(*this == p); }

// ----------------------------------------------------------------------------
//
IPoint &IPoint::operator=(const IPoint &p)
{
  dim = p.dimension();
  for (int d = 0 ; d < dim ; ++d)
    c[d] = p[d];

  return *this;
}

// ----------------------------------------------------------------------------
//
IPoint IPoint::operator+(const IPoint &p) const
{
  IPoint sum(dimension());
  for (int a = 0 ; a < dimension() ; ++a)
    sum[a] = (*this)[a] + p[a];
  return sum;
}

// ----------------------------------------------------------------------------
//
size_t IPoint::product() const
{
  int d = dimension();
  if (d == 0)
    return 0;

  size_t prod = 1;
  for (int a = 0 ; a < d ; ++a)
    prod *= (*this)[a];

  return prod;
}

// ----------------------------------------------------------------------------
//
SPoint IPoint::point() const
{
  double coords[DIM];
  for (int a = 0 ; a < dimension() ; ++a)
    coords[a] = (*this)[a];

  return SPoint(dimension(), coords);
}

// ----------------------------------------------------------------------------
//
IPoint IPoint::roll() const
{
  IPoint rolled(dimension());
  for (int a = 0 ; a+1 < dimension() ; ++a)
    rolled[a] = (*this)[a+1];
  rolled[dimension()-1] = (*this)[0];
  return rolled;
}

// ----------------------------------------------------------------------------
//
IPoint IPoint::permute(const IPoint &new_order) const
{
  IPoint permuted(dimension());
  for (int a = 0 ; a < dimension() ; ++a)
    permuted[a] = (*this)[new_order[a]];
  return permuted;
}

// ----------------------------------------------------------------------------
//
IPoint IPoint::unpermute(const IPoint &new_order) const
{
  IPoint unpermuted(dimension());
  for (int a = 0 ; a < dimension() ; ++a)
    unpermuted[new_order[a]] = (*this)[a];
  return unpermuted;
}

// ----------------------------------------------------------------------------
//
bool IPoint::is_permutation() const
{
  for (int k = 0 ; k < dimension() ; ++k)
    if (! has_element(*this, k))
      return false;
  return true;
}

// ----------------------------------------------------------------------------
//
IPoint IPoint::insert_component(int axis, int value) const
{
  IPoint p(dimension() + 1);
  for (int a = 0 ; a < p.dimension() ; ++a)
    p[a] = (a < axis ? (*this)[a] : (a > axis ? (*this)[a-1] : value));
  return p;
}

// ----------------------------------------------------------------------------
//
IPoint IPoint::remove_component(int axis) const
{
  IPoint p(dimension() - 1);
  for (int a = 0 ; a < p.dimension() ; ++a)
    p[a] = (a < axis ? (*this)[a] : (*this)[a+1]);
  return p;
}

// ----------------------------------------------------------------------------
//
static bool has_element(const IPoint &p, int e)
{
  for (int k = 0 ; k < p.dimension() ; ++k)
    if (p[k] == e)
      return true;
  return false;
}

// ----------------------------------------------------------------------------
//
IRegion::IRegion()
{
}

// ----------------------------------------------------------------------------
//
IRegion::IRegion(int dim) : min(dim), max(dim)
{
  for (int a = 0 ; a < dim ; ++a)
    max[a] = -1;			// empty region
}

// ----------------------------------------------------------------------------
//
IRegion::IRegion(const IPoint &min, const IPoint &max)
{
  this->min = min;
  this->max = max;
}

// ----------------------------------------------------------------------------
//
IRegion::IRegion(int xaxis, int xmin, int xmax,
		 int yaxis, int ymin, int ymax, const IPoint &p)
{
  min = p;
  max = p;
  min[xaxis] = xmin;
  max[xaxis] = xmax;
  min[yaxis] = ymin;
  max[yaxis] = ymax;
}

// ----------------------------------------------------------------------------
//
IRegion &IRegion::operator=(const IRegion &r)
{
  min = r.min;
  max = r.max;

  return *this;
}

// ----------------------------------------------------------------------------
//
IRegion IRegion::operator+(const IPoint &shift) const
{
  return IRegion(min + shift, max + shift);
}

// ----------------------------------------------------------------------------
//
bool IRegion::operator==(const IRegion &r) const
  { return r.min == min && r.max == max; }
bool IRegion::operator!=(const IRegion &r) const
  { return !(*this == r); }

// ----------------------------------------------------------------------------
//
int IRegion::dimension() const
{
  return min.dimension();
}

// ----------------------------------------------------------------------------
//
int IRegion::size(int axis) const
{
  return max[axis] - min[axis] + 1;
}

// ----------------------------------------------------------------------------
//
IPoint IRegion::size() const
{
  IPoint s(dimension());
  for (int a = 0 ; a < dimension() ; ++a)
    s[a] = size(a);
  return s;
}

// ----------------------------------------------------------------------------
//
int IRegion::longest_axis() const
{
  int long_axis = 0;
  for (int a = 1 ; a < dimension() ; ++a)
    if (size(a) > size(long_axis))
      long_axis = a;
  return long_axis;
}

// ----------------------------------------------------------------------------
//
size_t IRegion::volume() const
{
  int d = dimension();
  if (d == 0)
    return 0;

  size_t vol = 1;
  for (int a = 0 ; a < d ; ++a)
    vol *= size(a);

  return vol;
}

// ----------------------------------------------------------------------------
//
bool IRegion::contains(const IPoint &p) const
{
  for (int a = 0 ; a < dimension() ; ++a)
    if (p[a] < min[a] || p[a] > max[a])
      return false;

  return true;
}

// ----------------------------------------------------------------------------
//
bool IRegion::contains(const IRegion &r) const
{
  if (r.empty())
    return true;

  for (int a = 0 ; a < dimension() ; ++a)
    if (r.min[a] < min[a] || r.max[a] > max[a])
      return false;

  return true;
}

// ----------------------------------------------------------------------------
//
bool IRegion::empty() const
{
  for (int a = 0 ; a < dimension() ; ++a)
    if (min[a] > max[a])
      return true;

  return (dimension() == 0);
}

// ----------------------------------------------------------------------------
//
void IRegion::clip(const IRegion &r)
{
  for (int a = 0 ; a < dimension() ; ++a)
    {
      if (r.min[a] > min[a])
	min[a] = r.min[a];
      if (r.max[a] < max[a])
	max[a] = r.max[a];
    }
}

// ----------------------------------------------------------------------------
//
void IRegion::encompass(const IRegion &r)
{
  for (int a = 0 ; a < dimension() ; ++a)
    {
      if (r.min[a] < min[a])
	min[a] = r.min[a];
      if (r.max[a] > max[a])
	max[a] = r.max[a];
    }
}

// ----------------------------------------------------------------------------
//
void IRegion::encompass(const IPoint &p)
{
  for (int a = 0 ; a < dimension() ; ++a)
    {
      if (p[a] < min[a])
	min[a] = p[a];
      if (p[a] > max[a])
	max[a] = p[a];
    }
}

// ----------------------------------------------------------------------------
//
IPoint IRegion::random_point() const
{
  IPoint p(dimension());
  for (int a = 0 ; a < dimension() ; ++a)
    p[a] = random_int(min[a], max[a]);
  return p;
}

// ----------------------------------------------------------------------------
//
IPoint IRegion::first_point() const
{
  return min;
}

// ----------------------------------------------------------------------------
//
bool IRegion::next_point(IPoint *p) const
{
  for (int a = dimension()-1 ; a >= 0 ; --a)
    if ((*p)[a] < max[a])
      {
	(*p)[a] += 1;
	return true;
      }
    else
      (*p)[a] = min[a];

  return false;
}

// ----------------------------------------------------------------------------
//
bool IRegion::plane_axes(int *a1, int *a2) const
{
  int d1, d2, d3;

  for (d1 = 0 ; d1 < dimension() && min[d1] == max[d1] ; ++d1) ;
  for (d2 = d1+1 ; d2 < dimension() && min[d2] == max[d2] ; ++d2) ;
  for (d3 = d2+1 ; d3 < dimension() && min[d3] == max[d3] ; ++d3) ;

  bool two_dimensional = (d2 < dimension() && d3 == dimension());
  if (two_dimensional)
    {
      *a1 = d1;
      *a2 = d2;
    }

  return two_dimensional;
}

// ----------------------------------------------------------------------------
//
void IRegion::translate(int delta, int axis)
{
  min[axis] += delta;
  max[axis] += delta;
}

// ----------------------------------------------------------------------------
//
double distance(const SPoint &p1, const SPoint &p2)
{
  if (p1.dimension() != p2.dimension())
    fatal_error("distance(): Points have different dimensions.\n");

  double d2 = 0;
  for (int d = 0 ; d < p1.dimension() ; ++d)
    {
      double delta = p2[d] - p1[d];
      d2 += delta * delta;
    }

  return sqrt(d2);
}

// ----------------------------------------------------------------------------
//
const char *point_format(const SPoint &point, const char *fmt, bool brackets)
{
	static char	buf[DIM * 24];
	char	*cp = buf;
	char	sep = ' ';

	if (brackets) {
		*cp++ = '[';
		*cp++ = ' ';
	}
	for (int i = 0; i < point.dimension(); i++) {
		if (i > 0)
			*cp++ = sep;
		(void) sprintf(cp, fmt, point[i]);

		//
		// Skip to the end of the the buffer.
		//
		while (*cp)
			cp++;
	}
	if (brackets) {
		*cp++ = ' ';
		*cp++ = ']';
	}
	*cp = '\0';

	return buf;
}

// ----------------------------------------------------------------------------
//
SRegion flat_region(const SRegion &region, int axis1, int axis2)
{
  SRegion r = region;
  for (int a = 0 ; a < r.dimension() ; ++a)
    if (a != axis1 && a != axis2)
      r.min[a] = r.max[a] = r.center(a);
  return r;
}
