// ----------------------------------------------------------------------------
//  Classes defining points and regions in n-dim'l space.
//

#ifndef SPOINT_HEADER_INCLUDED
#define SPOINT_HEADER_INCLUDED

#include <stdlib.h>		// use size_t

#include "constant.h"		// Use DIM

class IPoint;
class IRegion;

// ----------------------------------------------------------------------------
//
class SPoint
{
 public:
  SPoint();
  SPoint(int dim);
  SPoint(int dim, const double *coords);
  SPoint(const IPoint &);
  IPoint rounded() const;
  int dimension() const;
  bool operator==(const SPoint &) const;
  bool operator!=(const SPoint &) const;
  double operator[](int axis) const;
  double &operator[](int axis);
  SPoint &operator=(const SPoint &);
  SPoint &operator+=(const SPoint &);
  SPoint &operator*=(const SPoint &);
  SPoint &operator*=(double scale);
  SPoint operator-(const SPoint &) const;
  SPoint operator-() const;
  SPoint operator+(const SPoint &) const;
  SPoint operator*(const SPoint &) const;
  SPoint operator*(double scale) const;
  SPoint normal() const;
  SPoint permute(const IPoint &new_order) const;
  SPoint unpermute(const IPoint &new_order) const;
  SPoint remove_component(int axis) const;
  double norm() const;

 private:
  int dim;
  double c[DIM];
};

bool points_are_close(const SPoint &p1, const SPoint &p2, const SPoint &limit);

// ----------------------------------------------------------------------------
//
class SRegion
{
 public:
  SRegion();
  SRegion(int dim);
  SRegion(const SPoint &min, const SPoint &max);
  SRegion(int xaxis, double xmin, double xmax,
	  int yaxis, double ymin, double ymax, const SPoint &p);
  SRegion(const IRegion &);
  SRegion &operator=(const SRegion &);
  IRegion rounded() const;
  int dimension() const;
  double size(int axis) const;
  SPoint size() const;
  double center(int axis) const;
  SPoint center() const;
  bool no_interior() const;
  bool contains(const SPoint &) const;
  bool intersects(const SRegion &) const;
  bool operator==(const SRegion &) const;
  bool operator!=(const SRegion &) const;
  SRegion &operator+=(const SPoint &);
  bool plane_axes(int *a1, int *a2) const;
  void clip(const SRegion &);
  void translate(double delta, int axis);
  void scale(double factor);			// About center
  void encompass(const SRegion &);
  void encompass(const SPoint &);
  bool intersect(const SRegion &);
  void recenter(int axis, double position);
  void recenter(SPoint center);
  SPoint min, max;
};


// ----------------------------------------------------------------------------
//
class IPoint
{
 public:
  IPoint();
  IPoint(int dim);
  IPoint(int dim, const int *coords);
  int dimension() const;
  bool operator==(const IPoint &) const;
  bool operator!=(const IPoint &) const;
  int operator[](int axis) const { return c[axis]; }
  int &operator[](int axis) { return c[axis]; }
  IPoint &operator=(const IPoint &);
  IPoint operator+(const IPoint &) const;
  size_t product() const;
  SPoint point() const;
  IPoint roll() const;
  IPoint permute(const IPoint &new_order) const;
  IPoint unpermute(const IPoint &new_order) const;
  bool is_permutation() const;
  IPoint insert_component(int axis, int value) const;
  IPoint remove_component(int axis) const;

 private:
  int dim;
  int c[DIM];
};

// ----------------------------------------------------------------------------
//
class IRegion
{
 public:
  IRegion();
  IRegion(int dim);
  IRegion(const IPoint &min, const IPoint &max);
  IRegion(int xaxis, int xmin, int xmax,
	  int yaxis, int ymin, int ymax, const IPoint &p);
  IRegion &operator=(const IRegion &);
  IRegion operator+(const IPoint &) const;
  bool operator==(const IRegion &) const;
  bool operator!=(const IRegion &) const;
  int dimension() const;
  int size(int axis) const;
  IPoint size() const;
  int longest_axis() const;
  size_t volume() const;
  bool empty() const;
  bool contains(const IPoint &) const;
  bool contains(const IRegion &) const;
  void clip(const IRegion &);
  void encompass(const IRegion &);
  void encompass(const IPoint &);
  IPoint random_point() const;
  IPoint first_point() const;
  bool next_point(IPoint *) const;
  bool plane_axes(int *a1, int *a2) const;
  void translate(int delta, int axis);
  IPoint min, max;
};

double distance(const SPoint &p1, const SPoint &p2);
const char *point_format(const SPoint &point, const char *fmt, bool brackets);
SRegion flat_region(const SRegion &r, int axis1, int axis2);

#endif
