// ----------------------------------------------------------------------------
//  Dialog components.
//
#include <string.h>		// Use strcmp()

#include "atom.h"		// Use Atom
#include "condition.h"		// Use Condition
#include "group.h"		// Use Group
#include "memalloc.h"		// use new()
#include "molecule.h"		// Use Molecule
#include "resonance.h"		// Use Resonance
#include "project.h"		// use Project
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "stringc.h"		// Use Stringy
#include "uicomponents.h"
#include "notifier.h"		// use Notifier
#include "uiview.h"		// Use View
#include "winsystem.h"		// Use ws.edit_field(), row_attachments(), ...

static int group_compare(const void *group1, const void *group2);
static Condition *named_condition(const List &clist, const Stringy &mcname);
static void add_edit_field_menu_entries(WinSys &, Widget, const List &strings);

// ----------------------------------------------------------------------------
//
Resonance_Field::Resonance_Field(Session &s, Widget parent,
				 const Stringy &name)
  : session(s), ws(s.window_system())
{
  this->f = ws.create_form(parent, name);

  this->cond = ws.edit_field(f, "resMolCond", true);
  this->condtext = ws.edit_field_text(cond);
  update_condition_menu();
  Notifier &n = session.notifier();
  n.notify_me(nt_added_condition_to_molecule, condition_list_changed_cb, this);
  n.notify_me(nt_removed_condition_from_molecule,
	    condition_list_changed_cb, this);
  ws.add_text_field_changed_callback(condtext, condition_changed_cb, this);
  n.notify_me(nt_added_resonance_to_condition, resonance_list_changed_cb, this);

  this->res = ws.edit_field(f, "resEdit", true);
  this->restext = ws.edit_field_text(res);

  ws.column_attachments(res, cond, res, END_OF_WIDGETS);

  update_resonance_menu(this);
}

// ----------------------------------------------------------------------------
//
Resonance_Field::~Resonance_Field()
{
  ws.remove_work_procedure(update_resonance_menu, this);
  Notifier &n = session.notifier();
  n.dont_notify_me(nt_added_condition_to_molecule,
		 condition_list_changed_cb, this);
  n.dont_notify_me(nt_removed_condition_from_molecule,
		 condition_list_changed_cb, this);
  ws.remove_text_field_changed_callback(condtext, condition_changed_cb, this);
  n.dont_notify_me(nt_added_resonance_to_condition,
		 resonance_list_changed_cb, this);
  f = NULL;
  cond = NULL;
  condtext = NULL;
  res = NULL;
  restext = NULL;
}

// ----------------------------------------------------------------------------
//
Widget Resonance_Field::frame()
  { return f; }
Widget Resonance_Field::resonance_text_field()
  { return restext; }
Widget Resonance_Field::condition_text_field()
  { return condtext; }

// ----------------------------------------------------------------------------
//
void Resonance_Field::update_condition_menu()
{
  Widget mc_menu = ws.edit_field_menu(cond);
  Widget mc_pane = ws.menu_pane(mc_menu);
  ws.delete_menu_buttons(mc_pane);
  List clist = session.project().condition_list();
  for (int ci = 0 ; ci < clist.size() ; ++ci)
    {
      Condition *c = (Condition *) clist[ci];
      ws.add_edit_field_menu_entry(mc_pane, c->fullname(), cond);
    }
}

// ----------------------------------------------------------------------------
//
void Resonance_Field::condition_list_changed_cb(void *rfield, void *)
{
  Resonance_Field *rf = (Resonance_Field *) rfield;
  rf->update_condition_menu();
}

// ----------------------------------------------------------------------------
//
Resonance *Resonance_Field::resonance_chosen()
{
  Stringy resname = ws.edit_value(res);
  Stringy mcname = ws.edit_value(cond);
  Condition *c = named_condition(session.project().condition_list(), mcname);
  Resonance *r = (c ? c->find_resonance(resname) : NULL);

  return r;
}

// ----------------------------------------------------------------------------
//
static Condition *named_condition(const List &clist, const Stringy &molcond)
{
  for (int ci = 0 ; ci < clist.size() ; ++ci)
    {
      Condition *c = (Condition *) clist[ci];
      if (c->fullname() == molcond)
	return c;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
void Resonance_Field::set_resonance_field(const Resonance *r)
{
  if (r)
    {
      ws.set_edit_value(cond, r->condition()->fullname());
      ws.set_edit_value(res, r->name());
    }
  else
    ws.set_edit_value(res, "");
}

// ----------------------------------------------------------------------------
//
void Resonance_Field::condition_changed_cb(Widget, CB_Data rfield, CB_Data)
{
  Resonance_Field *rf = (Resonance_Field *) rfield;
  rf->ws.add_work_procedure(update_resonance_menu, rfield);
}

// ----------------------------------------------------------------------------
//
void Resonance_Field::resonance_list_changed_cb(void *rfield, void *)
{
  Resonance_Field *rf = (Resonance_Field *) rfield;
  rf->ws.add_work_procedure(update_resonance_menu, rfield);
}

// ----------------------------------------------------------------------------
//
void Resonance_Field::update_resonance_menu(void *rfield)
{
  Resonance_Field *rf = (Resonance_Field *) rfield;
  Stringy mcname = rf->ws.edit_value(rf->cond);
  Condition *c = named_condition(rf->session.project().condition_list(),
				 mcname);

  Widget resedit = rf->res;
  Widget resmenu = rf->ws.edit_field_menu(resedit);
  Widget group_pane = rf->ws.menu_pane(resmenu);
  rf->ws.delete_menu_buttons(group_pane);

  if (c)
    {
      List glist = c->molecule()->GroupList();
      glist.sort(group_compare);
      for (int gi = 0 ; gi < glist.size() ; ++gi)
	{
	  Group *g = (Group *) glist[gi];
	  Widget atom_pane = rf->ws.cascade_pane(group_pane, g->name());
	  List alist = c->molecule()->AtomList(g);
	  for (int ai = 0 ; ai < alist.size() ; ++ai)
	    {
	      Atom *a = (Atom *) alist[ai];
	      rf->ws.add_edit_field_menu_entry(atom_pane, a->LongName(),
					       resedit);
	    }
	}
    }
  rf->ws.square_up_menu(resmenu);
}

// ----------------------------------------------------------------------------
//
static int group_compare(const void *group1, const void *group2)
{
  Group *g1 = (Group *) group1;
  Group *g2 = (Group *) group2;

  if (g1->number() < g2->number())
    return -1;
  if (g1->number() > g2->number())
    return +1;

  return strcmp(g1->name().cstring(), g2->name().cstring());
}

// ----------------------------------------------------------------------------
//
Molecule_Field::Molecule_Field(Session &s, Widget parent, const Stringy &name)
  : session(s), ws(s.window_system())
{
  mf = ws.edit_field(parent, name, true);
  ws.set_edit_title(mf, "Molecule: ");
  update_molecule_menu();
  Notifier &n = session.notifier();
  n.notify_me(nt_added_molecule_to_project, added_molecule_cb, this);
}

// ----------------------------------------------------------------------------
//
Molecule_Field::~Molecule_Field()
{
  Notifier &n = session.notifier();
  n.dont_notify_me(nt_added_molecule_to_project, added_molecule_cb, this);
}

// ----------------------------------------------------------------------------
//
Widget Molecule_Field::edit_field()
  { return mf; }

// ----------------------------------------------------------------------------
//
void Molecule_Field::added_molecule_cb(void *mfield, void *)
  { ((Molecule_Field *) mfield)->update_molecule_menu(); }

// ----------------------------------------------------------------------------
//
void Molecule_Field::update_molecule_menu()
{
  List mnames;
  List mlist = session.project().molecule_list();
  for (int mi = 0 ; mi < mlist.size() ; ++mi)
    mnames.append(new Stringy(((Molecule *) mlist[mi])->name()));
  mnames.sort_removing_duplicates(string_compare);
  add_edit_field_menu_entries(ws, mf, mnames);
  free_string_list_entries(mnames);
}

// ----------------------------------------------------------------------------
//
static void add_edit_field_menu_entries(WinSys &ws, Widget ef,
					const List &strings)
{
  Widget menu = ws.edit_field_menu(ef);
  Widget pane = ws.menu_pane(menu);
  ws.delete_menu_buttons(pane);
  for (int ni = 0 ; ni < strings.size() ; ++ni)
    ws.add_edit_field_menu_entry(pane, *(Stringy *) strings[ni], ef);
}

// ----------------------------------------------------------------------------
//
Condition_Menu::Condition_Menu(Session &s, Widget parent, const Stringy &name)
  : session(s), ws(s.window_system())
{
  menu = ws.option_menu(parent, name, NULL);

  update_condition_menu();
  Notifier &n = session.notifier();
  n.notify_me(nt_added_condition_to_molecule, added_condition_cb, this);

  Spectrum *sp = selected_spectrum(s);
  if (sp)
    ws.set_option(menu, sp->condition()->fullname());
}

// ----------------------------------------------------------------------------
//
Condition_Menu::~Condition_Menu()
{
  Notifier &n = session.notifier();
  n.dont_notify_me(nt_added_condition_to_molecule, added_condition_cb, this);
}

// ----------------------------------------------------------------------------
//
Widget Condition_Menu::option_menu()
  { return menu; }

// ----------------------------------------------------------------------------
//
void Condition_Menu::added_condition_cb(void *cmenu, void *)
  { ((Condition_Menu *) cmenu)->update_condition_menu(); }

// ----------------------------------------------------------------------------
//
void Condition_Menu::update_condition_menu()
{
  ws.delete_all_options(menu);
  List clist = session.project().condition_list();
  for (int ci = 0 ; ci < clist.size() ; ++ci)
    ws.add_option(menu, ((Condition *) clist[ci])->fullname());
}

// ----------------------------------------------------------------------------
//
Condition *Condition_Menu::condition_chosen()
{
  return named_condition(session.project().condition_list(),
			 ws.option_selected(menu));
}

// ----------------------------------------------------------------------------
//
void Condition_Menu::set_condition_choice(const Condition &c)
{
  if (c.fullname() != ws.option_selected(menu))
    ws.set_option(menu, c.fullname());
}

// ----------------------------------------------------------------------------
//
Condition_Field::Condition_Field(Session &s, Widget parent, const Stringy &name)
  : session(s), ws(s.window_system())
{
  cf = ws.edit_field(parent, name, true);
  ws.set_edit_title(cf, "Condition: ");

  update_condition_menu();
  Notifier &n = session.notifier();
  n.notify_me(nt_added_condition_to_molecule, added_condition_cb, this);
}

// ----------------------------------------------------------------------------
//
Condition_Field::~Condition_Field()
{
  Notifier &n = session.notifier();
  n.dont_notify_me(nt_added_condition_to_molecule, added_condition_cb, this);
}

// ----------------------------------------------------------------------------
//
Widget Condition_Field::edit_field()
  { return cf; }

// ----------------------------------------------------------------------------
//
void Condition_Field::added_condition_cb(void *cfield, void *)
  { ((Condition_Field *) cfield)->update_condition_menu(); }

// ----------------------------------------------------------------------------
//
void Condition_Field::update_condition_menu()
{
  List cnames;
  List clist = session.project().condition_list();
  for (int ci = 0 ; ci < clist.size() ; ++ci)
    cnames.append(new Stringy(((Condition *) clist[ci])->name()));
  cnames.sort_removing_duplicates(string_compare);
  add_edit_field_menu_entries(ws, cf, cnames);
  free_string_list_entries(cnames);
}

// ----------------------------------------------------------------------------
// Only top level view windows are put in the menu.
//
View_Menu::View_Menu(Session &s, Widget parent, const Stringy &name)
  : session(s), ws(s.window_system())
{
  menu = ws.option_menu(parent, name, NULL);

  update_view_menu();
  Notifier &n = session.notifier();
  n.notify_me(nt_added_view_to_project, view_added_cb, this);
  n.notify_me(nt_removed_view_from_project, view_deleted_cb, this);
  n.notify_me(nt_renamed_view, view_renamed_cb, this);

  set_view_choice(::selected_view(s));
}

// ----------------------------------------------------------------------------
//
View_Menu::~View_Menu()
{
  Notifier &n = session.notifier();
  n.dont_notify_me(nt_added_view_to_project, view_added_cb, this);
  n.dont_notify_me(nt_removed_view_from_project, view_deleted_cb, this);
  n.dont_notify_me(nt_renamed_view, view_renamed_cb, this);
}  

// ----------------------------------------------------------------------------
//
void View_Menu::set_view_choice(View *view)
{
  if (view && view->name() != ws.option_selected(menu))
    ws.set_option(menu, view->name());
}

// ----------------------------------------------------------------------------
//
View *View_Menu::selected_view()
{
  return session.project().find_view(ws.option_selected(menu));
}

// ----------------------------------------------------------------------------
//
void View_Menu::view_added_cb(void *vmenu, void *view)
{
  View *v = (View *) view;
  if (v->is_top_level_window())
    {
      WinSys &ws = v->session().window_system();
      ws.add_option(((View_Menu *) vmenu)->option_menu(), v->name());
    }
}

// ----------------------------------------------------------------------------
//
void View_Menu::view_deleted_cb(void *vmenu, void *view)
{
  View *v = (View *) view;
  if (v->is_top_level_window())
    {
      WinSys &ws = v->session().window_system();
      ws.delete_option(((View_Menu *)vmenu)->option_menu(), v->name());
    }
}

// ----------------------------------------------------------------------------
//
void View_Menu::view_renamed_cb(void *vmenu, void *view)
{
  ((View_Menu *) vmenu)->renamed_view((View *) view);
}

// ----------------------------------------------------------------------------
//
void View_Menu::renamed_view(View *v)
{
  if (v->is_top_level_window())
    {  
      update_view_menu();
      if (selected_view() == NULL && !ws.option_selected(menu).is_empty())
	ws.set_option(menu, v->name(), false);
    }
}

// ----------------------------------------------------------------------------
//
Widget View_Menu::option_menu()
  { return menu; }

// ----------------------------------------------------------------------------
//
void View_Menu::update_view_menu()
{
  ws.delete_all_options(menu);
  List vlist = session.project().top_level_views();
  for (int vi = 0 ; vi < vlist.size() ; ++vi)
    ws.add_option(menu, ((View *) vlist[vi])->name());
}

// ----------------------------------------------------------------------------
//
Spectrum_Menu::Spectrum_Menu(Session &s, Widget parent, const Stringy &name)
  : session(s), ws(s.window_system())
{
  menu = ws.option_menu(parent, name, NULL);

  update_spectrum_menu();
  Notifier &n = session.notifier();
  n.notify_me(nt_added_spectrum_to_project, spectrum_added_cb, this);
  n.notify_me(nt_removed_spectrum_from_project, spectrum_deleted_cb, this);
  n.notify_me(nt_renamed_spectrum, spectrum_renamed_cb, this);

  set_spectrum_choice(::selected_spectrum(s));
}

// ----------------------------------------------------------------------------
//
Spectrum_Menu::~Spectrum_Menu()
{
  Notifier &n = session.notifier();
  n.dont_notify_me(nt_added_spectrum_to_project, spectrum_added_cb, this);
  n.dont_notify_me(nt_removed_spectrum_from_project, spectrum_deleted_cb, this);
  n.dont_notify_me(nt_renamed_spectrum, spectrum_renamed_cb, this);
}

// ----------------------------------------------------------------------------
//
void Spectrum_Menu::spectrum_added_cb(void *smenu, void *spectrum)
{
  Spectrum *s = (Spectrum *) spectrum;
  WinSys &ws = s->session().window_system();
  ws.add_option(((Spectrum_Menu *) smenu)->option_menu(), s->name());
}

// ----------------------------------------------------------------------------
//
void Spectrum_Menu::spectrum_deleted_cb(void *smenu, void *spectrum)
{
  Spectrum *s = (Spectrum *) spectrum;
  WinSys &ws = s->session().window_system();
  ws.delete_option(((Spectrum_Menu *) smenu)->option_menu(), s->name());
}

// ----------------------------------------------------------------------------
//
void Spectrum_Menu::spectrum_renamed_cb(void *smenu, void *spectrum)
{
  ((Spectrum_Menu *) smenu)->renamed_spectrum((Spectrum *) spectrum);
}

// ----------------------------------------------------------------------------
//
void Spectrum_Menu::renamed_spectrum(Spectrum *sp)
{
  update_spectrum_menu();
  if (selected_spectrum() == NULL && !ws.option_selected(menu).is_empty())
    ws.set_option(menu, sp->name(), false);
}

// ----------------------------------------------------------------------------
//
void Spectrum_Menu::update_spectrum_menu()
{
  ws.delete_all_options(menu);
  const List &slist = session.project().spectrum_list();
  for (int si = 0 ; si < slist.size() ; ++si)
    {
      Spectrum *sp = (Spectrum *) slist[si];
      ws.add_option(menu, sp->name());
    }
}

// ----------------------------------------------------------------------------
//
void Spectrum_Menu::set_spectrum_choice(Spectrum *sp)
{
  if (sp && sp->name() != ws.option_selected(menu))
    ws.set_option(menu, sp->name());
}

// ----------------------------------------------------------------------------
//
Spectrum *Spectrum_Menu::selected_spectrum()
{
  return session.project().find_spectrum(ws.option_selected(menu));
}

// ----------------------------------------------------------------------------
//
Widget Spectrum_Menu::option_menu()
  { return menu; }

// ----------------------------------------------------------------------------
//
Stringy table_axis_label(Spectrum *sp, int axis, const Stringy &suffix)
{
  return formatted_string("w%d %s%s",
			  axis+1,
			  (sp == NULL ? "" : sp->nucleus_type(axis).cstring()),
			  suffix.cstring());
}
