// ----------------------------------------------------------------------------
//

#include "command.h"		// Use parse_key()
#include "notifier.h"		// use send_notice()
#include "session.h"		// use Session
#include "stringc.h"		// use Stringy, name_index()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uimode.h"
#include "utility.h"		// use array_size()
#include "winsystem.h"		// use WinSys, display

// ----------------------------------------------------------------------------
//
class mode_dialog : public Dialog
{
public:
  mode_dialog(Session &);
  ~mode_dialog();

  static mode_dialog *the(Session &);

  void show();
  Pointer_Mode mode();
  void set_mode(Pointer_Mode mode);
  void mode_changed();

private:
  Widget mode_menu;

  static void parse_key_cb(Widget, CB_Data, CB_Data);
};

// ----------------------------------------------------------------------------
//
struct pointer_mode_button
{
  Pointer_Mode mode;
  const char *button_name;
  const char *accelerator;
};

static struct pointer_mode_button mode_buttons[] =
{
  {MODE_SELECT, "select", "F1"},
  {MODE_CENTER, "center", "F2"},
  {MODE_GRIDBOTH, "addGridBoth", "F3"},
  {MODE_GRIDHOR, "addGridHorz", "F4"},
  {MODE_GRIDVERT, "addGridVert", "F5"},
  {MODE_LABEL, "addLabel", "F6"},
  {MODE_LINE, "addLine", "F7"},
  {MODE_PEAK, "findAddPeak", "F8"},
  {MODE_INTEGRATE, "integrate", "F10"},
  {MODE_ZOOM, "zoom", "F11"},
  {MODE_DUPZOOM, "duplicateZoom", "F12"},
  {MODE_COPYASSIGNMENT, "assignCopy", "sh-F1"},
  {MODE_GUESSASSIGNMENT, "assignGuess", "sh-F2"},
};

// ----------------------------------------------------------------------------
//
void mode_changed_cb(Widget, CB_Data dialog, CB_Data);

// ----------------------------------------------------------------------------
//
void show_mode_buttons(Session &s)
  { mode_dialog::the(s)->show(); }
void set_pointer_mode(Session &s, Pointer_Mode mode)
  { mode_dialog::the(s)->set_mode(mode); }
Pointer_Mode pointer_mode(Session &s)
  { return mode_dialog::the(s)->mode(); }

// ----------------------------------------------------------------------------
//
mode_dialog::mode_dialog(Session &s) : Dialog(s, "mode")
{
  mode_menu = ws.create_radio_buttons(dialog, "buttons");
  ws.row_attachments(mode_menu, mode_menu, END_OF_WIDGETS);

  for (int k = 0 ; k < array_size(mode_buttons) ; ++k)
    ws.add_radio_button(mode_menu, mode_buttons[k].button_name);

  ws.radio_callback(mode_menu, mode_changed_cb, this);

  ws.add_event_callback(key_press_event, dialog, parse_key_cb, this);
}

// ----------------------------------------------------------------------------
//
mode_dialog::~mode_dialog()
{
  session.dialog_table().delete_dialog("mode_dialog", this);

  ws.remove_event_callback(key_press_event, dialog, parse_key_cb, this);
}

// ----------------------------------------------------------------------------
// The default main_dialog instance.
//
mode_dialog *mode_dialog::the(Session &s)
{
  Stringy name = "mode_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new mode_dialog(s));
  return (mode_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void mode_dialog::show()
{
  ws.show_dialog(dialog);
}

// ----------------------------------------------------------------------------
//
void mode_dialog::parse_key_cb(Widget, CB_Data d, CB_Data event)
{
  mode_dialog *md = (mode_dialog *) d;
  md->session.command_dispatcher().parse_key_event(event);
}

// ----------------------------------------------------------------------------
//
Pointer_Mode mode_dialog::mode()
{
  Stringy choice = ws.radio_selection(mode_menu);
  return named_pointer_mode(choice);
}

// ----------------------------------------------------------------------------
//
void mode_dialog::set_mode(Pointer_Mode mode)
{
  ws.set_radio_button(mode_menu, pointer_mode_name(mode));
}

// ----------------------------------------------------------------------------
//
void mode_dialog::mode_changed()
{
  session.notifier().send_notice(nt_changed_pointer_mode, 0);
}

// ----------------------------------------------------------------------------
//
void mode_changed_cb(Widget, CB_Data dialog, CB_Data)
{
  mode_dialog *md = (mode_dialog *) dialog;
  md->mode_changed();
}

// ----------------------------------------------------------------------------
//
Pointer_Mode named_pointer_mode(const Stringy &mode_name)
{
  for (int k = 0 ; k < array_size(mode_buttons) ; ++k)
    if (mode_name == mode_buttons[k].button_name)
      return mode_buttons[k].mode;
  return MODE_SELECT;
}

// ----------------------------------------------------------------------------
//
Stringy pointer_mode_name(Pointer_Mode mode)
{
  for (int k = 0 ; k < array_size(mode_buttons) ; ++k)
    if (mode_buttons[k].mode == mode)
      return mode_buttons[k].button_name;
  return "";
}
