// ----------------------------------------------------------------------------
//
#include <stdio.h>		// Use FILE
#include <string.h>		// Use strcmp()

#include "command.h"		// use add_accelerator()
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "project.h"		// use Project
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "spoint.h"		// Use SPoint
#include "stringc.h"		// Use Stringy
#include "uicomponents.h"	// Use View_Menu
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uiview.h"		// Use View
#include "notifier.h"		// use Notifier
#include "winsystem.h"		// use WinSys

// ----------------------------------------------------------------------------
//
class region_dialog : public Dialog
{
public:
  region_dialog(Session &);
  ~region_dialog();

  static region_dialog *the(Session &);

  void show(View *view);
  void update(View *view);
  bool accelerator_goto(const Stringy &accelerator);
  void add_region(const View_Region &vr);

private:
  Widget name, accel, ranges, list;
  View_Menu *view_menu;
  View *view;

  static void will_delete_view_cb(void *rdialog, void *view);
  static void view_selected_cb(void *rdialog, void *view);
  static void current_cb(Widget, CB_Data, CB_Data);
  static void world_cb(Widget, CB_Data, CB_Data);
  static void select_cb(Widget, CB_Data, CB_Data);
  static void add_cb(Widget, CB_Data, CB_Data);
  static void delete_cb(Widget, CB_Data, CB_Data);
  static void show_region_cb(Widget, CB_Data, CB_Data);
  static void view_menu_cb(Widget, CB_Data, CB_Data);
  static void apply_mode_cb(Widget, CB_Data, CB_Data);
  static Stringy list_text(const View_Region &);

  void update_region_list();
  void current_region();
  void world_region();
  View_Region get_region();
  void set_region(const View_Region &vr);
  void delete_list_entry(const Stringy &text);
  void show_region();
  void apply_mode();
};

// ----------------------------------------------------------------------------
//
static bool goto_region_cb(Session &, const Stringy &, void *vregion);
static void remove_duplicate_regions(View_Regions &, const View_Region &);

// ----------------------------------------------------------------------------
//
void show_region_dialog(Session &s, View *view)
  { region_dialog::the(s)->show(view); }
void update_region_dialog(Session &s, View *view)
  { region_dialog::the(s)->update(view); }

// ----------------------------------------------------------------------------
//
Stringy region_dialog::list_text(const View_Region &vr)
{
  return formatted_string("%s - %s",
			  vr.name.cstring(),
			  vr.accelerator.cstring());
}

// ----------------------------------------------------------------------------
//
region_dialog::region_dialog(Session &s) : Dialog(s, "regionDialog")
{
  this->view = NULL;

  view_menu = new View_Menu(s, dialog, "viewMenu");
  ws.option_callback(view_menu->option_menu(), view_menu_cb, this);

  name = ws.edit_field(dialog, "name");
  accel = ws.edit_field(dialog, "accel");

  Table_Entry label = TABLE_LABEL;
  Table_Entry value = TABLE_TEXT_FIELD;
  ranges = ws.widget_table(dialog, "ranges", 5, 3,
			label, label, label,
			label, value, value,
			label, value, value,
			label, value, value,
			label, value, value);
  Widget cur_wor = ws.button_row(dialog, "curWor",
			      "current", current_cb, this,
			      "world", world_cb, this,
			      NULL);
  Widget scroller = ws.create_scrolled_list(dialog, "list", &list);
  ws.add_list_selection_callback(list, select_cb, this);
  Widget add_del = ws.button_row(dialog, "addDel",
			      "add", add_cb, this,
			      "delete", delete_cb, this,
			      NULL);
  Widget separator = ws.create_separator(dialog, "separator");
  Widget controls = ws.button_row(dialog, "controls",
			       "goto", show_region_cb, this,
			       "applyMode", apply_mode_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(scroller,
		     view_menu->option_menu(), name, accel, ranges, cur_wor,
		     scroller, add_del,
		     separator, controls, END_OF_WIDGETS);

  Notifier &n = session.notifier();
  n.notify_me(nt_will_delete_view, will_delete_view_cb, this);
}

// ----------------------------------------------------------------------------
//
region_dialog::~region_dialog()
{
  session.dialog_table().delete_dialog("region_dialog", this);

  Notifier &n = session.notifier();
  n.dont_notify_me(nt_will_delete_view, will_delete_view_cb, this);

  delete view_menu;
}

// ----------------------------------------------------------------------------
// The default region_dialog instance.
//
region_dialog *region_dialog::the(Session &s)
{
  Stringy name = "region_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new region_dialog(s));
  return (region_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void region_dialog::will_delete_view_cb(void *rdialog, void *view)
{
  region_dialog *rd = (region_dialog *) rdialog;
  View *v = (View *) view;

  if (rd->view == v)
    rd->update(NULL);
}

// ----------------------------------------------------------------------------
//
void region_dialog::view_selected_cb(void *rdialog, void *view)
{
  region_dialog *rd = (region_dialog *) rdialog;

  if (rd->shown())
    rd->update((View *) view);
}

// ----------------------------------------------------------------------------
//
void region_dialog::current_cb(Widget, CB_Data client_data, CB_Data)
{
  region_dialog *rd = (region_dialog *) client_data;
  rd->current_region();
}

// ----------------------------------------------------------------------------
//
void region_dialog::current_region()
{
  if (view)
    set_region(View_Region("current", "", view->view_region()));
}

// ----------------------------------------------------------------------------
//
void region_dialog::world_cb(Widget, CB_Data client_data, CB_Data)
{
  region_dialog *rd = (region_dialog *) client_data;
  rd->world_region();
}

// ----------------------------------------------------------------------------
//
void region_dialog::world_region()
{
  if (view)
    set_region(View_Region("world", "", view->spectrum()->ppm_region()));
}

// ----------------------------------------------------------------------------
//
View_Region region_dialog::get_region()
{
  int dim = (view == NULL ? 2 : view->spectrum()->dimension());
  SPoint ppm_min(dim), ppm_max(dim);
  Stringy region_name = ws.edit_value(name);
  Stringy region_accel = ws.edit_value(accel);

  for (int d = 0 ; d < dim ; ++d)
    {
      ppm_min[d] = ws.numeric_text_field(ws.table_element(ranges, d+1, 1));
      ppm_max[d] = ws.numeric_text_field(ws.table_element(ranges, d+1, 2));
    }

  View_Region vr(region_name, region_accel, SRegion(ppm_min, ppm_max));

  return vr;
}

// ----------------------------------------------------------------------------
//
void region_dialog::set_region(const View_Region &vr)
{
  ws.set_edit_value(name, vr.name);
  ws.set_edit_value(accel, vr.accelerator);
  for (int d = 0 ; d < vr.ppm_region.dimension() ; ++d)
    {
      ws.set_numeric_text_field(ws.table_element(ranges, d+1, 1),
			     "%.4f", vr.ppm_region.min[d]);
      ws.set_numeric_text_field(ws.table_element(ranges, d+1, 2),
			     "%.4f", vr.ppm_region.max[d]);
    }
}

// ----------------------------------------------------------------------------
//
void region_dialog::select_cb(Widget w, CB_Data client_data, CB_Data)
{
  region_dialog *rd = (region_dialog *) client_data;
  Stringy text = rd->ws.selected_list_item(w);

  const List &rlist = rd->session.project().view_regions.region_list();
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      View_Region *vr = (View_Region *) rlist[ri];
      if (list_text(*vr) == text)
	{ rd->set_region(*vr); return; }
    }
  rd->update_region_list();		// List widget is out of date.
}

// ----------------------------------------------------------------------------
//
void region_dialog::add_cb(Widget, CB_Data client_data, CB_Data)
{
  region_dialog *rd = (region_dialog *) client_data;
  rd->add_region(rd->get_region());
}

// ----------------------------------------------------------------------------
//
void region_dialog::add_region(const View_Region &vregion)
{
  delete_list_entry(list_text(vregion));
  View_Regions &vr = session.project().view_regions;
  vr.add_region(vregion);
  update_region_list();
}

// ----------------------------------------------------------------------------
//
void region_dialog::delete_cb(Widget, CB_Data client_data, CB_Data)
{
  region_dialog *rd = (region_dialog *) client_data;

  Stringy text;
  while ((text = rd->ws.selected_list_item(rd->list), !text.is_empty()))
    rd->delete_list_entry(text);
}

// ----------------------------------------------------------------------------
//
void region_dialog::delete_list_entry(const Stringy &text)
{
  List dlist;
  Project &proj = session.project();
  List copy = proj.view_regions.region_list();
  for (int ri = 0 ; ri < copy.size() ; ++ri)
    {
      View_Region *vr = (View_Region *) copy[ri];
      if (list_text(*vr) == text)
	proj.view_regions.delete_region(vr);
    }
  update_region_list();
}

// ----------------------------------------------------------------------------
//
void region_dialog::show_region_cb(Widget, CB_Data client_data, CB_Data)
{
  region_dialog *rd = (region_dialog *) client_data;
  rd->show_region();
}

// ----------------------------------------------------------------------------
//
void region_dialog::show_region()
{
  if (view)
    view->zoom(get_region().ppm_region);
}

// ----------------------------------------------------------------------------
//
void region_dialog::view_menu_cb(Widget, CB_Data client_data, CB_Data)
{
  region_dialog *rd = (region_dialog *) client_data;
  View *view = rd->view_menu->selected_view();
  if (view)
    rd->update(view);
}

// ----------------------------------------------------------------------------
//
void region_dialog::apply_mode_cb(Widget, CB_Data client_data, CB_Data)
{
  region_dialog *rd = (region_dialog *) client_data;
  rd->apply_mode();
}

// ----------------------------------------------------------------------------
//
void region_dialog::apply_mode()
{
  if (view)
    apply_mode_to_region(view, false, get_region().ppm_region);
}

// ----------------------------------------------------------------------------
//
void region_dialog::show(View *view)
{
  update(view);
  update_region_list();
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void region_dialog::update(View *view)
{
  this->view = view;

  Stringy dialog_title = (view == NULL ?
			  Stringy("Spectrum Regions") :
			  Stringy("Region ") + view->name());
  ws.set_dialog_title(dialog, dialog_title);

  view_menu->set_view_choice(view);

  current_region();

  Spectrum *sp = (view == NULL ? NULL : view->spectrum());
  int dim = (sp == NULL ? 2 : sp->dimension());
  for (int a = 0 ; a < dim ; ++a)
    ws.set_label_text(ws.table_element(ranges, a+1, 0),
		      table_axis_label(sp, a, ""));
  ws.manage_table_children(ranges, dim + 1, 3);
}

// ----------------------------------------------------------------------------
//
void region_dialog::update_region_list()
{
  ws.delete_list_items(list);
  const List &rlist = session.project().view_regions.region_list();
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      View_Region *vr = (View_Region *) rlist[ri];
      ws.add_list_item(list, list_text(*vr));
    }
}

// ----------------------------------------------------------------------------
//
bool region_dialog::accelerator_goto(const Stringy &accelerator)
{
  const List &rlist = session.project().view_regions.region_list();
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      View_Region *vr = (View_Region *) rlist[ri];
      if (accelerator == vr->accelerator)
	{
	  set_region(*vr);
	  show_region();
	  return true;
	}
    }

  return false;
}

// ----------------------------------------------------------------------------
//
View_Region::View_Region(const Stringy &name, const Stringy &accel,
			 const SRegion &ppm_reg)
{
  this->name = name;
  this->accelerator = accel;
  this->ppm_region = ppm_reg;
}

// ----------------------------------------------------------------------------
//
View_Regions::View_Regions(Command_Dispatcher &cd) : command_dispatcher(cd)
{
}

// ----------------------------------------------------------------------------
//
View_Regions::~View_Regions()
{
  delete_regions();
}

// ----------------------------------------------------------------------------
//
void View_Regions::add_region(const View_Region &vr)
{
  remove_duplicate_regions(*this, vr);
  View_Region *r = new View_Region(vr);
  rlist.append(r);
  if (!vr.accelerator.is_empty())
    command_dispatcher.add_accelerator("rg" + vr.accelerator, "",
				       goto_region_cb, r);
}

// ----------------------------------------------------------------------------
//
void View_Regions::delete_region(View_Region *vr)
{
  rlist.erase(vr);
  if (!vr->accelerator.is_empty())
    command_dispatcher.remove_accelerator("rg" + vr->accelerator);
  delete vr;
}

// ----------------------------------------------------------------------------
//
static bool goto_region_cb(Session &s, const Stringy &, void *vregion)
{
  Stringy accel = ((View_Region *) vregion)->accelerator;
  View *v = selected_view(s);
  return (v != NULL && s.project().view_regions.goto_region(accel, v));
}

// ----------------------------------------------------------------------------
//
bool View_Regions::goto_region(const Stringy &accelerator, View *v)
{
  region_dialog *d = region_dialog::the(v->session());
  d->update(v);
  return d->accelerator_goto(accelerator);
}

// ----------------------------------------------------------------------------
//
const List &View_Regions::region_list()
  { return rlist; }

// ----------------------------------------------------------------------------
//
void View_Regions::delete_regions()
{
  while (rlist.size() > 0)
    delete_region((View_Region *) rlist[0]);
}

// ----------------------------------------------------------------------------
//
static void remove_duplicate_regions(View_Regions &vreg, const View_Region &vr)
{
  List remove;
  const List &rlist = vreg.region_list();
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      View_Region *r = (View_Region *) rlist[ri];
      if (r->accelerator == vr.accelerator && r->name == vr.name)
	remove.append(r);
    }
  for (int ri = 0 ; ri < remove.size() ; ++ri)
    vreg.delete_region((View_Region *) remove[ri]);
}
