// ----------------------------------------------------------------------------
//
#include <sstream>		// Use ostringstream
#include <stdio.h>		// Use FILE
#include <string.h>		// use strchr()

#include "atom.h"		// use Atom
#include "condition.h"		// Use Condition
#include "group.h"		// Use Group
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "molecule.h"		// Use Molecule
#include "num.h"		// Use max()
#include "project.h"		// use Project
#include "reporter.h"		// use Reporter
#include "resonance.h"		// Use Resonance
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "stringc.h"		// Use Stringy
#include "uicomponents.h"	// Use Condition_Menu
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uipeak.h"		// Use show_peak_list_dialog()
#include "utility.h"		// Use message()
#include "winsystem.h"		// use WinSys, ...

static Stringy table_string(const List &gnames, const List &anames,
			    const List &reslist,
			    bool show_shifts, bool show_sdev,
			    bool show_counts);
static Stringy table_row(Stringy &rowname, Stringy (*etext)(Resonance *),
			 const List &anames, const List &rlist);
static List group_resonances(const List &rlist, const Stringy &gname);
static List filter_resonances(const List &rlist, bool (*accept)(Resonance *));
static bool has_assignment(Resonance *r);
static Resonance *find_resonance(const Stringy &aname, const List &rlist);
static List atom_names(const List &rlist);
static Stringy chemical_shift(Resonance *r);
static Stringy standard_deviation(Resonance *r);
static Stringy assignment_count(Resonance *r);
static Stringy assignment_exists(Resonance *r);
static List group_names(const List &rlist);
static int resonance_compare(const void *a, const void *b);
static int group_compare(const void *a, const void *b);
static Stringy text_rectangle(const Stringy &text,
			      int x1, int y1, int x2, int y2);
static void text_size(const Stringy &text, int *rows, int *cols);

// ----------------------------------------------------------------------------
//
class resonance_table_dialog : public Dialog
{
public:
  resonance_table_dialog(Session &);
  ~resonance_table_dialog();

  static resonance_table_dialog *the(Session &);

  void show(Spectrum *sp);
  void update();

private:
  Widget switches, row_heading, column_heading, values;
  Condition_Menu *cond;
  Stringy table;
  List group_names, atom_names;
  int rows_per_group;

  static void update_cb(Widget, CB_Data, CB_Data);
  static void save_cb(Widget, CB_Data, CB_Data);
  static void text_position_cb(Widget, CB_Data, CB_Data);

  bool write_file(const Stringy &path);
  Resonance *resonance_at_position(int row, int column);
};

// ----------------------------------------------------------------------------
//
void show_resonance_table_dialog(Session &s, Spectrum *sp)
  { resonance_table_dialog::the(s)->show(sp); }

// ----------------------------------------------------------------------------
//
resonance_table_dialog::resonance_table_dialog(Session &s)
  : Dialog(s, "resonanceTableDialog")
{
  cond = new Condition_Menu(s, dialog, "condition");

  const char *swnames[] = {"chemShifts",
			   "standardDev",
			   "counts",
			   "predefined",
			   NULL};

  switches = ws.switches(dialog, "switches", swnames);
  ws.set_switch(switches, "chemShifts", true);

  Widget scroller = ws.create_text_table(dialog, "text",
				      &row_heading, &column_heading, &values);

  ws.add_text_position_callback(values, text_position_cb, this);

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "update", update_cb, this,
			       "save", save_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(scroller, cond->option_menu(), switches, scroller,
		     separator, controls, END_OF_WIDGETS);
}

// ----------------------------------------------------------------------------
//
resonance_table_dialog::~resonance_table_dialog()
{
  session.dialog_table().delete_dialog("resonance_table_dialog", this);

  free_string_list_entries(group_names);
  free_string_list_entries(atom_names);

  ws.remove_text_position_callback(values, text_position_cb, this);

  delete cond;
}

// ----------------------------------------------------------------------------
// The default resonance_table_dialog instance.
//
resonance_table_dialog *resonance_table_dialog::the(Session &s)
{
  Stringy name = "resonance_table_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new resonance_table_dialog(s));
  return (resonance_table_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void resonance_table_dialog::update_cb(Widget, CB_Data client_data, CB_Data)
{
  resonance_table_dialog *atd = (resonance_table_dialog *) client_data;

  atd->update();
}

// ----------------------------------------------------------------------------
//
void resonance_table_dialog::save_cb(Widget, CB_Data client_data, CB_Data)
{
  resonance_table_dialog *atd = (resonance_table_dialog *) client_data;

  Condition *c = atd->cond->condition_chosen();
  Stringy file = (c ? c->name()+".shifts" : Stringy("assignment.shifts"));
  Stringy defpath = atd->session.project().list_path(file);
  Stringy path = atd->ws.saveas_file_dialog(atd->dialog, "Save Assignment Table",
					    defpath, "shifts", true);
  if (! path.is_empty())
    atd->write_file(path);
}

// ----------------------------------------------------------------------------
//
bool resonance_table_dialog::write_file(const Stringy &path)
{
  FILE *fp = fopen(path.cstring(), "w");
  if (fp == NULL)
    session.reporter().message("Couldn't write %s\n", path.cstring());

  if (fp)
    {
      fprintf(fp, "%s", table.cstring());
      fclose(fp);
    }

  return fp != NULL;
}

// ----------------------------------------------------------------------------
//
void resonance_table_dialog::text_position_cb(Widget w,
					       CB_Data atdialog,
					       CB_Data tcbdata)
{
  resonance_table_dialog *atd = (resonance_table_dialog *) atdialog;

  int row, column;
  atd->ws.selected_text_position(w, tcbdata, &row, &column);

  Resonance *r = atd->resonance_at_position(row, column);
  if (r)
    show_peak_list_dialog(atd->session, r);
}

// ----------------------------------------------------------------------------
// Return the resonance at the given table position.
//
Resonance *resonance_table_dialog::resonance_at_position(int row, int column)
{
  int chars_per_atom = 7;
  int g = row / rows_per_group;
  int a = column / chars_per_atom;

  if (g >= 0 && g < group_names.size() && a >= 0 && a < atom_names.size())
    {
      const Stringy &gname = *(Stringy *) group_names[g];
      const Stringy &aname = *(Stringy *) atom_names[a];
      Condition *c = cond->condition_chosen();
      if (c)
	return c->find_resonance(gname, aname);
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
void resonance_table_dialog::show(Spectrum *sp)
{
  if (sp)
    ws.set_option(cond->option_menu(), sp->condition()->fullname());

  update();
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void resonance_table_dialog::update()
{
  Condition *c = cond->condition_chosen();
  if (c == NULL)
    return;

  bool show_shifts = ws.switch_state(switches, "chemShifts");
  bool show_sdev = ws.switch_state(switches, "standardDev");
  bool show_counts = ws.switch_state(switches, "counts");
  bool show_predefined = ws.switch_state(switches, "predefined");

  List reslist = c->resonance_list();
  if (!show_predefined)
    reslist = filter_resonances(c->resonance_list(), has_assignment);
  reslist.sort(resonance_compare);

  free_string_list_entries(atom_names);
  atom_names = ::atom_names(reslist);
  atom_names.sort(string_compare);

  free_string_list_entries(group_names);
  group_names = ::group_names(reslist);

  rows_per_group = max((show_shifts ? 1 : 0) + (show_sdev ? 1 : 0) +
		       (show_counts ? 1 : 0), 1);

  table = table_string(group_names, atom_names, reslist,
		       show_shifts, show_sdev, show_counts);

  int rows, cols;
  text_size(table, &rows, &cols);
  ws.replace_text(column_heading, text_rectangle(table, 6, 0, cols-1, 0));
  ws.replace_text(row_heading, text_rectangle(table, 0, 1, 5, rows-1));
  ws.replace_text(values, text_rectangle(table, 6, 1, cols-1, rows-1));
}

// ----------------------------------------------------------------------------
//
static Stringy table_string(const List &gnames, const List &anames,
			    const List &reslist,
			    bool show_shifts, bool show_sdev, bool show_counts)
{
  std::ostringstream text;	// ostringstream concatenation more efficient than Stringy.

  Stringy heading = formatted_string("%6s", "");
  for (int ai = 0 ; ai < anames.size() ; ++ai)
    heading << formatted_string(" %6.6s", ((Stringy *) anames[ai])->cstring());
  text << heading << std::endl;

  for (int gi = 0 ; gi < gnames.size() ; ++gi)
    {
      Stringy *gname = (Stringy *) gnames[gi];
      List rlist = group_resonances(reslist, *gname);
      Stringy rowname = formatted_string("%6s", gname->cstring());
      if (show_shifts)
	text << table_row(rowname, chemical_shift, anames, rlist);
      if (show_sdev)
	text << table_row(rowname, standard_deviation, anames, rlist);
      if (show_counts)
	text << table_row(rowname, assignment_count, anames, rlist);
      if (!show_shifts && !show_sdev && !show_counts)
	text << table_row(rowname, assignment_exists, anames, rlist);
    }
  text << std::ends;

  std::string t = text.str();
  Stringy ts = t.c_str();

  return ts;
}

// ----------------------------------------------------------------------------
//
static Stringy table_row(Stringy &rowname, Stringy (*etext)(Resonance *),
			 const List &anames, const List &rlist)
{
  Stringy row = rowname;
  rowname = formatted_string("%6s", "");  // Subsequent rows don't show group.

  for (int ai = 0 ; ai < anames.size() ; ++ai)
    {
      Stringy *aname = (Stringy *) anames[ai];
      Resonance *r = find_resonance(*aname, rlist);
      row << (r ? etext(r) : Stringy("    -  "));
    }
  row << "\n";

  return row;
}

// ----------------------------------------------------------------------------
//
static List group_resonances(const List &rlist, const Stringy &gname)
{
  List grlist;
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      Resonance *r = (Resonance *) rlist[ri];
      if (r->group()->name() == gname)
	grlist.append(r);
    }
  return grlist;
}

// ----------------------------------------------------------------------------
//
static List filter_resonances(const List &rlist, bool (*accept)(Resonance *))
{
  List reslist;

  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      Resonance *r = (Resonance *) rlist[ri];
      if (accept(r))
	reslist.append(r);
    }

  return reslist;
}

// ----------------------------------------------------------------------------
//
static bool has_assignment(Resonance *r)
  { return r->assignment_count() > 0; }

// ----------------------------------------------------------------------------
//
static Resonance *find_resonance(const Stringy &aname, const List &rlist)
{
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      Resonance *r = (Resonance *) rlist[ri];
      if (r->atom()->name() == aname)
	return r;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
static List atom_names(const List &rlist)
{
  List anames;
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      Stringy name = ((Resonance *) rlist[ri])->atom()->name();
      if (!anames.contains(&name, string_compare))
	anames.append(new Stringy(name));
    }
  return anames;
}

// ----------------------------------------------------------------------------
//
static Stringy chemical_shift(Resonance *r)
  { return formatted_string(" %#6.4g", r->frequency()); }
static Stringy standard_deviation(Resonance *r)
  { return formatted_string(" %#6.4f", r->frequencySigma()); }
static Stringy assignment_count(Resonance *r)
  { return formatted_string(" %6d", r->assignment_count()); }
static Stringy assignment_exists(Resonance *)
  { return formatted_string("    *  "); }

// ----------------------------------------------------------------------------
//
static List group_names(const List &rlist)
{
  List gnames;
  for (int ri = 0 ; ri < rlist.size() ; ++ri)
    {
      Stringy name = ((Resonance *) rlist[ri])->group()->name();
      if (!gnames.contains(&name, string_compare))
	gnames.append(new Stringy(name));
    }
  return gnames;
}

// ----------------------------------------------------------------------------
//
static int resonance_compare(const void *a, const void *b)
{
  Resonance *r1 = (Resonance *) a;
  Resonance *r2 = (Resonance *) b;

  int cmp = group_compare(r1->group(), r2->group());
  if (cmp == 0)
    {
      Stringy a1 = r1->atom()->name();
      Stringy a2 = r2->atom()->name();
      cmp = string_compare(&a1, &a2);
    }

  return cmp;
}

// ----------------------------------------------------------------------------
//
static int group_compare(const void *a, const void *b)
{
  Group *g1 = (Group *) a;
  Group *g2 = (Group *) b;

  if (g1->number() > g2->number())	return +1;
  else if (g1->number() < g2->number()) return -1;

  Stringy sym1 = g1->symbol();
  Stringy sym2 = g2->symbol();

  return string_compare(&sym1, &sym2);
}

// ----------------------------------------------------------------------------
//
static void text_size(const Stringy &text, int *rows, int *cols)
{
  int lines = 0, line_length = 0;
  const char *s = text.cstring();
  for (const char *nl = strchr(s, '\n') ; nl ; s = nl+1, nl = strchr(s, '\n'))
    {
      lines += 1;
      line_length = max(line_length, static_cast<int> (nl - s));
    }
  *rows = lines;
  *cols = line_length;
}

// ----------------------------------------------------------------------------
//
static Stringy text_rectangle(const Stringy &text,
			      int x1, int y1, int x2, int y2)
{
  Stringy tr;

  const char *s = text.cstring();
  int line = 0;
  for (const char *nl = strchr(s, '\n') ;
       nl && line <= y2 ;
       s = nl+1, line += 1, nl = strchr(s, '\n'))
    if (line >= y1)
      {
	if (s + x2 < nl)	tr << substring(s+x1, s+x2+1);
	else if (s + x1 < nl)	tr << substring(s+x1, nl);
	if (line < y2)		tr << "\n";
      }

  return tr;
}
