// ----------------------------------------------------------------------------
// Views display contoured spectra with peak markers and assignment labels,
// ppm scales, resonances panels, 1-D slices, scrollbars, ...,
// and allow zooming and peak selection with the mouse.
//
// The View class is derived from the View_Window class which manages
// the generic user interface (window layout and events) and knows nothing
// about spectra or peaks.
//

#include <float.h>		// use FLT_MAX
#include <math.h>		// use sqrt(), log(), fabs()

#include "assigncopy.h"		// Use assignment_copy()
#include "assignguess.h"	// use Guess_Assignments
#include "axismap.h"		// Use Axis_Map
#include "color.h"
#include "command.h"		// use Command_Dispatcher
#include "condition.h"
#include "contour.h"
#include "grid.h"		// Use Grid
#include "integrate.h"		// Use integrate_list()
#include "label.h"		// Use Label
#include "line.h"		// Use Line
#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "num.h"
#include "ornament.h"
#include "peak.h"		// use Peak
#include "peakgp.h"		// use PeakGp
#include "peakpick.h"		// Use peak_pick_region()
#include "print.h"
#include "project.h"
#include "reporter.h"		// use Reporter
#include "session.h"		// use Session
#include "spectrum.h"
#include "resonance.h"
#include "uicscale.h"		// Use ContourScale
#include "uidialog.h"		// use Dialog_Table
#include "uidialogs.h"		// Use show_label_dialog()
#include "uimain.h"		// Use query()
#include "uimenu.h"		// Use popup_view_menu()
#include "uimode.h"		// Use pointer_mode()
#include "uipkpanel.h"		// Use Peak_Panel
#include "uirpanel.h"		// Use RPanel
#include "uiscale.h"		// Use scale
#include "uislice.h"		// Use Slice
#include "uiwait.h"		// Use wait_callbacks()
#include "undo.h"		// Use Undo
#include "utility.h"		// Use fatal_error()
#include "uiview.h"
#include "winsystem.h"		// Use modal_dialog_shown()

#define	SNAP_DISTANCE	10	// Snapping lines and grids to peaks (pixels)

// ----------------------------------------------------------------------------
//
class Ornament_Routines : public ODraw
{
public:
  Ornament_Routines(View *view);
  virtual void set_drawing_color(const Color &color);
  virtual void draw_line(double x1, double y1, double x2, double y2);
  virtual void draw_text(double cx, double cy, const Stringy &text);
  virtual void text_size(const Stringy &text, double *width,
			 double *ascent, double *descent);
  virtual void draw_rectangle(Rectangle r);
  virtual void draw_triangle(double x1, double y1,
			     double x2, double y2,
			     double x3, double y3);
  virtual void draw_ellipse(Rectangle r);
  virtual int axis(Axis a);
  virtual double aspect();
  virtual SRegion clip_region();
  virtual double ornament_size(Ornament_Type type, int axis);
  virtual double select_size(int axis);
  virtual void xy_point(const SPoint &p, double *x, double *y);
  virtual bool is_visible(const Ornament *op, Rectangle r);
  virtual int in_plane(const SPoint &p);
  virtual bool black_labels();
  virtual bool subtract_fit_peaks();
private:
  View *v;
};

static Rectangle full_view(Spectrum *sp, int xaxis, int yaxis);
static SPoint initial_pixel_size(Spectrum *sp);
static SPoint initial_depth(Spectrum *sp);
static SPoint standard_font_height(Spectrum *sp);
static SPoint standard_scroll_step(Spectrum *sp);
static SRegion standard_redraw_tile(Spectrum *sp);
static void draw_contour_path(float *xy, int len, View *from_view,
			      bool transpose, View *onto_view);
static void trim_zoom_stack(List &zs);
static void free_sregion_list(List &srlist);
static void position_axes(const view_axis &from, const List *to_axes);
static void position_axis(View *view, int axis, View *from, int from_axis);
static int in_plane(const SPoint &p, View *v);
static double distance(Ornament *op, View *view, double x, double y);
static void do_assignment_copy(View *view);
static bool integrate_region(View *view, const SRegion &region);
static List region_peaklets(View *view, const SRegion &region);
static void select_ornament(Ornament *op, bool extend, Project &);
static void select_crosspeaks(View *view, bool extend, const SRegion &region);
static void select_ornaments(View *view, bool extend, const SRegion &region);
static Ornament *snap_nearest(View *view, double x, double y, Ornament_Type type);
static CrossPeak *ornament_snap_nearest(View *view, double x, double y);
static void move_ornament(Ornament *op, double dx, double dy, View *view);
static bool is_visible(const Ornament *op, View *view, Rectangle r);
static bool ok_to_unload_spectrum(Session &,Spectrum *sp);
static bool map_crosshair(View *from, const SPoint &p,
			  View *to, Axis toaxis, double *to_position);

// ----------------------------------------------------------------------------
//
Contour_Parameters::Contour_Parameters()
{
  levels.factor	= 1;
  levels.levels	= 0;
  levels.lowest	= 0;
  set_color_scheme("white");
}

// ----------------------------------------------------------------------------
//
Contour_Parameters::Contour_Parameters(bool positive)
{
  levels.factor	= 1.4;
  levels.levels	= 1;
  levels.lowest	= (positive ? 10000 : -10000);
  set_color_scheme(positive ? "red" : "green");
}

// ----------------------------------------------------------------------------
//
Contour_Parameters::Contour_Parameters(const Contour_Parameters &cp)
{
  *this = cp;
}

// ----------------------------------------------------------------------------
//
Contour_Parameters::~Contour_Parameters()
{
  free_color_list(colors);
}

// ----------------------------------------------------------------------------
//
Contour_Parameters &Contour_Parameters::operator=(const Contour_Parameters &cp)
{
  this->levels = cp.levels;
  this->clr_scheme = cp.clr_scheme;

  //
  // Duplicate color list
  //
  free_color_list(colors);
  const List &clist = cp.colors;
  for (int ci = 0 ; ci < clist.size() ; ++ci)
    colors.append(new Color(*(Color *)clist[ci]));

  return *this;
}

// ----------------------------------------------------------------------------
//
const Color &Contour_Parameters::level_color(double level) const
{
  if (colors.empty())
    fatal_error("Contour_Parameters:level_color(): color list empty\n");

  if (colors.size() == 1)
    return *(Color *) colors[0];

  if (levels.levels == 0 || levels.lowest == 0 || levels.factor == 0)
    return *(Color *) colors[0];

  double r = level / levels.lowest;
  if (r <= 0)
    return *(Color *) colors[0];

  double e = log(r) / log(levels.factor);
  int i = nearest_int(colors.size() * e / levels.levels);
  if (i < 0)
    i = 0;
  else if (i >= colors.size())
    i = colors.size() - 1;

  return *(Color *) colors[i];
}

// ----------------------------------------------------------------------------
//
Stringy Contour_Parameters::color_scheme() const
  { return clr_scheme; }

// ----------------------------------------------------------------------------
//
void Contour_Parameters::set_color_scheme(const Stringy &scheme)
{
  clr_scheme = scheme;
  free_color_list(colors);
  if (clr_scheme == "red-blue")
    colors = color_list("red", "dark orange", "orange",
			"gold", "yellow", "chartreuse",
			"green", "spring green",
			"cyan", "deep sky blue", "blue", NULL);
  else if (clr_scheme == "red-yellow")
    colors = color_list("red", "dark orange", "orange",
			"gold", "yellow", NULL);
  else if (clr_scheme == "green-blue")
    colors = color_list("green", "spring green",
			"cyan", "deep sky blue", "blue", NULL);
  else
    colors.append(new Color(scheme));
}

// ----------------------------------------------------------------------------
//
View_Settings::View_Settings(Spectrum *sp)
  : pos_contours(true), neg_contours(false)
{
  if (sp)
    view_name = sp->name();

  int dim = (sp == NULL ? 2 : sp->dimension());
  axis_order = IPoint(dim);
  for (int a = 0 ; a < dim ; ++a)
    axis_order[a] = dim - a - 1;

  x = 100;				// default placement
  y = 100;
  width = 500;				// default size
  height = 500;
  if (sp)
    {
      center = sp->ppm_region().center();
      pixel_size = initial_pixel_size(sp);
      visible_depth = initial_depth(sp);
      double noise = sp->noise_level();
      pos_contours.levels.lowest = 5 * noise;
      neg_contours.levels.lowest = -5 * noise;
    }
  subtract_fit_peaks = false;
  scale_units = PPM;
  show_view = true;
  show_ornaments = true;
  show_peaks = true;
  show_peakgroups = true;
  show_labels = true;
  show_lines = true;
  show_grids = true;
  show_scales = true;
  show_scrollbars = true;
  show_resonance_panels = false;
  filter_resonances = false;
  show_slices = false;
  slice_auto_scale = true;
  slice_subtract_peaks = false;
  show_peak_info = false;
  show_contour_scale = false;
  show_crosshair = true;
  show_crosshair_from_other_views = true;
  show_transpose_crosshair = false;
  show_nucleus_names = false;
}

// ----------------------------------------------------------------------------
//
bool View_Settings::show_ornament(Ornament_Type t)
{
  switch(t)
    {
    case peak:		return show_peaks;
    case peakgroup:	return show_peakgroups;
    case label:		return show_labels;
    case line:		return show_lines;
    case grid:		return show_grids;
    }
  return false;	// t = -1 when peak being deleted.
}

// ----------------------------------------------------------------------------
//
void View_Settings::show_ornament(Ornament_Type t, bool show)
{
  switch(t)
    {
    case peak:		 show_peaks = show;		break;
    case peakgroup:	 show_peakgroups = show;	break;
    case label:		 show_labels = show;		break;
    case line:		 show_lines = show;		break;
    case grid:		 show_grids = show;		break;
    }
}

/*
 * Create a View.
 */
#define VIEW_REPAINT_BLOCKSIZE CONTOUR_TILE_SIZE
#define VIEW_REPAINT_PRIORITY -20
View::View(Session &ses, Widget parent, Spectrum *sp,
	   const View_Settings &settings)
     : View_Window(ses.window_system(), parent,
		   settings.width, settings.height,
		   settings.axis_order, settings.center, settings.pixel_size,
		   settings.visible_depth,
		   sp->ppm_region(),
		   standard_font_height(sp),
		   standard_scroll_step(sp),
		   standard_redraw_tile(sp),
		   VIEW_REPAINT_PRIORITY,
		   ses.project().preferences.contour_graying,
		   popup_view_menu(ses)),
       ses(ses),
       config(settings)
{
  this->spect = sp;
  this->mPts = 0;
  this->odroutines = new Ornament_Routines(this);
  this->crosshair_drawn[X] = false;
  this->crosshair_drawn[Y] = false;
  this->transpose_crosshair_drawn[X] = false;
  this->transpose_crosshair_drawn[Y] = false;
  this->local_region = SRegion(sp->dimension());
  this->pick.ornament = NULL;
  this->pick.down = false;
  this->pick.extend = false;
  this->pick.drag = false;

  this->xscale = NULL;
  this->yscale = NULL;
  this->xrpanel = NULL;
  this->yrpanel = NULL;
  this->xslice = NULL;
  this->yslice = NULL;
  this->pkpanel = NULL;
  this->cscale = NULL;

  set_screen_location(settings.x, settings.y);
  set_title(fullname(), name());
  set_contour_parameters(settings.pos_contours);
  set_contour_parameters(settings.neg_contours);
  configure_edge_panels(settings, true);
  update_label_size();

  Notifier &n = session().notifier();
  n.notify_me(nt_shifted_spectrum, ppm_scale_shifted_cb, this);
  n.notify_me(nt_created_ornament, ornament_erase_cb, this);
  n.notify_me(nt_will_change_ornament, ornament_erase_cb, this);
  n.notify_me(nt_changed_ornament, ornament_erase_cb, this);
  n.notify_me(nt_will_delete_ornament, will_delete_ornament_cb, this);
  n.notify_me(nt_resized_spectrum_labels, labels_resized_cb, this);
  n.notify_me(nt_changed_resonance, resonance_update_cb, this);
  n.notify_me(nt_added_resonance_to_condition, resonance_update_cb, this);
  n.notify_me(nt_removed_resonance_from_condition, resonance_update_cb, this);

  show_view(settings.show_view);

  n.send_notice(nt_created_view, this);
}

// ----------------------------------------------------------------------------
//
void View::mapped_or_unmapped()
{
  Notifier &n = session().notifier();
  n.send_notice(nt_showed_view, this);
}

/*
 * Return true if it is OK to unload this Spectrum.
 */
static bool ok_to_unload_spectrum(Session &s, Spectrum *sp)
{
	if (sp->save_file().NeedsSaving()) {
	  Stringy msg = formatted_string("Spectrum %s needs saving.\n",
					 sp->fullname().cstring());
	  if (query(s, msg, "Unload it, losing all changes", "Cancel unload")
	      != 1)
	    return false;
	}
	return true;
}

// ----------------------------------------------------------------------------
//
Session &View::session() const
  { return ses; }

// ----------------------------------------------------------------------------
//
Stringy View::name() const
  { return config.view_name; }

// ----------------------------------------------------------------------------
//
Stringy View::fullname() const
{
  Stringy condname = spectrum()->condition()->fullname();
  return (condname.is_empty() ? name() : condname + " / " + name());
}

// ----------------------------------------------------------------------------
//
Spectrum *View::spectrum() const
  { return spect; }

// ----------------------------------------------------------------------------
//
Rectangle View::full_view()
{
  return ::full_view(spectrum(), axis(X), axis(Y));
}

// ----------------------------------------------------------------------------
//
static Rectangle full_view(Spectrum *sp, int xaxis, int yaxis)
{
  return Rectangle(sp->ppm_region().min[xaxis], sp->ppm_region().min[yaxis], 
		   sp->ppm_region().max[xaxis], sp->ppm_region().max[yaxis]);
}

// ----------------------------------------------------------------------------
// Initial display scale is one index unit per screen pixel along axis 0.
// Other axes are adjusted according to spectrum aspect defaults.
//
static SPoint initial_pixel_size(Spectrum *sp)
{
  int dim = sp->dimension();
  SPoint pixel_size(dim);
  for (int a = 0 ; a < dim ; ++a)
    pixel_size[a] = sp->nucleus_ppm_factor(a);

  double f = 0;
  for (int a = 0 ; a < dim ; ++a)
    f = max(f, sp->scale(pixel_size[a], a, PPM, INDEX));
  pixel_size *= 1 / f;

  return pixel_size;
}

// ----------------------------------------------------------------------------
//
static SPoint standard_font_height(Spectrum *sp)
{
  int dim = sp->dimension();
  SPoint fh(dim);
  for (int a = 0 ; a < dim ; ++a)
    {
      double f = sp->nucleus_ppm_factor(a) / sp->nucleus_ppm_factor(0);
      fh[a] = sp->ornament_size(label) * f;
    }
  return fh;
}

// ----------------------------------------------------------------------------
//
static SPoint standard_scroll_step(Spectrum *sp)
{
  int dim = sp->dimension();
  SPoint step(dim);
  for (int a = 0 ; a < dim ; ++a)
    step[a] = sp->scale(1.0, a, INDEX, PPM);
  return step;
}

// ----------------------------------------------------------------------------
// Initial display depth is 1 data plane.
//
static SPoint initial_depth(Spectrum *sp)
{
  int dim = sp->dimension();
  SPoint depth(dim);
  for (int a = 0 ; a < dim ; ++a)
    depth[a] = sp->scale(1.0, a, INDEX, PPM);
  return depth;
}

// ----------------------------------------------------------------------------
//
static SRegion standard_redraw_tile(Spectrum *sp)
{
  IRegion index_tile(sp->dimension());
  for (int a = 0 ; a < sp->dimension() ; ++a)
    {
      index_tile.min[a] = 0;
      index_tile.max[a] = VIEW_REPAINT_BLOCKSIZE;
    }
  return sp->map(index_tile, INDEX, PPM);
}

// ----------------------------------------------------------------------------
// Reflect the viewed region across the diagonal for homonuclear.
// Reflect the viewed region about the view middle for heteronuclear.
// Keep the pixel sizes fixed.
//
void View::swap_axes()
{
  IPoint new_axis_order = axis_order();
  new_axis_order[0] = axis_order()[1];
  new_axis_order[1] = axis_order()[0];
  set_axis_order(new_axis_order);

  SPoint c = center();
  if (is_homonuclear())
    swap(&c[axis(X)], &c[axis(Y)]);
  set_view(c, pixel_size());
}

// ----------------------------------------------------------------------------
// Cyclically permute the axes keeping the center of the view fixed.
//
void View::roll_axes()
{
  set_axis_order(axis_order().roll());
}

// ----------------------------------------------------------------------------
//
void View::set_axis_order(const IPoint &axis_order)
{
  View_Window::set_axis_order(axis_order);
  update_scales();			// Redisplay scales
}

// ----------------------------------------------------------------------------
//
void View::labels_resized_cb(void *v, void *s)
{
  View *view = (View *) v;
  Spectrum *sp = (Spectrum *)s;
  if (sp == view->spectrum())
    view->update_label_size();
}

// ----------------------------------------------------------------------------
//
void View::resonance_update_cb(void *v, void *r)
{
  View *view = (View *) v;
  Resonance *res = (Resonance *)r;
  if (res->condition() == view->spectrum()->condition())
    view->erase_resonances();
}

// ----------------------------------------------------------------------------
//
void View::update_label_size()
{
  int dim = dimension();
  SPoint fh(dim);
  for (int a = 0 ; a < dim ; ++a)
    fh[a] = ornament_size(label, a);
  set_font_height(fh);
}

// ----------------------------------------------------------------------------
//
class Screen_Routines : public Drawing_Routines
{
public:
  Screen_Routines(View *view)
    { this->view = view; }
  virtual void set_contour_level(float level)
    {
      const Contour_Parameters &cp = view->contour_parameters(level);
      view->set_drawing_color(cp.level_color(level));
    }
  virtual void draw_contour_path(float *xy, int len)
    {
      bool transpose = (view->axis(X) > view->axis(Y));
      ::draw_contour_path(xy, len, view, transpose, view);
    }

private:
  View *view;
};

// ----------------------------------------------------------------------------
//
class Overlay_Routines : public Drawing_Routines
{
public:
  Overlay_Routines(View *from_view, View *onto_view, Axis_Map *axis_map)
    {
      this->from_view = from_view;
      this->onto_view = onto_view;
      this->axis_map = axis_map;
    }
  virtual void set_contour_level(float level)
    {
      const Contour_Parameters &cp = from_view->contour_parameters(level);
      onto_view->set_drawing_color(cp.level_color(level));
    }
  virtual void draw_contour_path(float *xy, int len)
    {
      int from_x = axis_map->invert(onto_view->axis(X));
      int from_y = axis_map->invert(onto_view->axis(Y));
      bool transpose = (from_x > from_y);
      ::draw_contour_path(xy, len, from_view, transpose, onto_view);
    }

private:
  View *from_view, *onto_view;
  Axis_Map *axis_map;
};

// ----------------------------------------------------------------------------
//
static void draw_contour_path(float *xy, int len, View *from_view,
			      bool transpose, View *onto_view)
{
  float *x = (transpose ? xy + 1 : xy);
  float *y = (transpose ? xy : xy + 1);
  for (int s = 0 ; s < len-1 ; ++s)
    {
      double x1_ppm = from_view->map((double) x[2*s], X, INDEX, PPM);
      double y1_ppm = from_view->map((double) y[2*s], Y, INDEX, PPM);
      double x2_ppm = from_view->map((double) x[2*(s+1)], X, INDEX, PPM);
      double y2_ppm = from_view->map((double) y[2*(s+1)], Y, INDEX, PPM);
      onto_view->draw_line(x1_ppm, y1_ppm, x2_ppm, y2_ppm);
    }

  //
  // The following draws the terminal (pixel) point of the path.
  // This is needed because draw_line() does not draw the final pixel.
  // This allows segments to be chained together without drawing any
  // pixel more that once.  Unfortunately there is no guarantee that
  // segments of contour loops will be chained together with the same
  // orientation.  So I need to draw the end pixels to avoid single
  // pixel gaps occuring at contour tile boundaries.
  //
  // There is another more serious problem causing contour path gaps
  // at tile boundaries.  The contour tiles overlap on one pixel boundaries.
  // When a tile is drawn it erases its screen area.  This erases the pixels
  // on the boundary from neighboring tiles.  And these pixels will not be
  // redrawn.  By drawing end point pixels of paths segments that cross a
  // tile boundary perpendicularly don't get messed up.  But glanzing crosses
  // get a few pixels deleted.
  //
  if (len > 1 && (x[0] != x[2*(len-1)] || y[0] != y[2*(len-1)]))
    {
      double x_ppm = from_view->map((double) x[2*(len-1)], X, INDEX, PPM);
      double y_ppm = from_view->map((double) y[2*(len-1)], Y, INDEX, PPM);
      onto_view->draw_point(x_ppm, y_ppm);
    }
			  
}

// ----------------------------------------------------------------------------
//
void View::redraw_rectangle(Rectangle r)
{
  bool blanked_x = false, blanked_y = false;
  if (contains_crosshair(r))
    blank_crosshairs(&blanked_x, &blanked_y);

  erase_rectangle(r, false);

  SRegion region = flat_region(r);
  Screen_Routines sr(this);
  Project &proj = session().project();
  Contour_Cache *cc = proj.contour_cache();
  draw_contours(spectrum(), region,
		positive_contours().levels, negative_contours().levels,
		config.subtract_fit_peaks, sr, cc, NULL);

  List vlist = proj.overlaid_views(this);
  for (int vi = 0 ; vi < vlist.size() ; ++vi)
    {
      View *v = (View *) vlist[vi];
      Spectrum *sp = v->spectrum();
      Axis_Map axismap;
      if (identity_axis_map(sp, spectrum(), &axismap) ||
	  unique_axis_map(sp, spectrum(), &axismap))
	{
	  SRegion vregion = axismap.invert(region);
	  Overlay_Routines ovr(v, this, &axismap);
	  draw_contours(sp, vregion, v->positive_contours().levels,
			v->negative_contours().levels,
			v->config.subtract_fit_peaks, ovr, cc, NULL);
	}
    }

  ornament_update(r);

  unblank_crosshairs(blanked_x, blanked_y);
}

// ----------------------------------------------------------------------------
//
void View::ornament_update(const Rectangle &r)
{
  if (ornament_update_rectangle.no_interior())
    ornament_update_rectangle = r;
  else
    ornament_update_rectangle.encompass(r);

  if (repaint_region().is_empty())
    {
      ornament_update_rectangle.clip(view_rectangle());

      bool blanked_x, blanked_y;
      blank_crosshairs(&blanked_x, &blanked_y);

      display_ornaments(ornament_update_rectangle);

      unblank_crosshairs(blanked_x, blanked_y);

      ornament_update_rectangle = Rectangle();
    }
}

// ----------------------------------------------------------------------------
// Display all ornaments overlapping a rectangle.
//
void View::display_ornaments(Rectangle rect)
{
  List olist = visible_ornaments(rect);
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      op->display(ornament_drawing_routines());
      if (!local_ornaments.contains(op))
	local_ornaments.append(op);
    }
}

// ----------------------------------------------------------------------------
// This routine on a 1995 DEC Alpha with about 3000 peaks and 3000 labels
// takes about 1 second to execute.  This causes a bug where pressing
// on a scrollbar single step button causes two steps to be taken -- mostly
// a problem when trying to single step through planes of a 3D spectrum.
// The trouble is that this routine is called in order to redraw the ornaments
// before the button release event has been received.  This delays the button
// release event and in Tk 8.0.5 this makes it appear that the user held the
// button down causing a second scrollbar step to be taken.  I don't see a
// simple way to fix this problem.
//
List View::visible_ornaments(Rectangle rect)
{
  List visible;

  if (config.show_ornaments)
    {
      const List &olist = spectrum()->ornaments();
      for (int oi = 0 ; oi < olist.size() ; ++oi)
	{
	  Ornament *op = (Ornament *) olist[oi];
	  if (ornaments_shown(op->type()) && is_visible(op, this, rect))
	    visible.append(op);
	}
    }
  return visible;
}

// ----------------------------------------------------------------------------
//
ODraw &View::ornament_drawing_routines()
  { return *odroutines; }

// ----------------------------------------------------------------------------
//
void View::buttondown(double x, double y, bool shift)
{
  WinSys &ws = session().window_system();
  ws.set_keyboard_focus_child(drawing_widget());

	erase_crosshair(X);
	erase_crosshair(Y);

	pick.downX	= x;
	pick.downY	= y;
	pick.lastX	= x;
	pick.lastY	= y;

	pick.ornament = pick_ornament(x, y);

	pick.down = true;
	pick.extend = shift;
	pick.drag = false;
}

// ----------------------------------------------------------------------------
//
CrossPeak *View::over_visible_crosspeak(double x, double y)
{
  Ornament *op = over_visible_ornament(x, y);
  return (op && is_crosspeak(op)) ? (CrossPeak *) op : NULL;
}

// ----------------------------------------------------------------------------
//
Ornament *View::pick_ornament(double x, double y)
{
	Ornament *nearest = over_visible_ornament(x, y);
	bool pickable = false;

	if (nearest)
	  {
	    Ornament_Type type = nearest->type();
	    int mode = pointer_mode(session());

	    /*
	     * Depending on the mode, pick on different objects
	     */
	    if (mode == MODE_SELECT)
	      pickable = true;
	    else if (mode == MODE_GRIDBOTH ||
		     mode == MODE_GRIDHOR ||
		     mode == MODE_GRIDVERT)
	      pickable = (type == grid);
	    else if (mode == MODE_LABEL)
	      pickable = (type == label || type == peak || type == peakgroup);
	    else if (mode == MODE_PEAK
		     || mode == MODE_CENTER
		     || mode == MODE_COPYASSIGNMENT
		     || mode == MODE_GUESSASSIGNMENT)
	      pickable = (type == label || type == peak || type == peakgroup);
	  }

	return (pickable ? nearest : NULL);
}

// ----------------------------------------------------------------------------
//
Ornament *View::over_visible_ornament(double x, double y)
{
	Ornament	*nearest_op = NULL;

	if (config.show_ornaments)
	  {
	    if (!(local_region == view_region()))
	      {
		local_region = view_region();
		local_ornaments = ornaments(local_region);
	      }
	    double		nearest_distance = FLT_MAX;
	    Rectangle		r = Rectangle(x, y, x, y);
	    SRegion		s = thicken(r);
	    ODraw &dr = ornament_drawing_routines();
	    const List &olist = local_ornaments;
	    for (int oi = 0 ; oi < olist.size() ; ++oi)
	      {
		Ornament *op = (Ornament *) olist[oi];

		if (ornaments_shown(op->type()) &&
		    op->selection_region(dr).intersects(s))
		  {
		    double d = ::distance(op, this, x, y);
		    if (d < nearest_distance)
		      {
			nearest_op = op;
			nearest_distance = d;
		      }
		  }
	      }
	  }

	return nearest_op;
}

// ----------------------------------------------------------------------------
//
List View::ornaments(const SRegion &region)
{
  List list;

  ODraw &dr = ornament_drawing_routines();
  const List &olist = spectrum()->ornaments();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (op->selection_region(dr).intersects(region))
	list.append(op);
    }

  return list;
}

// ----------------------------------------------------------------------------
//
void View::ornament_erase_cb(void *v, void *o)
{
  View *view = (View *) v;
  Ornament *op = (Ornament *)o;
  if (op->spectrum() == view->spectrum() && view->ornaments_shown(op->type()))
    view->erase(op);
}

// ----------------------------------------------------------------------------
//
void View::will_delete_ornament_cb(void *v, void *o)
{
  View *view = (View *) v;
  Ornament *op = (Ornament *)o;
  if (op->spectrum() == view->spectrum())
    {
      if (view->ornaments_shown(op->type()))
	view->erase(op);
      view->local_ornaments.erase(op);
      if (view->pick.ornament == op)
	view->pick.ornament = NULL;
    }
}

// ----------------------------------------------------------------------------
//
void View::buttonup(double x, double y)
{
	if (pick.drag)
	  if (pick.ornament)
	    draw_move_box();
	  else
	    draw_drag_box();

	pick.lastX = x;
	pick.lastY = y;

	if (pick.drag)
	  if (pick.ornament)
	    move_ornament();
	  else
	    {
	      Rectangle r = Rectangle(pick.downX, pick.downY,
				      pick.lastX, pick.lastY);
	      SRegion reg = thicken(r);
	      apply_mode_to_region(this, pick.extend, reg);

	      Notifier &n = session().notifier();
	      n.send_notice(nt_drag_sregion, (void *) &reg);
	    }
	else
	  buttonup_action();

	pick.down = false;
	pick.extend = false;
	pick.drag = false;

	draw_crosshair(X, x);
	draw_crosshair(Y, y);
}

// ----------------------------------------------------------------------------
//
void View::buttonup_action()
{
	Ornament	*nearest;
	int mode = pointer_mode(session());
	View *view = this;

	if (!pick.down)
	  return;			// Got button up but not button down.

	/*
	 * Strip centering does not depend on the mode. If there is
	 * a peak under the current point, though, use it to center
	 * the view(s).
	 */
	if (mode == MODE_CENTER) {
		Ornament	*pick = this->pick.ornament;
		SPoint p = point(this->pick.downX, this->pick.downY);


		if (pick && is_crosspeak(pick))
		  p = ((CrossPeak *)pick)->position();
		else if (pick && pick->type() == label)
		  {
		    CrossPeak *xp = ((Label *)pick)->attached();
		    if (xp)
		      p = xp->position();
		  }

		set_center(p);

		return;
	}

	/*
	 * Snap grids and lines to peaks/crosspeaks.
	 */
	nearest = NULL;
	if (mode == MODE_GRIDBOTH ||
	    mode == MODE_GRIDHOR ||
	    mode == MODE_GRIDVERT)
		nearest = ornament_snap_nearest(view, pick.downX, pick.downY);
	else if (mode == MODE_LINE)
		nearest = ornament_snap_nearest(view, pick.downX, pick.downY);

	SPoint r = point(pick.downX, pick.downY);

	if (nearest) {
		if (is_crosspeak(nearest))
			r = (is_peaklet(nearest) ?
			     ((Peak *)nearest)->peakgp()->position() :
			     ((CrossPeak *)nearest)->position());
		pick.ornament = NULL;
	}

	Project &proj = session().project();
	if (mode == MODE_SELECT) {
		select_ornament(pick.ornament, pick.extend, proj);
	}

	else if (mode == MODE_GRIDHOR) {
		select_ornament(pick.ornament, pick.extend, proj);
		Grid *g = new Grid(spectrum(), r, axis(X));
		g->select(true);
	}

	else if (mode == MODE_GRIDVERT) {
		select_ornament(pick.ornament, pick.extend, proj);
		Grid *g = new Grid(spectrum(), r, axis(Y));
		g->select(true);
	}

	else if (mode == MODE_GRIDBOTH) {
		select_ornament(pick.ornament, pick.extend, proj);
		Grid *gv = new Grid(spectrum(), r, axis(X));
		gv->select(true);
		Grid *gh = new Grid(spectrum(), r, axis(Y));
		gh->select(true);
	}

	else if (mode == MODE_LABEL) {
		select_ornament(pick.ornament, pick.extend, proj);
		if (pick.ornament)
		  show_label_dialog(session(), pick.ornament);
		else
		  show_label_dialog(session(), spectrum(),
				    point(pick.downX, pick.downY));
	}

	else if (mode == MODE_LINE) {
		mPts = 1 + mPts % 2;

		if (mPts == 1)
			line_start = r;
		else if (mPts == 2) {	// Make a line
		  proj.unselect_all_ornaments();
		  Line *line = new Line(spectrum(), line_start, r);
		  line->select(true);
		}
	}

	else if (mode == MODE_COPYASSIGNMENT) {
		select_ornament(pick.ornament, pick.extend, proj);
		do_assignment_copy(view);
	}

	else if (mode == MODE_GUESSASSIGNMENT) {
		select_ornament(pick.ornament, pick.extend, proj);
		Spectrum *sp = spectrum();
		const Guess_Assignments &ga = sp->GetAssignGuessOptions();
		List xplist = sp->selected_crosspeaks();
		List assigned = ga.apply_unique_guesses(xplist);
		label_peaks(assigned);
	}

	else if (mode == MODE_PEAK) {
		select_ornament(pick.ornament, pick.extend, proj);
		if (pick.ornament == NULL)
		  {
		    Peak *pk = new Peak(spectrum(), r);
		    select_ornament(pk, pick.extend, proj);
		  }
	}
}

// ----------------------------------------------------------------------------
//

#define DRAGMIN 2	// Screen pixels.

void View::drag(double x, double y)
{
	if (!pick.down ||		// We didn't get button down event.
	    (fabs(x - pick.downX) < DRAGMIN * pixel_size(X) &&
	     fabs(y - pick.downY) < DRAGMIN * pixel_size(Y)))
	  return;

	if (pick.drag)
	  if (pick.ornament)
	    draw_move_box();			// Erase box.
	  else
	    draw_drag_box();			// Erase box.

	pick.lastX = x;
	pick.lastY = y;
	pick.drag = true;

	if (pick.ornament)
	  draw_move_box();			// Draw box.
	else
	  draw_drag_box();			// Draw box.
}

// ----------------------------------------------------------------------------
//
void View::enter_window(double x, double y)
{
  if (!pick.down)		// Get EnterNotify when drawing a drag box
    draw_all_crosshairs(x, y);

  View_Window::enter_window(x, y);
}

// ----------------------------------------------------------------------------
//
void View::exit_window(double x, double y)
{
  erase_all_crosshairs();

  View_Window::exit_window(x, y);
}

// ----------------------------------------------------------------------------
//
void View::move(double x, double y)
{
  draw_all_crosshairs(x, y);
  View_Window::move(x, y);
}

// ----------------------------------------------------------------------------
//
void View::draw_all_crosshairs(double x, double y)
{
  List views = session().project().view_list();
  for (int v = 0 ; v < views.size() ; ++v)
    {
      View *view = (View *) views[v];

      bool show = (((view != this &&
		     view->config.show_crosshair_from_other_views) ||
		    (view == this && view->config.show_crosshair)) &&
		   view->shown());
	
      if (show)
	{
	  SPoint p = point(x, y);
	  double to_x, to_y;
	  if (map_crosshair(this, p, view, X, &to_x))
	    view->draw_crosshair(X, to_x);
	  if (map_crosshair(this, p, view, Y, &to_y))
	    view->draw_crosshair(Y, to_y);
	}
    }
}

// ----------------------------------------------------------------------------
//
void View::erase_all_crosshairs()
{
  const List &views = session().project().view_list();
  for (int v = 0 ; v < views.size() ; ++v)
    {
      View *view = (View *) views[v];
      view->erase_crosshair(X);
      view->erase_crosshair(Y);
    }
}

// ----------------------------------------------------------------------------
//
void View::draw_crosshair(Axis a, double p)
{
  erase_crosshair(a);

  if (config.show_crosshair || config.show_crosshair_from_other_views)
    {
      crosshair_drawn[a] = toggle_line(p, a);
      crosshair_xy[a] = p;
    }

  if (config.show_transpose_crosshair && is_homonuclear())
    {
      
      transpose_crosshair_drawn[a] = toggle_line(p, other_axis(a));
      crosshair_xy[a] = p;
    }
}

// ----------------------------------------------------------------------------
//
void View::erase_crosshair(Axis a)
{
  if (crosshair_drawn[a])
    {
      toggle_line(crosshair_xy[a], a);
      crosshair_drawn[a] = false;
    }

  if (transpose_crosshair_drawn[a])
    {
      toggle_line(crosshair_xy[a], other_axis(a));
      transpose_crosshair_drawn[a] = false;
    }
}

// ----------------------------------------------------------------------------
//
bool View::toggle_line(double p, Axis a)
{
  Rectangle vr = view_rectangle();
  if (p >= vr.min(a) && p <= vr.max(a))
    {
      set_drawing_color("white");
      if (a == X)
	draw_line(p, vr.min(Y), p, vr.max(Y), true);
      else
	draw_line(vr.min(X), p, vr.max(X), p, true);
      return true;
    }
  return false;
}

// ----------------------------------------------------------------------------
//
void View::blank_crosshairs(bool *blanked_x, bool *blanked_y)
{
  *blanked_x = (crosshair_drawn[X] || transpose_crosshair_drawn[X]);
  *blanked_y = (crosshair_drawn[Y] || transpose_crosshair_drawn[Y]);
  erase_crosshair(X);
  erase_crosshair(Y);
}

// ----------------------------------------------------------------------------
//
void View::unblank_crosshairs(bool blanked_x, bool blanked_y)
{
  if (blanked_x)
    draw_crosshair(X, crosshair_xy[X]);
  if (blanked_y)
    draw_crosshair(Y, crosshair_xy[Y]);
}

// ----------------------------------------------------------------------------
//
bool View::contains_crosshair(const Rectangle &r)
{
  return ((crosshair_drawn[X] &&
	   is_between(crosshair_xy[X], r.min(X), r.max(X))) ||
	  (crosshair_drawn[Y] &&
	   is_between(crosshair_xy[Y], r.min(Y), r.max(Y))) ||
	  (transpose_crosshair_drawn[X] &&
	   is_between(crosshair_xy[X], r.min(Y), r.max(Y))) ||
	  (transpose_crosshair_drawn[Y] &&
	   is_between(crosshair_xy[Y], r.min(X), r.max(X))));
}

// ----------------------------------------------------------------------------
//
void View::set_crosshair_position(const SPoint &pos)
{
  if (config.show_crosshair)
    {
      draw_crosshair(X, pos[axis(X)]);
      draw_crosshair(Y, pos[axis(Y)]);
    }
}

// ----------------------------------------------------------------------------
//
void View::key_pressed(char c)
{
  session().command_dispatcher().parse_key(c);
}

// ----------------------------------------------------------------------------
//
void View::function_key_pressed(int f, bool shifted)
{
  session().command_dispatcher().parse_function_key(f, shifted);
}

// ----------------------------------------------------------------------------
// Animate moving of an ornament.
//
void View::draw_move_box()
{
  ODraw &dr = ornament_drawing_routines();
  Rectangle r = flatten(pick.ornament->selection_region(dr));

  double dx = pick.lastX - pick.downX;
  double dy = pick.lastY - pick.downY;
  r.translate(dx, dy);
  set_drawing_color("white");
  draw_rectangle(r, true);
}

// ----------------------------------------------------------------------------
// Draw a region selection box.
//
void View::draw_drag_box()
{
  Rectangle r = Rectangle(pick.downX, pick.downY, pick.lastX, pick.lastY);
  set_drawing_color("white");
  draw_rectangle(r, true);
}

// ----------------------------------------------------------------------------
//
void View::redraw()
{
  crosshair_drawn[X] = false;
  crosshair_drawn[Y] = false;
  transpose_crosshair_drawn[X] = false;
  transpose_crosshair_drawn[Y] = false;
  erase_rectangle(view_rectangle());
  erase_resonances();
  set_axis_units(axis_units());	// Redisplay axis scales.
}

/*
 * Delete a view.
*/
View::~View()
{
  Notifier &n = session().notifier();
  n.send_notice(nt_will_delete_view, this);

  session().dialog_table().delete_dialog("current_view", this);

  n.dont_notify_me(nt_shifted_spectrum, ppm_scale_shifted_cb, this);
  n.dont_notify_me(nt_created_ornament, ornament_erase_cb, this);
  n.dont_notify_me(nt_will_change_ornament, ornament_erase_cb, this);
  n.dont_notify_me(nt_changed_ornament, ornament_erase_cb, this);
  n.dont_notify_me(nt_will_delete_ornament, will_delete_ornament_cb, this);
  n.dont_notify_me(nt_resized_spectrum_labels, labels_resized_cb, this);
  n.dont_notify_me(nt_changed_resonance, resonance_update_cb, this);
  n.dont_notify_me(nt_added_resonance_to_condition, resonance_update_cb, this);
  n.dont_notify_me(nt_removed_resonance_from_condition,
		 resonance_update_cb, this);

  //
  // Hide before destroying. This is to fix a bug where new
  // views aren't able to receive keyboard or pointer input
  // because button is down on view window when destroyed.
  //
  show_view(false);

  free_sregion_list(zoom_prev);
  free_sregion_list(zoom_nxt);

  delete odroutines;
  odroutines = 0;

  if (xscale) { delete xscale; xscale = 0; }
  if (yscale) { delete yscale; yscale = 0; }
  if (xrpanel) { delete xrpanel; xrpanel = 0; }
  if (yrpanel) { delete yrpanel; yrpanel = 0; }
  if (xslice) { delete xslice; xslice = 0; }
  if (yslice) { delete yslice; yslice = 0; }
  if (pkpanel) { delete pkpanel; pkpanel = 0; }
  if (cscale) { delete cscale; cscale = 0; }

  n.send_notice(nt_deleted_view_of_spectrum, spectrum());
}

// ----------------------------------------------------------------------------
//
void View::ppm_scale_shifted_cb(void *v, void *s)
{
  View *view = (View *) v;
  Spectrum *sp = (Spectrum *)s;

  bool need_update = (sp == view->spectrum());

  //
  // Also need to update if overlaid view on this view has shifted.
  //
  List vlist = view->session().project().overlaid_views(view);
  for (int vi = 0 ; !need_update && vi < vlist.size() ; ++vi)
    if (((View *)vlist[vi])->spectrum() == sp)
      need_update = true;

  if (need_update)
    {
      Spectrum *vsp = view->spectrum();
      view->set_repaint_region(vsp->ppm_region());
      view->set_repaint_tile(standard_redraw_tile(vsp));
      view->update_scales();
      // ppm view region fixed and ppm scale shifted so redisplay contours.
      view->Drawing::erase();
    }
}

// ----------------------------------------------------------------------------
// If the user attempts to delete a view using the window manager
// and this is the last view of a spectrum ask the user for confirmation.
//
bool View::ok_to_destroy()
{
  WinSys &ws = session().window_system();
  if (ws.modal_dialog_shown())
    return false;

  bool last_view = (session().project().view_list(spectrum()).size() == 1);
  bool ok = (!last_view || ok_to_unload_spectrum(session(), spectrum()));

  return ok;
}

// ----------------------------------------------------------------------------
//
Axis View::axis_name(int a) const
{
  if (a == axis(X))		return X;
  else if (a == axis(Y))	return Y;

  return NOT_XY;
}

// ----------------------------------------------------------------------------
//
Units View::axis_units()
 { return config.scale_units; }

// ----------------------------------------------------------------------------
//
void View::set_axis_units(Units u)
{
  config.scale_units = u;

  Spectrum *sp = spectrum();
  SPoint lo_scale = sp->map(sp->ppm_region().min, PPM, u);
  SPoint hi_scale = sp->map(sp->ppm_region().max, PPM, u);
  Stringy axis_labels[DIM];
  for (int a = 0 ; a < dimension() ; ++a)
    axis_labels[a] = axis_label(a);
  View_Window::set_axes(axis_labels, lo_scale, hi_scale);
  update_scales();
}

// ----------------------------------------------------------------------------
//
void View::update_scales()
{
  if (xscale) xscale->update();
  if (yscale) yscale->update();
}

// ----------------------------------------------------------------------------
//
void View::data_point(const SPoint &p, double *x, double *y)
{
  *x = p[axis(X)];
  *y = p[axis(Y)];
}

// ----------------------------------------------------------------------------
//
SPoint View::point(double x, double y)
{
  SPoint p = view_region().center();

  p[axis(X)] = x;
  p[axis(Y)] = y;

  return p;
}

// ----------------------------------------------------------------------------
// Scale from one set of units to another.
//
double	View::scale(double x, Axis a, Units from, Units to) const
  { return spectrum()->scale(x, axis(a), from, to); }

// ----------------------------------------------------------------------------
// Map from one set of units to another.
//
double	View::map(double x, Axis a, Units from, Units to) const
  { return spectrum()->map(x, axis(a), from, to); }

// ----------------------------------------------------------------------------
//
bool View::is_homonuclear()
{
  return spectrum()->is_homonuclear(axis(X), axis(Y));
}

/*
 * Set the view footer (shown along the bottom edge of the view) to <text>
 */
void View::set_footer(const char *)
{
}

/*
 * Return <view>'s contour thresholds -- the lowest contour level for positive
 * and negative contours.
 */
void View::lowest_contour_levels(double *pos, double *neg)
{
  const Contour_Levels &p = config.pos_contours.levels;
  const Contour_Levels &n = config.neg_contours.levels;

  *pos = (p.levels ? p.lowest : FLT_MAX);
  *neg = (n.levels ? n.lowest : -FLT_MAX);
}

// ----------------------------------------------------------------------------
// Zoom view <view> to show region r.
//
void View::zoom(Rectangle r)
{
  zoom(thicken(r));
}

// ----------------------------------------------------------------------------
//
void View::zoom(const SRegion &r, bool remember)
{
  if (scale(r.size(axis(X)), X, PPM, INDEX) < 1 ||
      scale(r.size(axis(Y)), Y, PPM, INDEX) < 1)
    return;

  if (remember)
    zoom_stack_push();

  SPoint c = r.center();
  SPoint ps = pixel_size();
  SRegion cur = view_region();
  double factor = max(r.size(axis(X)) / cur.size(axis(X)),
		      r.size(axis(Y)) / cur.size(axis(Y)));
  ps *= factor;

  set_view(c, ps);
}

// ----------------------------------------------------------------------------
//
void View::zoom_previous()
{
  if (zoom_prev.empty())
    return;

  zoom_nxt.append(new SRegion(view_region()));
  trim_zoom_stack(zoom_nxt);

  SRegion *vr = (SRegion *) zoom_prev.pop();
  zoom(*vr, false);
}

// ----------------------------------------------------------------------------
//
void View::zoom_next()
{
  if (zoom_nxt.empty())
    return;

  zoom_prev.append(new SRegion(view_region()));
  trim_zoom_stack(zoom_prev);

  SRegion *vr = (SRegion *) zoom_nxt.pop();
  zoom(*vr, false);
}

// ----------------------------------------------------------------------------
//
void View::zoom_stack_push()
{
  zoom_prev.append(new SRegion(view_region()));
  trim_zoom_stack(zoom_prev);
}

// ----------------------------------------------------------------------------
//
static void trim_zoom_stack(List &zs)
{
  int max_size = 16;
  if (zs.size() > max_size)
    {
      SRegion *r = (SRegion *) zs[0];
      zs.erase(0);
      delete r;
    }
}

// ----------------------------------------------------------------------------
//
static void free_sregion_list(List &srlist)
{
  for (int ri = 0 ; ri < srlist.size() ; ++ri)
    delete (SRegion *) srlist[ri];
  srlist.erase();
}

// ----------------------------------------------------------------------------
//
void View::synchronize_other_views()
{
  for (int a = 0 ; a < spectrum()->dimension() ; ++a)
    {
      view_axis from = view_axis(this, a);
      const List *to_axes = session().project().synchronized_view_axes(from);
      position_axes(from, to_axes);
    }
}

// ----------------------------------------------------------------------------
//
static void position_axes(const view_axis &from, const List *to_axes)
{
  if (to_axes)
    for (int ai = 0 ; ai < to_axes->size() ; ++ai)
      {
	view_axis *to = (view_axis *) (*to_axes)[ai];
	if (!(*to == from))
	  position_axis(to->view(), to->axis(), from.view(), from.axis());
      }
}

// ----------------------------------------------------------------------------
//
static void position_axis(View *view, int axis, View *from, int from_axis)
{
  SPoint c = view->center();
  c[axis] = from->center()[from_axis];

  view->set_view(c, view->pixel_size(), false);
}

// ----------------------------------------------------------------------------
//
const View_Settings &View::settings()
{
  config.axis_order = axis_order();
  screen_location(&config.x, &config.y);
  screen_size(&config.width, &config.height);
  config.center = center();
  config.pixel_size = pixel_size();
  config.visible_depth = visible_depth();
  config.scale_units = axis_units();
  config.show_view = shown();
  config.show_scales = (xscale && edge_panel_shown(xscale));
  config.show_scrollbars = scrollbars_shown();
  config.show_resonance_panels = (xrpanel && edge_panel_shown(xrpanel));
  if (xrpanel) config.filter_resonances = xrpanel->filter_resonances();
  config.show_slices = (xslice && edge_panel_shown(xslice));
  if (xslice) config.slice_auto_scale = xslice->auto_scale();
  if (xslice) config.slice_subtract_peaks = xslice->subtract_fit_peaks();
  config.show_peak_info = (pkpanel && edge_panel_shown(pkpanel));
  config.show_contour_scale = (cscale && edge_panel_shown(cscale));

  return config;
}

// ----------------------------------------------------------------------------
// Doesn't set screen position, width, height, view name, or contour levels.
//
void View::configure(const View_Settings &s)
{
  show_ornaments(s.show_ornaments);
  show_ornaments(peak, s.show_peaks);
  show_ornaments(peakgroup, s.show_peakgroups);
  show_ornaments(label, s.show_labels);
  show_ornaments(line, s.show_lines);
  show_ornaments(grid, s.show_grids);

  configure_edge_panels(s, session().project().preferences.resize_views);

  erase_crosshair(X);
  erase_crosshair(Y);
  config.show_crosshair = s.show_crosshair;
  config.show_transpose_crosshair = s.show_transpose_crosshair;
  config.show_crosshair_from_other_views = s.show_crosshair_from_other_views;

  bool change_subtract = (config.subtract_fit_peaks != s.subtract_fit_peaks);
  config.subtract_fit_peaks = s.subtract_fit_peaks;
  if (change_subtract)
    erase_rectangle(view_rectangle());
}

// ----------------------------------------------------------------------------
//
void View::configure_edge_panels(const View_Settings &s, bool resize_window)
{
  List ipanels, opanels;

  if (s.show_scales)
    {
      ipanels.append(axis_scale(X));
      ipanels.append(axis_scale(Y));
    }

  config.show_nucleus_names = s.show_nucleus_names;
  set_axis_units(s.scale_units);

  if (s.show_resonance_panels)
    {
      ipanels.append(resonance_panel(X));
      ipanels.append(resonance_panel(Y));
    }

  config.filter_resonances = s.filter_resonances;
  if (xrpanel) xrpanel->set_resonance_filtering(s.filter_resonances);
  if (yrpanel) yrpanel->set_resonance_filtering(s.filter_resonances);

  if (s.show_slices)
    {
      ipanels.append(slice(X));
      ipanels.append(slice(Y));
    }

  config.slice_auto_scale = s.slice_auto_scale;
  if (xslice) xslice->set_auto_scale(s.slice_auto_scale);
  if (yslice) yslice->set_auto_scale(s.slice_auto_scale);

  config.slice_subtract_peaks = s.slice_subtract_peaks;
  if (xslice) xslice->show_peaks_subtracted(s.slice_subtract_peaks);
  if (yslice) yslice->show_peaks_subtracted(s.slice_subtract_peaks);

  if (s.show_peak_info)
    opanels.append(peak_panel());

  if (s.show_contour_scale)
    opanels.append(contour_scale());

  show_edge_panels(ipanels, opanels, s.show_scrollbars, resize_window);
}

/*
 * Create a duplicate view the same screen scale.
 */
#define VIEW_DUP_OFFSET 50

View *View::duplicate(Rectangle r)
{
  View_Settings s = settings();
  s.view_name = spectrum()->name();
  s.x += VIEW_DUP_OFFSET;
  s.y += VIEW_DUP_OFFSET;

  View *newview = new ::View(session(), NULL, spectrum(), s);
  newview->zoom(r);

  return newview;
}

/*
 * Return the view's contour parameters.
 */
const Contour_Parameters &View::positive_contours()
  { return config.pos_contours; }
const Contour_Parameters &View::negative_contours()
  { return config.neg_contours; }
const Contour_Parameters &View::contour_parameters(double level)
  { return (level >= 0 ? positive_contours() : negative_contours()); }

/*
 * Set the view's contour parameters.
 */
void View::set_contour_parameters(const Contour_Parameters &cp)
{
  Contour_Parameters c = cp;
  if (c.levels.levels < 0)
    c.levels.levels = 0;
  if (c.levels.levels > 100)
    c.levels.levels = 100;

  if (c.levels.lowest >= 0)
    config.pos_contours = c;
  else
    config.neg_contours = c;

  /*
   * Redraw the view and views which overlay it.
   */
  Drawing::erase();

  Project &proj = session().project();
  List vlist = proj.view_list();
  for (int vi = 0 ; vi < vlist.size() ; ++vi)
    {
      View *onto = (View *) vlist[vi];
      if (proj.overlaid_views(onto).contains(this))
	onto->Drawing::erase();
    }

  Notifier &n = session().notifier();
  n.send_notice(nt_changed_view_contours, this);
}

// ----------------------------------------------------------------------------
//
IRegion View::two_dimensional_region(int x, int y, int w, int h)
{
  Rectangle rect(x, y, x+w-1, y+h-1);

  return flat_region(rect).rounded();
}

// ----------------------------------------------------------------------------
//
bool View::pointer_position(SPoint *p)
{
  double x, y;
  if (Drawing::pointer_position(&x, &y))
    {
      *p = point(x, y);
      return true;
    }

  return false;
}

// ----------------------------------------------------------------------------
//
bool View::ornaments_shown(Ornament_Type type)
  { return config.show_ornament(type); }

// ----------------------------------------------------------------------------
//
void View::show_ornaments(Ornament_Type type, bool show)
{
  bool shown = config.show_ornament(type);
  if (show != shown)
    {
      config.show_ornament(type, show);
      if (config.show_ornaments)
	erase_ornaments(type);
    }
}

// ----------------------------------------------------------------------------
//
void View::erase_ornaments(Ornament_Type type)
{
  const List &olist = spectrum()->ornaments();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (op->type() == type)
	erase(op);
    }
}

// ----------------------------------------------------------------------------
//
void View::show_ornaments(bool show)
{
  if (show != config.show_ornaments)
    {
      config.show_ornaments = show;

      for (int type = 0; type < (int) ORNAMENT_TYPE_COUNT; type++)
	erase_ornaments((Ornament_Type) type);
    }
}

// ----------------------------------------------------------------------------
//
double View::ornament_size(Ornament_Type type, int a) const
  { return spectrum()->ornament_size(type) * pixel_size(a) / pixel_size(0); }

/*
 * Return the name of axis label
 */
Stringy View::axis_label(int x)
{
	Stringy axislabel = formatted_string("w%d", x+1);
	if (config.show_nucleus_names)
	  {
	    Stringy nucleus = spectrum()->nucleus_type(x);
	    return nucleus + " " + axislabel;
	  }

	/*
	 * Otherwise show the "wX" name
	 */
	return axislabel;
}

// ----------------------------------------------------------------------------
//
void View::show_view(bool show)
{
  View_Window::show_view(show);
  set_redraw_priority();
}

// ----------------------------------------------------------------------------
//
bool View::subtract_fit_peaks() const
  { return config.subtract_fit_peaks; }

/*
 * Center View
 */
void View::set_center(const SPoint &c)
{
  zoom_stack_push();
  set_view(c, pixel_size());
}

// ----------------------------------------------------------------------------
//
void View::erase_resonances()
{
  if (xrpanel) xrpanel->resonance_changed();
  if (yrpanel) yrpanel->resonance_changed();
}

// ----------------------------------------------------------------------------
//
Scale *View::axis_scale(Axis xy)
{
  Scale *s = (xy == X ? xscale : yscale);
  if (s == NULL)
    {
      int axis = this->axis(xy);
      SRegion view = view_region();
      double a = view.min[axis];
      double b = view.max[axis];
      int length = (xy == X ? config.width : config.height);
      bool horz = (xy == X);
      s = new Scale(this, a, b, length, horz);
      (xy == X ? xscale : yscale) = s;
    }
  return s;
}

// ----------------------------------------------------------------------------
//
RPanel *View::resonance_panel(Axis xy)
{
  RPanel *rp = (xy == X ? xrpanel : yrpanel);
  if (rp == NULL)
    {
      int axis = this->axis(xy);
      SRegion view = view_region();
      double a = view.min[axis];
      double b = view.max[axis];
      int length = (xy == X ? config.width : config.height);
      bool horz = (xy == X);
      rp = new RPanel(this, a, b, length, horz);
      rp->set_resonance_filtering(config.filter_resonances);
      (xy == X ? xrpanel : yrpanel) = rp;
    }
  return rp;
}

// ----------------------------------------------------------------------------
//
Slice *View::slice(Axis xy)
{
  Slice *s = (xy == X ? xslice : yslice);
  if (s == NULL)
    {
      int axis = this->axis(xy);
      SRegion view = view_region();
      double a = view.min[axis];
      double b = view.max[axis];
      int length = (xy == X ? config.width : config.height);
      bool horz = (xy == X);
      s = new Slice(this, a, b, length, horz);
      s->set_auto_scale(config.slice_auto_scale);
      s->show_peaks_subtracted(config.slice_subtract_peaks);
      (xy == X ? xslice : yslice) = s;
    }
  return s;
}

// ----------------------------------------------------------------------------
//
Peak_Panel *View::peak_panel()
{
  if (pkpanel == NULL)
    {
      int xaxis = axis(X);
      SRegion view = view_region();
      double a = view.min[xaxis];
      double b = view.max[xaxis];
      pkpanel = new Peak_Panel(this, a, b, config.width);
    }
  return pkpanel;
}

// ----------------------------------------------------------------------------
//
ContourScale *View::contour_scale()
{
  if (cscale == NULL)
    {
      int yaxis = axis(Y);
      SRegion view = view_region();
      double a = view.min[yaxis];
      double b = view.max[yaxis];
      cscale = new ContourScale(this, a, b, config.height);
    }
  return cscale;
}

// ----------------------------------------------------------------------------
//
void View::shift_slices(double shift)
{
  if (xslice && yslice)
    {
      double hmin, hmax;
      xslice->height(&hmin, &hmax);
      xslice->set_height(hmin + shift, hmax + shift);
      yslice->set_height(hmin + shift, hmax + shift);
    }
}

// ----------------------------------------------------------------------------
//
void View::scale_slices(double factor)
{
  if (xslice && yslice)
    {
      double hmin, hmax;
      xslice->height(&hmin, &hmax);
      xslice->set_height(factor * hmin, factor * hmax);
      yslice->set_height(factor * hmin, factor * hmax);
    }
}

// ----------------------------------------------------------------------------
//
void View::set_name(const Stringy &n)
{
  if (n == name())
    return;

  this->config.view_name = n;
  set_title(fullname(), name());
  Notifier &nt = session().notifier();
  nt.send_notice(nt_renamed_view, this);
}

// ----------------------------------------------------------------------------
//
void View::got_focus()
{
  View *last_view = selected_view(session());
  if (this == last_view)
    return;

  session().dialog_table().set_dialog("current_view", this);

  if (last_view)
    {
      last_view->set_title(last_view->fullname(), last_view->name());
      last_view->set_redraw_priority();
    }

  Stringy title = fullname() + Stringy(" -- selected");
  set_title(title, name());
  set_redraw_priority();
  Notifier &n = session().notifier();
  n.send_notice(nt_selected_view, this);
}

/*
 * Return the current view or NULL if there is no current view or if the
 * current view is being destroyed right now.
 */
View *selected_view(Session &s)
{
  return (View *) s.dialog_table().get_dialog("current_view");
}

//-----------------------------------------------------------------------------
//
Spectrum *selected_spectrum(Session &s)
{
  View *v = selected_view(s);

  return (v ? v->spectrum() : NULL);
}

// ----------------------------------------------------------------------------
//
void View::set_redraw_priority()
{
  double priority = VIEW_REPAINT_PRIORITY;
  if (!shown())
    priority -= 10;
  else if (this == selected_view(session()))
    priority += 10;
  Drawing::set_redraw_priority(priority);
}

// ----------------------------------------------------------------------------
//
void Drawing_Routines::set_contour_level(float) {}
void Drawing_Routines::draw_contour_path(float *, int) {}

// ----------------------------------------------------------------------------
//
void update_resonance_panels(const List &views)
{
  for (int vi = 0 ; vi < views.size() ; ++vi)
    ((View *) views[vi])->erase_resonances();
}

// ----------------------------------------------------------------------------
// Return +1, 0, -1 for peak above, in, or below view plane in 3-D spectrum.
// If spectrum is not 3-D return 0.
//
static int in_plane(const SPoint &p, View *v)
{
  int plane = 0;
  Spectrum *sp = v->spectrum();

  if (sp->dimension() == 3)
    {
      int zaxis = v->axis_order()[2];
      double z = sp->map(p[zaxis], zaxis, PPM, INDEX);
      double z_display = nearest_int(sp->map(v->center(zaxis), zaxis, PPM, INDEX));
      if (z - z_display > .5)
	plane = 1;
      else if (z - z_display < -.5)
	plane = -1;
    }

  return plane;
}

// ----------------------------------------------------------------------------
// Erase an ornament and generate exposure events.
//
void View::erase(Ornament *op)
{
  ODraw &dr = ornament_drawing_routines();
  SRegion region = op->erasure_region(dr);
  if (region.intersect(dr.clip_region()))
    erase_rectangle(flatten(region));
  local_ornaments.erase(op);
}

// ----------------------------------------------------------------------------
// Compute distance to ornament for selecting ornaments with mouse.
//
static double distance(Ornament *op, View *view, double x, double y)
{
  SPoint c = op->selection_region(view->ornament_drawing_routines()).center();
  double dx, dy;

  dx = x - c[view->axis(X)];
  dy = y - c[view->axis(Y)];

  return sqrt(dx * dx + dy * dy);
}

// ----------------------------------------------------------------------------
// Is any part of ornament visible in the given region?
// This is used for redrawing ornaments.
//
static bool is_visible(const Ornament *op, View *view, Rectangle r)
{
  SRegion region = op->erasure_region(view->ornament_drawing_routines());
  return region.intersects(view->thicken(r));
}

// ----------------------------------------------------------------------------
// Location in data units.
//
void xy_location(CrossPeak *xp, View *view, double *x, double *y)
{
  view->data_point(xp->position(), x, y);
}

/*
 * Do copy assignment mode on the selected peaks.
 */
static void do_assignment_copy(View *view)
{
  Spectrum *from = view->session().project().assignment_copy_from_spectrum();
  Reporter &rr = view->session().reporter();
  if (from)
    assignment_copy(from, view->spectrum()->selected_crosspeaks(), true, rr);
  else
    rr.warning("You need to select a spectrum to copy assignments from.\n"
	       "Use the Assignment Copy Options dialog to do this.\n");
}

/*
 * Do the current mode to region.
 */
void apply_mode_to_region(View *view, bool extend, const SRegion &region)
{
	int mode = pointer_mode(view->session());
	Rectangle r = view->flatten(region);

	if (mode == MODE_ZOOM)
		view->zoom(r);
	else if (mode == MODE_DUPZOOM)
		view->duplicate(r);
	else if (mode == MODE_INTEGRATE) {
		select_crosspeaks(view, extend, region);
		integrate_region(view, region);
	}
	else if (mode == MODE_COPYASSIGNMENT) {
		select_crosspeaks(view, extend, region);
		do_assignment_copy(view);
	}
	else if (mode == MODE_GUESSASSIGNMENT) {
		select_crosspeaks(view, extend, region);
		Spectrum *sp = view->spectrum();
		const Guess_Assignments &ga = sp->GetAssignGuessOptions();
		List xplist = sp->selected_crosspeaks();
		List assigned = ga.apply_unique_guesses(xplist);
		label_peaks(assigned);
	}
	else if (mode == MODE_PEAK) {
		Peak_Pick_Parameters params = view->spectrum()->mPeakPick;
		view->lowest_contour_levels(&params.minPosHt,
					    &params.minNegHt);
		List peaks = peak_pick_region(view->spectrum(), region, params,
					      wait_callbacks(view->session()));
		if (!extend)
		  view->session().project().unselect_all_ornaments();
		for (int pi = 0 ; pi < peaks.size() ; ++pi)
		  ((Peak *) peaks[pi])->select(true);
		Stringy msg;
		if (peaks.size() == 0)
		  msg = "No new peaks\n";
		else
		  msg = formatted_string("%d new peak%s\n",
					 peaks.size(),
					 (peaks.size() == 1 ? "" : "s"));
		Reporter &rr = view->session().reporter();
		rr.message(msg.cstring());
	}

	/*
	 * Other modes are like point mode.
	 */
	else {
	  select_ornaments(view, extend, region);
	}
}

/*
 *	Compute integrals for the region bounded by the bounding box.
 */
static bool integrate_region(View *view, const SRegion &region)
{
	Reporter &rr = view->session().reporter();
	List peaks = region_peaklets(view, region);
	if (!peaks.empty())
	  {
	    Spectrum *sp = view->spectrum();
	    double pos_thresh, neg_thresh;
	    view->lowest_contour_levels(&pos_thresh, &neg_thresh);
	    Session &s = view->session();
	    Project &proj = s.project();
	    proj.MakeUndoIntegrate(peaks);
	    integrate_list(sp, peaks, region, pos_thresh, neg_thresh,
			   wait_callbacks(s), rr);
	  }
	else 
	  rr.message("No peaks to integrate\n");

	return true;
}

// ----------------------------------------------------------------------------
//
static List region_peaklets(View *view, const SRegion &region)
{
  List rlist;

  ODraw &dr = view->ornament_drawing_routines();
  const List &olist = view->spectrum()->ornaments();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (op->type() == peak && op->selection_region(dr).intersects(region))
	rlist.append((Peak *) op);
    }

  return rlist;
}

/*
 * Select Ornament <op>.  If <extend> is true add to the current set
 * of selected ornaments.
 */
static void select_ornament(Ornament *op, bool extend, Project &proj)
{
	if (op == NULL)
	  {
	    if (!extend)
	      proj.unselect_all_ornaments();
	  }
	else if (extend)
		op->select(!op->IsSelected());
	else
	  {
	    proj.unselect_all_ornaments();
	    op->select(true);
	  }
}

/*
 * Select all crosspeaks in the region. If extend is false, unselect
 * every other thing and select those things in the region. If the extend
 * is true, toggle the selection of things in the region.
 */
static void select_crosspeaks(View *view, bool extend, const SRegion &region)
{
	if (!extend)
		view->session().project().unselect_all_ornaments();

	List xplist;
	ODraw &dr = view->ornament_drawing_routines();
	const List &list = view->spectrum()->crosspeaks();
	for (int pi = 0 ; pi < list.size() ; ++pi)
	  {
	    CrossPeak *xp = (CrossPeak *) list[pi];
	    if (xp->selection_region(dr).intersects(region)) {
	        bool state = (extend ? !xp->IsSelected() : true);
		xp->select(state);
		xplist.append(xp);
	    }
	}
}

/*
 * Select all "things" in the region. If extend is false, unselect
 * every other thing and select those things in the region.  If extend
 * is true, toggle the selection of things in the region.
 */
static void select_ornaments(View *view, bool extend, const SRegion &region)
{
	if (!extend)
		view->session().project().unselect_all_ornaments();

	List olist = view->region_ornament_list(false, region);
	for (int oi = 0 ; oi < olist.size() ; ++oi) {
		Ornament *op = (Ornament *) olist[oi];
		bool state = (extend ? !op->IsSelected() : true);
		op->select(state);
	}
}

/*
 * Return a list of ornaments in this region. If 'selected' is true, only
 * return a list of selected ornaments.
 */
List View::region_ornament_list(bool selected, const SRegion &region)
{
	List		list;

	if (config.show_ornaments)
	  {
	    ODraw &dr = ornament_drawing_routines();
	    const List &olist = spectrum()->ornaments();
	    for (int oi = 0 ; oi < olist.size() ; ++oi)
	      {
		Ornament *op = (Ornament *) olist[oi];
		if (ornaments_shown(op->type()))
		  if ((!selected || op->IsSelected())
		      && op->selection_region(dr).intersects(region))
		    list.append(op);
	      }
	  }

	return list;
}

/*
 * Return a pointer to the ornament of given type that is nearest to the
 * point x, y.  The snap distance must be less than SNAP_DISTANCE.
 */
static Ornament *snap_nearest(View *view, double x, double y, Ornament_Type type)
{
	Ornament	*nearest = NULL;
	double		dxppm, dyppm, dx, dy, cx, cy, d, dist = 0;

		/*
		 * Convert the snap_distance from screen into ppm.
		 */
		dxppm = SNAP_DISTANCE * view->pixel_size(X);
		dyppm = SNAP_DISTANCE * view->pixel_size(Y);
		Rectangle rect = Rectangle(x - dxppm, y - dyppm,
					   x + dxppm, y + dyppm);
		SRegion region = view->thicken(rect);

		List olist = view->region_ornament_list(false, region);
		for (int oi = 0 ; oi < olist.size() ; ++oi) {
			Ornament *op = (Ornament *) olist[oi];

			if (op->type() == type) // type is peak or peakgp
			  {
			    CrossPeak *xp = (CrossPeak *)op;

			    xy_location(xp, view, &cx, &cy);

			    dx = cx - x;
			    dy = cy - y;
			    d = dx * dx + dy * dy;

			    if (nearest == NULL || d < dist) {
			      dist = d;
			      nearest = op;
			    }
			  }
		    }

	return nearest;
}

// ----------------------------------------------------------------------------
// Snap to nearest peak group.  If none in range then snap to nearest peak.
//
static CrossPeak *ornament_snap_nearest(View *view, double x, double y)
{
  CrossPeak *nearest;

  nearest = (CrossPeak *) snap_nearest(view, x, y, peakgroup);
  if (nearest == NULL)
    nearest = (CrossPeak *) snap_nearest(view, x, y, peak);

  return nearest;
}


//
// Move an ornament from its previous position to a new position.
// 
// For unparented labels, we reparent them to the
// crosspeak we have dragged them to.
//

#define MINMOVE		5

void View::move_ornament()
{
	bool		cansnap;
	double		dxdata, dydata;
	Ornament	*op;
	CrossPeak	*nearest;

	op = pick.ornament;

	/*
	 * We can snap for unparented labels.
	 * Labels are parented to the ornament.
	 */
	cansnap = (op->type() == label && ((Label *)op)->attached() == NULL);

	/*
	 * Snap order is peak group then peak.
	 */
	if (cansnap)
		nearest = ornament_snap_nearest(this, pick.lastX, pick.lastY);
	/*
	 * If this is a label, snap it to the nearest crosspeak.
	 */
	if (cansnap && nearest &&
	    op->type() == label &&
	    nearest->label() == NULL) {
		((Label *) op)->attach(nearest);

		return;
	}

	/*
	 * If this is a grid place at the mouse pointer position.
	 */
	dxdata = pick.lastX - pick.downX;
	dydata = pick.lastY - pick.downY;

	if (fabs(dxdata) < MINMOVE * pixel_size(X) &&
	    fabs(dydata) < MINMOVE * pixel_size(Y))
		return;

	//
	// Prepare to undo the move, then move Move this ornament
	//
	Undo	*undo = session().project().MakeUndoMove();
	undo->add(op);
	::move_ornament(op, dxdata, dydata, this);
	undo->finish();
}

/*
 * Increment an ornament location, but only if it isn't locked.
 */
static void move_ornament(Ornament *op, double dx, double dy, View *view)
{
	if (op->IsLocked())
	  {
	    Reporter &rr = view->session().reporter();
	    rr.message("Ornament is locked, cannot be moved\n");
	  }
	else {
		if (op->type() == label)
		  ((Label *) op)->IncrementLocation(dx, dy,
						    view->axis(X),
						    view->axis(Y));
		else
		  {
		    SPoint  delta(view->spectrum()->dimension());

		    delta[view->axis(X)] = dx;
		    delta[view->axis(Y)] = dy;

		    bool drag_peak = (op->type() == peak);
		    Notifier &n = view->session().notifier();
		    if (drag_peak)
		      n.send_notice(nt_will_drag_peak, (Peak *)op);

		    op->IncrementLocation(delta);

		    if (drag_peak)
		      n.send_notice(nt_dragged_peak, (Peak *)op);
		  }

	}
}

// ----------------------------------------------------------------------------
//
Ornament_Routines::Ornament_Routines(View *view)
  { this->v = view; }
void Ornament_Routines::set_drawing_color(const Color &color)
  { v->set_drawing_color(color); }
void Ornament_Routines::draw_line(double x1, double y1, double x2, double y2)
  { v->draw_line(x1, y1, x2, y2); }
void Ornament_Routines::draw_text(double cx, double cy, const Stringy &text)
  { v->draw_text(cx, cy, text); }
void Ornament_Routines::text_size(const Stringy &text, double *width,
		      double *ascent, double *descent)
  { v->text_size(text, width, ascent, descent); }
void Ornament_Routines::draw_rectangle(Rectangle r)
  { v->draw_rectangle(r); }
void Ornament_Routines::draw_triangle(double x1, double y1,
			  double x2, double y2,
			  double x3, double y3)
  { v->draw_triangle(x1, y1, x2, y2, x3, y3); }
void Ornament_Routines::draw_ellipse(Rectangle r)
  { v->draw_ellipse(r); }
int Ornament_Routines::axis(Axis a)
  { return v->axis(a); }
double Ornament_Routines::aspect()
  { return v->aspect(); }
SRegion Ornament_Routines::clip_region()
  { return v->view_region(); }
double Ornament_Routines::ornament_size(Ornament_Type type, int axis)
  { return v->ornament_size(type, axis); }
double Ornament_Routines::select_size(int axis)
  { return (v->spectrum()->select_size()
	    * v->pixel_size(axis) / v->pixel_size(0)); }
void Ornament_Routines::xy_point(const SPoint &p, double *x, double *y)
  { v->data_point(p, x, y); }
bool Ornament_Routines::is_visible(const Ornament *op, Rectangle r)
  { return ::is_visible(op, v, r); }
int Ornament_Routines::in_plane(const SPoint &p)
  { return ::in_plane(p, v); }
bool Ornament_Routines::black_labels()
  { return v->session().project().print_options.black_labels; }
bool Ornament_Routines::subtract_fit_peaks()
  { return v->subtract_fit_peaks(); }

// ----------------------------------------------------------------------------
//
bool orthogonal_views(View *v1)
{
  int dim = v1->spectrum()->dimension();
  if (dim != 3)
    return false;

  CrossPeak *xp = v1->session().project().single_peak_selected();
  if (xp)
    v1->set_center(xp->position());

  View *v2 = v1->duplicate(v1->view_rectangle());
  v2->roll_axes();

  View *v3 = v1->duplicate(v1->view_rectangle());
  v3->roll_axes();
  v3->roll_axes();

  Project &proj = v1->session().project();
  for (int a = 0 ; a < dim ; ++a)
    {
      proj.synchronize_view_axes(view_axis(v1, a), view_axis(v2, a));
      proj.synchronize_view_axes(view_axis(v1, a), view_axis(v3, a));
    }

  return true;
}

// ----------------------------------------------------------------------------
//
static bool map_crosshair(View *from, const SPoint &p,
			  View *to, Axis toaxis, double *to_position)
{
  int from_axis;
  int to_axis = to->axis(toaxis);
  int from_x = from->axis(X);
  int from_y = from->axis(Y);
  Spectrum *from_sp = from->spectrum();
  Spectrum *to_sp = to->spectrum();
  Stringy to_nucleus = to_sp->nucleus_type(to_axis);
  bool nuc_match = (from_sp->dimension() > to_axis &&
		    from_sp->nucleus_type(to_axis) == to_nucleus);

  if (nuc_match && (to_axis == from_x || to_axis == from_y))
    from_axis = to_axis;
  else
    {
      bool xmatch = (from_sp->nucleus_type(from_x) == to_nucleus);
      bool ymatch = (from_sp->nucleus_type(from_y) == to_nucleus);
      if (xmatch && !ymatch)		
	from_axis = from_x;
      else if (!xmatch && ymatch)
	from_axis = from_y;
      else if (xmatch && ymatch)
	{
	  int from_a = from->axis(toaxis);
	  int other_to_axis = to->axis(other_axis(toaxis));
	  if (other_to_axis == from_a &&
	      to_sp->nucleus_type(other_to_axis)
	      == from_sp->nucleus_type(from_a))
	    from_axis = from->axis(other_axis(toaxis));
	  else
	    from_axis = from_a;
	}
      else if (nuc_match)
	from_axis = to_axis;
      else if (from_sp->is_nucleus_unique(to_nucleus, &from_axis))
	  ;
      else
	return false;
    }

  *to_position = p[from_axis];

  return true;
}

// ----------------------------------------------------------------------------
// Attempt to move visible labels around so that they do not overlap
// each other or peak markers.  The undo command should undo all moved labels.
//
void View::unoverlap_labels()
{
  List olist = visible_ornaments(view_rectangle());
  ODraw &dr = ornament_drawing_routines();
  List peakrect, labelrect, labels;
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (is_crosspeak(op))
	peakrect.append(new Rectangle(flatten(op->selection_region(dr))));
      else if (op->type() == label)
	{
	  labels.append((Label *) op);
	  labelrect.append(new Rectangle(flatten(op->selection_region(dr))));
	}
    }

  unoverlap_rectangles(labelrect, peakrect, .3, 20);

  Undo	*undo = session().project().MakeUndoMove();
  for (int li = 0 ; li < labels.size() ; ++li)
    {
      Label *label = (Label *) labels[li];
      undo->add(label);
      Rectangle current = flatten(label->selection_region(dr));
      Rectangle moveto = *(Rectangle *) labelrect[li];
      double dx = moveto.center(X) - current.center(X);
      double dy = moveto.center(Y) - current.center(Y);
      ::move_ornament(label, dx, dy, this);
    }
  undo->finish();

  free_rectangle_list_entries(labelrect);
  free_rectangle_list_entries(peakrect);
}

// ----------------------------------------------------------------------------
// Move visible peak labels to the right of attached peaks.
//
void View::right_offset_labels()
{
  List olist = visible_ornaments(view_rectangle());
  ODraw &dr = ornament_drawing_routines();
  int xaxis = axis(X);
  int yaxis = axis(Y);
  Undo	*undo = session().project().MakeUndoMove();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (op->type() == label)
	{
	  Label *label = (Label *) op;
	  CrossPeak *xp = label->attached();
	  if (xp)
	    {
	      SRegion rlabel = label->selection_region(dr);
	      SRegion rpeak = xp->selection_region(dr);
	      double offset = -2*rpeak.size(xaxis) - .5*rlabel.size(xaxis);
	      double dx = rpeak.center(xaxis) + offset - rlabel.center(xaxis);
	      double dy = rpeak.center(yaxis) - rlabel.center(yaxis);
	      undo->add(label);
	      ::move_ornament(label, dx, dy, this);
	    }
	}
    }
  undo->finish();
}

// ----------------------------------------------------------------------------
//
void shift_views(const List &vlist, const SPoint &delta)
{
  for (int vi = 0 ; vi < vlist.size() ; ++vi)
    {
      View *v = (View *) vlist[vi];
      v->set_center(v->center() + delta);
    }
}
