// ----------------------------------------------------------------------------
// Read NMR data processed using Varian's VNMR program
//

#include <fstream>		// Use ifstream
#include <stdio.h>		// Use FILE *
#include <string.h>		// Use strchr()

#include "binaryIO.h"		// Use read_s32(), read_f32(), ...
#include "blockfile.h"		// Use Block_File
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "memcache.h"		// use Memory_Cache
#include "nmrdata.h"		// Use NMR_Data
#include "spoint.h"		// Use IPoint, SPoint
#include "stringc.h"		// Use Stringy
#include "subarray.h"		// use position_index()
#include "system.h"		// Use file_exists(), file_path(), file_size()
#include "utility.h"		// use array_size(), fatal_error()

//
// The following structures are taken from VNMR User Programming manual,
// version 5.3, chapter 5, p 273-5
//

//
// File header and block header status bits
//
#define VNMR_S_DATA		0x1	// 0 = no data, 1 = data
#define VNMR_S_SPEC		0x2	// 0 = FID, 1 = spectrum
#define VNMR_S_32		0x4	// *
#define VNMR_S_FLOAT		0x8	// 0 = integer, 1 = floating point
#define VNMR_S_COMPLEX		0x10	// 0 = real, 1 = complex
#define VNMR_S_HYPERCOMPLEX	0x20	// 1 = hypercomplex

//
// file header status bits
//
#define VNMR_S_ACQPAR		0x80	// 0 = not Acqpar, 1 = Acqpar
#define VNMR_S_SECND		0x100	// 0 = first FT, 1 = second FT
#define VNMR_S_TRANSF		0x200	// 0 = regular, 1 = transposed
#define VNMR_S_NP		0x800	// 1 = np dimension is active
#define VNMR_S_NF		0x1000	// 1 = nf dimension is active
#define VNMR_S_NI		0x2000	// 1 = ni dimension is active
#define VNMR_S_NI2		0x4000	// 1 = ni2 dimension is active

//
// block header status bits
//
#define VNMR_MORE_BLOCKS	0x80	// 0 = absent, 1 = present
#define VNMR_NP_COMPLEX		0x100	// 0 = real, 1 = complex
#define VNMR_NF_COMPLEX		0x200	// 0 = real, 1 = complex
#define VNMR_NI_COMPLEX		0x400	// 0 = real, 1 = complex
#define VNMR_NI2_COMPLEX	0x800	// 0 = real, 1 = complex

//
// hypercomplex header status bits
//
#define VNMR_U_HYPERCOMPLEX	0x2	// 1 = hypercomplex block structure

//
// Used at start of each data file (FIDs, spectra, 2D)
//
struct vnmr_datafilehead
{
  long nblocks;		// number of blocks in file
  long ntraces;		// number of traces per block
  long np;		// number of elements per trace
  long ebytes;		// number of bytes per element
  long tbytes;		// number of bytes per trace
  long bbytes;		// number of bytes per block
  short vers_id;	// software version, file_id status bits
  short status;		// status of whole file
  long nbheaders;	// number of block headers per block
};

//
// Each file block contains the following header
//
struct vnmr_datablockhead
{
  short scale;		// scaling factor
  short status;		// status of data in block
  short index;		// block index
  short mode;		// mode of data in block
  long ctcount;		// ct value for FID
  float lpval;		// f2 (2D-f1) left phase in phasefile
  float rpval;		// f2 (2D-f1) right phase in phasefile
  float lvl;		// level drift correction
  float tlt;		// tilt drift correction
};

//
// Additional data block header for hypercomplex 2D data
//
struct vnmr_hypercmplxbhead
{
  short s_spare1;
  short status;		// status word for block header
  short s_spare2;
  short s_spare3;
  long l_spare1;
  float lpval1;		// 2D-f2 left phase
  float rpval1;		// 2D-f2 right phase
  float f_spare1;
  float f_spare2;
};

#define VNMR_FILE_HEADER_SIZE 32
#define VNMR_BLOCK_HEADER_SIZE 28

#define VNMR_SCALE_FACTOR 1e6

// ----------------------------------------------------------------------------
//
class VNMR_2D_Block_File : public Block_File
{
public:
  VNMR_2D_Block_File(const Stringy &path,
		     int block_header_size, bool transposed,
		     const IPoint &size, const IPoint &block_size,
		     Memory_Cache *);
  virtual ~VNMR_2D_Block_File();

  virtual bool read_block(const IPoint &block, float *values);
  virtual bool write_block(const IPoint &block, float *values);
private:
  FILE *fp;
  int block_header_size;
  bool transposed;

  offset_type block_location(const IPoint &block);
};

// ----------------------------------------------------------------------------
//
class VNMR_3D_Block_File : public Block_File
{
public:
  VNMR_3D_Block_File(const Stringy &base_path, int plane_axis,
		     int block_header_size, bool transposed,
		     const IPoint &size, const IPoint &block_size,
		     Memory_Cache *);
  virtual ~VNMR_3D_Block_File();

  virtual bool read_block(const IPoint &block, float *values);
  virtual bool write_block(const IPoint &block, float *values);
  virtual int cache_size();

private:
  int plane_axis;
  int block_header_size;
  bool transposed;
};

// ----------------------------------------------------------------------------
//
class procpar
{
public:
  procpar(const Stringy &path);
  ~procpar();

  bool int_value(const Stringy &tag, int *i) const;
  bool double_value(const Stringy &tag, double *d) const;
  bool string_value(const Stringy &tag, Stringy *s) const;
  int entry_count() const;

private:
  List tags, values;

  bool find(const Stringy &tag, Stringy *value) const;
};

static bool read_varian_parameters(const Stringy &procpar,
				   const List &nucleus_names,
				   const Stringy &phasefile,
				   int *block_header_size,
				   bool *transposed,
				   IPoint *size, IPoint *block_size,
				   SPoint *spectrometer_freq,
				   SPoint *spectral_width,
				   SPoint *origin_ppm,
				   Stringy *error_msg);
static bool check_phasefile(const Stringy &phasefile, const IPoint &sz,
			    int *block_header_size, bool *transposed,
			    IPoint *bsize, Stringy *error_msg);
static bool check_phasefile(FILE *fp, const Stringy &file, const IPoint &sz,
			    int *block_header_size, bool *transposed,
			    IPoint *bsize, Stringy *error_msg);
static bool unquote(const Stringy &quoted, Stringy *unquoted);
static bool read_procpar_entry(std::istream &strm,
			       Stringy *tag, Stringy *value, Stringy *next);
static bool read_procpar(const Stringy &path, const List &nucleus_names,
			 IPoint *size,
			 SPoint *spectrometer_freq, SPoint *spectral_width,
			 SPoint *origin_ppm, Stringy *error_msg);
static bool read_transmitter_frequency(const procpar &pp,
				       const Stringy &nucleus,
				       double *freq);
static bool read_varian_file_header(FILE *fp, struct vnmr_datafilehead *h);
static bool read_varian_block_header(FILE *fp, struct vnmr_datablockhead *h);

// ----------------------------------------------------------------------------
//
NMR_Data *varian_nmr_data(const Stringy &procpar_path,
			  const Stringy &vnmr_data_path,
			  const List &nucleus_names,
			  Stringy *error_msg,
			  Memory_Cache *mcache)
{
  int dim = nucleus_names.size();
  if (dim != 2 && dim != 3)
    { *error_msg = "must have 2 or 3 nucleus names"; return NULL; }

  bool transposed;
  int block_header_size;
  IPoint size, block_size;
  SPoint spectrometer_freq, spectral_width, origin_ppm;

  NMR_Data *nmrdata = NULL;
  Stringy phasefile = (dim == 2 ? vnmr_data_path : vnmr_data_path + ".1");
  if (read_varian_parameters(procpar_path, nucleus_names, phasefile,
			     &block_header_size, &transposed,
			     &size, &block_size,
			     &spectrometer_freq, &spectral_width,
			     &origin_ppm, error_msg))
    {
      Block_File *bf = NULL;
      if (dim == 2)
	bf = new VNMR_2D_Block_File(vnmr_data_path, block_header_size,
				    transposed, size, block_size, mcache);
      else if (dim == 3)
	{
	  int plane_axis = 1;
	  bf = new VNMR_3D_Block_File(vnmr_data_path, plane_axis,
				      block_header_size, transposed,
				      size, block_size, mcache);
	}

      if (bf)
	nmrdata = block_file_nmr_data(bf, spectrometer_freq, spectral_width,
				      origin_ppm, nucleus_names);
    }

  return nmrdata;
}

// ----------------------------------------------------------------------------
//
static bool read_varian_parameters(const Stringy &procpar,
				   const List &nucleus_names,
				   const Stringy &phasefile,
				   int *block_header_size,
				   bool *transposed,
				   IPoint *size, IPoint *block_size,
				   SPoint *spectrometer_freq,
				   SPoint *spectral_width,
				   SPoint *origin_ppm,
				   Stringy *error_msg)
{
  bool trans;
  int bhsize;
  IPoint sz, bsize;
  SPoint sfreq, swidth, orig;
  
  if (read_procpar(procpar, nucleus_names,
		   &sz, &sfreq, &swidth, &orig, error_msg)
      && check_phasefile(phasefile, sz, &bhsize, &trans, &bsize, error_msg))
    {
      *block_header_size = bhsize;
      *transposed = trans;
      *size = sz;
      *block_size = bsize;
      *spectrometer_freq = sfreq;
      *spectral_width = swidth;
      *origin_ppm = orig;
      return true;
    }

  return false;
}

// ----------------------------------------------------------------------------
//
static bool check_phasefile(const Stringy &phasefile, const IPoint &sz,
			    int *block_header_size, bool *transposed,
			    IPoint *bsize, Stringy *error_msg)
{
  FILE *fp = fopen(phasefile.cstring(), "rb");
  if (fp == NULL)
    {
      *error_msg = "Couldn't open phase file: " + phasefile;
      return false;
    }
  bool read_header = check_phasefile(fp, phasefile, sz,
				     block_header_size, transposed,
				     bsize, error_msg);
  fclose(fp);

  return read_header;
}

// ----------------------------------------------------------------------------
//
static bool check_phasefile(FILE *fp, const Stringy &file, const IPoint &sz,
			    int *block_header_size, bool *transposed,
			    IPoint *bsize, Stringy *error_msg)
{
  int dim = sz.dimension();

  struct vnmr_datafilehead fh;
  if (!read_varian_file_header(fp, &fh))
    { *error_msg = "Couldn't read " + file + " header."; return false; }
  else if ((fh.status & VNMR_S_DATA) == 0)
    { *error_msg = file + " contains no data."; return false; }
  else if ((fh.status & VNMR_S_SPEC) == 0)
    { *error_msg = file + " contains FID, not spectrum."; return false; }
  else if ((fh.status & VNMR_S_FLOAT) == 0 ||
	   (fh.status & VNMR_S_COMPLEX) != 0 ||
	   (fh.status & VNMR_S_HYPERCOMPLEX) != 0)
    { *error_msg = file + " is not real."; return false; }
  else if (fh.nbheaders != 1 && fh.nbheaders != 2)
    {
      *error_msg = file + " doesn't have 1 or 2 headers per block.";
      return false;
    }
  else if (fh.np != sz[0])
    {
      *error_msg = formatted_string("%s elements per trace np = %d "
				    "conflicts with procpar size %d",
				    file.cstring(), fh.np, sz[0]);
      return false;
    }
  else if (fh.nblocks * fh.ntraces < sz[dim-1])
    {
      *error_msg = formatted_string("%s has fewer traces (%d) than "
				    "expected from procpar (%d)",
				    file.cstring(),
				    fh.nblocks * fh.ntraces, sz[dim-1]);
      return false;
    }

  offset_type expected_size = (VNMR_FILE_HEADER_SIZE + fh.nblocks *
			       (fh.nbheaders * VNMR_BLOCK_HEADER_SIZE +
				4 * fh.np * fh.ntraces));  
  offset_type size = file_size(file);
  if (size < expected_size)
    {
      *error_msg = formatted_string("%s is smaller (%ld bytes) "
				    "than expected (%ld bytes)",
				    file.cstring(),
				    static_cast<long>(size),
				    static_cast<long>(expected_size));
      return false;
    }

  *block_header_size = static_cast<int>(fh.nbheaders * VNMR_BLOCK_HEADER_SIZE);

  *transposed = (fh.status & VNMR_S_TRANSF);

  *bsize = IPoint(dim);
  (*bsize)[0] = static_cast<int> (fh.np);
  (*bsize)[dim-1] = static_cast<int> (fh.ntraces);
  if (dim == 3)
    (*bsize)[1] = 1;

  return true;
}

// ----------------------------------------------------------------------------
//
procpar::procpar(const Stringy &path)
{
  std::ifstream pp(path.cstring());

  Stringy tag, value, next;
  while (read_procpar_entry(pp, &tag, &value, &next))
    {
      tags.append(new Stringy(tag));
      values.append(new Stringy(value));
    }
}

// ----------------------------------------------------------------------------
//
procpar::~procpar()
{
  free_string_list_entries(tags);
  free_string_list_entries(values);
}

// ----------------------------------------------------------------------------
//
bool procpar::int_value(const Stringy &tag, int *i) const
  { Stringy v; return find(tag, &v) && sscanf(v.cstring(), "%d", i) == 1; }
bool procpar::double_value(const Stringy &tag, double *d) const
  { Stringy v; return find(tag, &v) && sscanf(v.cstring(), "%lf", d) == 1; }
bool procpar::string_value(const Stringy &tag, Stringy *s) const
  { Stringy v; return find(tag, &v) && unquote(v, s); }
int procpar::entry_count() const
  { return tags.size(); }

// ----------------------------------------------------------------------------
//
bool procpar::find(const Stringy &tag, Stringy *value) const
{
  for (int t = 0 ; t < tags.size() ; ++t)
    if (*(Stringy *) tags[t] == tag)
      {
	*value = *(Stringy *) values[t];
	return true;
      }
  return false;
}

// ----------------------------------------------------------------------------
//
static bool unquote(const Stringy &quoted, Stringy *unquoted)
{
  const char *q = quoted.cstring();
  const char *q1 = strchr(q, '\"');
  const char *q2 = strrchr(q, '\"');
  if (q1 && q2 && q1 != q2)
    *unquoted = substring(q1 + 1, q2);

  return q1 && q2 && q1 != q2;
}

// ----------------------------------------------------------------------------
//
static bool read_procpar_entry(std::istream &strm,
			       Stringy *tag, Stringy *value, Stringy *next)
{
  while (next->is_empty() && stream_line(strm, next)) ;

  if (next->is_empty())
    return false;

  *tag = first_token(*next, NULL);
  *value = "";
  while (stream_line(strm, next))
    {
      Stringy rest;
      Stringy first = first_token(*next, &rest);
      int c;
      if (sscanf(first.cstring(), "%d", &c) == 1)
	*value << rest;
      else if (!first.is_empty())
	break;
    }

  return true;
}

// ----------------------------------------------------------------------------
//
static bool read_procpar(const Stringy &path, const List &nucleus_names,
			 IPoint *size,
			 SPoint *spectrometer_freq, SPoint *spectral_width,
			 SPoint *origin_ppm, Stringy *error_msg)
{
  int dim = nucleus_names.size();
  if (dim != 2 && dim != 3)
    {
      *error_msg = "must specify 2 or 3 nucleus names";
      return false;
    }

  procpar pp(path);

  if (pp.entry_count() == 0)
    {
      *error_msg = formatted_string("Couldn't find any parameters "
				    "in the procpar file %s", path.cstring());
      return false;
    }

  double sfreq[DIM];
  for (int a = 0 ; a < dim ; ++a)
    {
      Stringy nucleus =  *(Stringy *) nucleus_names[a];
      if (!read_transmitter_frequency(pp, nucleus, &sfreq[a]))
	{
	  *error_msg = formatted_string("Couldn't find the transmitter "
					"frequency for nucleus %s\n"
					"in the procpar file %s",
					nucleus.cstring(), path.cstring());
	  return false;
	}
    }

  double swidth[DIM], refloc[DIM], reffreq[DIM], origin[DIM];
  int siz[DIM];

  bool read_params = (pp.int_value("fn1", &siz[0]) &&
		      pp.double_value("sw1", &swidth[0]) &&
		      pp.double_value("rfl1", &refloc[0]) &&
		      pp.double_value("rfp1", &reffreq[0]));
  if (read_params && dim == 3)
    read_params = (pp.int_value("fn2", &siz[1]) &&
		   pp.double_value("sw2", &swidth[1]) &&
		   pp.double_value("rfl2", &refloc[1]) &&
		   pp.double_value("rfp2", &reffreq[1]));
  if (read_params)
    read_params = (pp.int_value("fn", &siz[dim-1]) &&
		   pp.double_value("sw", &swidth[dim-1]) &&
		   pp.double_value("rfl", &refloc[dim-1]) &&
		   pp.double_value("rfp", &reffreq[dim-1]));

  if (!read_params)
    {
      *error_msg = Stringy("Couldn't parse procpar file: ") + path;
      return false;
    }

  for (int a = 0 ; a < dim ; ++a)
    {
      siz[a] /= 2;	// real + imag --> #complex

      double refoffset = (swidth[a] - refloc[a]) / sfreq[a];
      double refppm = reffreq[a] / sfreq[a];
      origin[a] = refppm + refoffset;
    }

  *size = IPoint(dim, siz);
  *spectrometer_freq = SPoint(dim, sfreq);
  *spectral_width = SPoint(dim, swidth);
  *origin_ppm = SPoint(dim, origin);

  return true;
}

// ----------------------------------------------------------------------------
//
static bool read_transmitter_frequency(const procpar &pp,
				       const Stringy &nucleus,
				       double *freq)
{
  static char *nuctags[] = {"tn", "dn", "dn2", "dn3"};
  static char *freqtags[] = {"sfrq", "dfrq", "dfrq2", "dfrq3"};

  Stringy tnucleus;
  double tfreq;
  for (int k = 0 ; k < array_size(nuctags) ; ++k)
    if (pp.string_value(nuctags[k], &tnucleus) &&
	!tnucleus.is_empty() &&
	standard_nucleus_name(tnucleus) == nucleus &&
	pp.double_value(freqtags[k], &tfreq))
      {
	*freq = tfreq;
	return true;
      }
  return false;
}

// ----------------------------------------------------------------------------
//
static bool read_varian_file_header(FILE *fp, struct vnmr_datafilehead *h)
{
  return (read_s32(&h->nblocks, fp) &&
	  read_s32(&h->ntraces, fp) &&
	  read_s32(&h->np, fp) &&
	  read_s32(&h->ebytes, fp) &&
	  read_s32(&h->tbytes, fp) &&
	  read_s32(&h->bbytes, fp) &&
	  read_s16(&h->vers_id, fp) &&
	  read_s16(&h->status, fp) &&
	  read_s32(&h->nbheaders, fp));
}

// ----------------------------------------------------------------------------
//
static bool read_varian_block_header(FILE *fp, struct vnmr_datablockhead *h)
{
  return (read_s16(&h->scale, fp) &&
	  read_s16(&h->status, fp) &&
	  read_s16(&h->index, fp) &&
	  read_s16(&h->mode, fp) &&
	  read_s32(&h->ctcount, fp) &&
	  read_f32(&h->lpval, fp) &&
	  read_f32(&h->rpval, fp) &&
	  read_f32(&h->lvl, fp) &&
	  read_f32(&h->tlt, fp));
}

// ----------------------------------------------------------------------------
//
VNMR_2D_Block_File::VNMR_2D_Block_File(const Stringy &path,
				       int block_header_size,
				       bool transposed,
				       const IPoint &size,
				       const IPoint &block_size,
				       Memory_Cache *mcache) :
  Block_File(path, size, block_size, mcache)
{
  fp = fopen(path.cstring(), "rb");
  if (fp == NULL)
    fatal_error("VNMR_2D_Block_File::Failed opening: %s", path.cstring());

  this->fp = fp;
  this->block_header_size = block_header_size;
  this->transposed = transposed;
}

// ----------------------------------------------------------------------------
//
VNMR_2D_Block_File::~VNMR_2D_Block_File()
{
  flush_cache();
  fclose(fp);
  fp = NULL;
}

// ----------------------------------------------------------------------------
//
bool VNMR_2D_Block_File::read_block(const IPoint &block, float *values)
{
  offset_type position = block_location(block);
  offset_type size = block_volume();

  struct vnmr_datablockhead bh;

  bool read;
  if (transposed)
    read = (seek_position(fp, position, 0) &&
	    read_varian_block_header(fp, &bh) &&
	    read_f32_array(values, size, fp));
  else
    {
      float *temp = new float [size];
      read = (seek_position(fp, position, 0) &&
	      read_varian_block_header(fp, &bh) &&
	      read_f32_array(temp, size, fp));
      if (read)
	little_to_big_endian_indexing(block_region(block), temp, values);
      delete [] temp;
    }

  if (read)
    for (int k = 0 ; k < size ; ++k)
      values[k] *= VNMR_SCALE_FACTOR;

  return read;
}

// ----------------------------------------------------------------------------
//
bool VNMR_2D_Block_File::write_block(const IPoint &, float *)
{
  return false;		// Not implemented.
}

// ----------------------------------------------------------------------------
//
offset_type VNMR_2D_Block_File::block_location(const IPoint &block)
{
  offset_type block_number = position_index(block, block_space(), true);
  offset_type data_start = VNMR_FILE_HEADER_SIZE;
  offset_type block_size = block_header_size + block_volume() * sizeof(float);

  return data_start + block_number * block_size;
}

// ----------------------------------------------------------------------------
//
VNMR_3D_Block_File::VNMR_3D_Block_File(const Stringy &base_path,
				       int plane_axis,
				       int block_header_size,
				       bool transposed,
				       const IPoint &size,
				       const IPoint &block_size,
				       Memory_Cache *mcache) :
  Block_File(base_path, size, block_size, mcache)
{
  this->plane_axis = plane_axis;
  this->block_header_size = block_header_size;
  this->transposed = transposed;
}

// ----------------------------------------------------------------------------
//
VNMR_3D_Block_File::~VNMR_3D_Block_File()
{
  flush_cache();
}

// ----------------------------------------------------------------------------
//
bool VNMR_3D_Block_File::read_block(const IPoint &block, float *values)
{
  Stringy path = this->path() + formatted_string(".%d", block[plane_axis] + 1);
  IPoint size_2d = size().remove_component(plane_axis);
  IPoint bsize_2d = block_size().remove_component(plane_axis);
  VNMR_2D_Block_File bf2d(path, block_header_size, transposed,
			  size_2d, bsize_2d, memory_cache());
  IPoint block_2d = block.remove_component(plane_axis);
  return bf2d.read_block(block_2d, values);
}

// ----------------------------------------------------------------------------
//
bool VNMR_3D_Block_File::write_block(const IPoint &, float *)
{
  return false;		// Not implemented.
}

// ----------------------------------------------------------------------------
//
int VNMR_3D_Block_File::cache_size()
{
  return 1;
}
