// ----------------------------------------------------------------------------
// Window system routines common to all platforms.
// This file is included by the platform specific winsystem-PLATFORM.cc
//

#include <ctype.h>		// use isupper(), tolower()
#include <math.h>		// use fabs()
#include <stdarg.h>		// use ...
#include <stdio.h>		// Use sscanf()
#include <stdlib.h>		// use getenv()
#include <string.h>		// Use strlen(), strcmp(), strchr(), strcpy()

#include <tcl.h>		// use Tcl_* routines
#include <tk.h>			// use Tk_* routines

//
// Tk directly uses XEvent, GC, and XColor structures from Xlib
//
// I call the following Xlib routines that are not part of Tk.
//
// Graphics contexts:
//
//	XCreateGC(), XFreeGC()				in create_gc()
//	XSetFont()					in set_scaled_font()
//	XSetForeground()				in set_foreground()
//	XGetGCValues(), XSetFunction()			in xor_gc_mode()
//	XSetBackground(), XSetGraphicsExposures()	in DrawWinP()
//
// Drawing primitives:
//
//	XDrawLine()					in draw_line()
//	XDrawRectangle()				in draw_rectangle()
//	XFillRectangle()				in fill_rectangle()
//	XFillPolygon()					in fill_triangle()
//	XDrawArc()					in draw_arc()
//	XClearArea()					in clear_area()
//	XCopyArea()					in translate_contents()
//
// Displaying vertical text:
//
//	XCreatePixmap(), XDrawImageString(),
//	XGetImage(), XPutImage(), XPutPixel(),
//	XDestroyImage(), XFreePixmap()		     in draw_vertical_string()
//
// Translating key press events to characters:
//
//	XLookupString(), IsFunctionKey(), XK_F1		in key_pressed()
//
// Miscellaneous
//
//	XSync()
//
// Looks like all of the X calls are available in the Microsoft Windows
// version of Tk except the XImage routines I use for drawing vertical
// text.
//

#include <X11/Xlib.h>
#include <X11/Xutil.h>		// use XPutPixel(), XGetPixel(),
				// XDestroyImage(), IsFunctionKey()
#include <X11/keysym.h>		// Use XK_F1
#include <X11/Xatom.h>		// use Atom

#include "color.h"		// use Color
#include "list.h"		// Use List
#include "memalloc.h"		// use new()
#include "num.h"		// Use max(), round()
#include "stringc.h"		// Use Stringy
#include "system.h"		// Use is_data_available(), file_path()
#include "table.h"		// use Table
#include "utility.h"		// Use fatal_error()
#include "winsystem.h"

// ----------------------------------------------------------------------------
//
class Work_Proc_Manager;
class Event_Callback_Table;
class Timer_Callback_List;
class Pointer_Pause_List;
class Variable_Callback_Manager;
class Scroll_Callback_Table;
class Input_Callback_Manager;
class Scaled_Font_Manager;
class Color_Manager;

// ----------------------------------------------------------------------------
// State variables for window interface.
//
class WinSysP
{
public:
  WinSysP(WinSys &, Tcl_Interp *);
  ~WinSysP();

  WinSys &ws;
  bool own_tcl_interp;
  Tcl_Interp *tcl_interp;
  bool debug_tcl;
  List hidden_dialogs;
  int unique_name_count;
  Time last_double_click_time;
  bool exit_requested;

  //
  // These managers maintain a list or table of objects so that the
  // objects can be looked up, mostly in order to remove them and their
  // associeated Tcl callbacks.
  //
  // These are pointers to manager instances because there was no other
  // way to avoid separating the manager class declarations from their
  // implementations.
  //
  Work_Proc_Manager *		work_procs;
  Event_Callback_Table *	event_callbacks;
  Timer_Callback_List *		timer_callbacks;
  Pointer_Pause_List *		pause_callbacks;
  Variable_Callback_Manager *	variable_callbacks;
  Scroll_Callback_Table *	scroll_callbacks;
  Input_Callback_Manager *	input_callbacks;
  Scaled_Font_Manager *		fonts;
  Color_Manager *		colors;
};

// ----------------------------------------------------------------------------
// This object packages a Widget and WinSys for callbacks that require
// the two together and registers a destroy handler to delete itself if the
// widget is destroyed.
//
// This is currently unused and unimplemented.
//
class WinSys_Widget
{
 public:
  WinSys_Widget(WinSys &, Widget);
  ~WinSys_Widget();

  WinSys &ws;
  Widget w;
};

typedef bool (*Filter_Proc)(WinSysP *, XEvent *);

// ----------------------------------------------------------------------------
//
static void update_argv(WinSysP *wsp, int *argc, char **argv);
static void initialize_tk(WinSysP *wsp, const Stringy &classname,
			  const Stringy &appname,
			  const Stringy &resourcedir);
static Widget main_widget(WinSysP *wsp);
extern "C"
{
  static void install_idle_callback_cb(CB_Data);
  static void work_proc_cb(CB_Data);
}
static void add_tcl_command(WinSysP *wsp, const Stringy &name,
			    Tcl_CmdProc, CB_Data);
static void remove_tcl_command(WinSysP *wsp, const Stringy &name);
extern "C"
{
static void ec_event_cb(CB_Data, XEvent *);
static void ec_destroy_cb(ClientData ecb, XEvent *event);
}
static void add_event_handler(WinSysP *wsp, Widget w, unsigned long mask,
			      Filter_Proc, CB_Proc cb, CB_Data cb_data);
static void remove_event_handler(WinSysP *wsp, Widget w, unsigned long mask,
				 Filter_Proc, CB_Proc cb, CB_Data cb_data);
extern "C"
{
static void tc_timer_cb(CB_Data client_data);
static void pp_destroy_cb(ClientData pcb, XEvent *event);
static void pp_motion_cb(CB_Data ppause, XEvent *);
}
static void add_pointer_pause_callback(WinSysP *wsp, Widget w, CB_Proc, CB_Data);
static void remove_pointer_pause_callback(WinSysP *wsp, Widget w, CB_Proc, CB_Data);
extern "C"
{
static Tk_RestrictAction motion_filter(ClientData, XEvent *);
static Tk_RestrictAction exposure_filter(ClientData, XEvent *);
}
static void add_tk_event_handler(Widget w, unsigned long mask,
				 Tk_EventProc cb, CB_Data cb_data);
static void remove_tk_event_handler(Widget w, unsigned long mask,
				    Tk_EventProc cb, CB_Data cb_data);
static void create_named_font(WinSysP *wsp, const Stringy &name);
static Stringy capitalize(const Stringy &s);
static void unshow_cb(Widget shell, CB_Data, CB_Data);
static void toplevel_geometry_option(WinSysP *wsp, Widget t);
static void toplevel_title_option(WinSysP *wsp, Widget t);
static void move_onto_screen(WinSysP *wsp, Widget w, int *x, int *y);
static void debug_message(const Stringy &msg);
static bool tcl_command(WinSysP *wsp, const Stringy &cmd);
static bool tcl_command(WinSysP *wsp, const Stringy &format, const Stringy &arg1);
static bool tcl_command(WinSysP *wsp, const Stringy &format, const Stringy &arg1,
			const Stringy &arg2);
static bool tcl_command(WinSysP *wsp, const Stringy &format, const Stringy &arg1,
			const Stringy &arg2, const Stringy &arg3);
static bool tcl_command(WinSysP *wsp, const Stringy &format, const Stringy &arg1,
			const Stringy &arg2, const Stringy &arg3,
			const Stringy &arg4);
static bool tcl_command(WinSysP *wsp, const Stringy &format, const Stringy &arg1,
			const Stringy &arg2, const Stringy &arg3,
			const Stringy &arg4, const Stringy &arg5);
static Stringy command_result(WinSysP *wsp);
static bool command_result_integer(WinSysP *wsp, int *i);
static void report_tcl_error(WinSysP *wsp, const Stringy &msg);
static bool command_result_double_pair(WinSysP *wsp, double *a, double *b);
static bool command_result_list(WinSysP *wsp, List *strings);
static Stringy quoted_word(const Stringy &word);
static Stringy tcl_word_list(const List &words);
static Tk_Window window(Widget w);
static Window x_window(Widget w);
static bool window_exists(Widget w);
static void make_window_exist(Widget w);
static void map_widget(Widget w);
static bool is_mapped(Widget w);
static Widget parent_widget(Widget w);
static Stringy path(Widget w);
static bool tcl_command_exists(WinSysP *wsp, const Stringy &name);
static Stringy unique_child_path(WinSysP *wsp, Widget parent, const Stringy &name);
static Stringy child_path(Widget parent, const Stringy &child);
static Stringy command_name(WinSysP *wsp, Widget parent, const Stringy &name);
static Stringy unique_name(WinSysP *wsp, const Stringy &name);
static Stringy get_option(WinSysP *wsp, Widget w, const Stringy &option);
extern "C"
{
  static void cc_destroy_cb(ClientData ccb, XEvent *event);
  static int tk_cb(ClientData self, Tcl_Interp *, int argc, const char *argv[]);
  static char *variable_set_cb(ClientData cmdcb, Tcl_Interp *,
			       const char *, const char *, int);
}
static void variable_changed_callback(WinSysP *wsp, Widget w, CB_Proc cb, CB_Data cb_data);
static void remove_variable_changed_callback(WinSysP *wsp, Widget, CB_Proc, CB_Data);
static Stringy widget_variable(Widget w);
static Stringy get_variable(WinSysP *wsp, Widget w);
static void set_variable(WinSysP *wsp, Widget w, const Stringy &value);
static void default_button_text(WinSysP *wsp, const Stringy &path, const Stringy &text);
static Widget edit_field_title(WinSysP *wsp, Widget ef);
static void set_edit_field_cb(Widget pane, CB_Data edit_field, CB_Data event);
static bool scrollbar_position(WinSysP *wsp, Widget sbar, double *a, double *b);
static void stack_widgets(WinSysP *wsp, bool vertical, Widget widgets[], Widget stretch_me);
static void clear_grid(WinSysP *wsp, Widget w);
static Stringy button_label_option(WinSysP *wsp, Widget pane, const Stringy &name);
static int menu_entry_count(WinSysP *wsp, Widget pane);
static bool has_tearoff_entry(WinSysP *wsp, Widget pane);
static void list_select_cb(Widget list, CB_Data, CB_Data event);
static int visible_lines(WinSysP *wsp, Widget list);
static Stringy list_item(WinSysP *wsp, Widget list, int position);
static List list_items(WinSysP *wsp, Widget list);
static void forget_bit_gravity(Widget w);
static bool exposed_region(XEvent *event, int *x, int *y, int *w, int *h);
static XColor *drawing_area_background(WinSysP *wsp, Widget w);
static void xor_gc_mode(Display *display, GC gc,
			unsigned long *prev_foreground, int *prev_gc_mode);
static void restore_gc_mode(Display *display, GC gc,
			    unsigned long foreground, int gc_mode);
static Stringy default_path(WinSysP *wsp, const Stringy &filetype, const Stringy &path);
static void set_default_path(WinSysP *wsp, const Stringy &filetype, const Stringy &path);
static Widget option_button(WinSysP *wsp, Widget option_menu);
static Widget option_pane(WinSysP *wsp, Widget option_menu);
static void option_selected_cb(Widget w, CB_Data menubutton, CB_Data);
static void set_delete_protocol(WinSysP *wsp, Widget shell, CB_Proc cb, CB_Data cb_data);
static void remove_delete_protocol(WinSysP *wsp, Widget shell, CB_Proc cb, CB_Data cb_data);
static bool map_filter(WinSysP *, XEvent *event);
static bool button_1_filter(WinSysP *, XEvent *event);
static bool button_2_filter(WinSysP *, XEvent *event);
static bool button_3_filter(WinSysP *, XEvent *event);
static bool move_filter(WinSysP *, XEvent *event);
static bool drag_filter(WinSysP *, XEvent *event);
static bool got_focus_filter(WinSysP *, XEvent *event);
static bool destroy_filter(WinSysP *, XEvent *event);
static bool resize_filter(WinSysP *, XEvent *event);
static bool enter_key_filter(WinSysP *, XEvent *event);
static bool key_press_filter(WinSysP *, XEvent *event);
static bool double_button_1_filter(WinSysP *, XEvent *event);
static Stringy table_element_path(Widget table, int row, int column);
static List child_widgets(WinSysP *wsp, Widget parent);
static void destroyed_modal_widget_cb(Widget, CB_Data destroyed, CB_Data);
static void raise_modal_widget_cb(Widget, CB_Data, CB_Data);
static Widget toplevel(WinSysP *wsp, Widget w);
extern "C"
{
static void iconify_all_cb(ClientData, XEvent *);
}
static Display *widget_display(Widget w);
static int widget_width(Widget w);
static int widget_height(Widget w);
static void unshow_dialog_cb(Widget, CB_Data client_data, CB_Data);
extern "C"
{
static void ic_input_cb(ClientData self, int);
}
static GC create_gc(Display *display, Window w);
static void free_gc(Display *display, GC gc);
static Tk_Font label_font(WinSysP *wsp, Widget w);
static void text_size(Tk_Font font, const Stringy &string,
		      int *w, int *ascent, int *descent);
static void draw_vertical_text(Widget w, GC gc, Tk_Font font, int x, int y,
			       const Stringy &string, bool up);

// ----------------------------------------------------------------------------
// The following functions are implemented in the platform specific window
// system file winsystem-PLATFORM.cc which includes this file.
//
// Most of these functions are needed because Tk does not provide a complete
// interface.  Some X windows calls are needed and are not available or
// poorly implemented in the win32 port of Tk.
//

//
// The win32 Tk 8 implementation of XLookupString() is inadequate doesn't
// handle keysyms outside the range 0-255 which misses delete and escape.
// Also it returns only the char, a keysym is never returned.
//
static bool x_key(XEvent *, char *c);
static bool x_function_key(XEvent *, int *f, bool *shifted);

//
// Win32 Tk 8 doesn't implement XClearArea().
//
static void eventually_redraw(Widget, int x, int y, int w, int h);
static void draw_background(Display *, Window, GC, int x, int y, int w, int h);

//
// Win32 Tk 8 doesn't erase when window when exposure is generated.
//
static void set_background_erase(Widget, GC, unsigned long bg);

//
// Win32 Tk 8 doesn't adequately implement XGetImage() / XPutImage() to
// rotate text.
//
static void draw_vertical_text(Widget, GC, Tk_Font, int x, int y,
			       const Stringy &string, bool up);

//
// Win32 Tk 8 XCopyArea() does not handle graphics exposures.
//
static void translate_window_contents(Widget, GC, int dx, int dy);

//
// Communication with Netscape uses X properties under unix and dynamics
// data exchange (DDE) under win32.
//
static bool platform_specific_show_url(WinSysP *, const Stringy &url);

// ----------------------------------------------------------------------------
// The Tcl_MakeFileHandler() is Unix specific.
//
static void create_tcl_file_handler(FILE *, int, Tcl_FileProc *, ClientData);
static void remove_tcl_file_handler(FILE *, Tcl_FileProc *, ClientData);

// ----------------------------------------------------------------------------
//
WinSys::WinSys(const Stringy &classname, const Stringy &appname,
	       const Stringy &resourcedir, int *argc, char **argv)
{
  //
  // sparky-no-python.exe crashes on start-up on Windows in Tcl_Init() using
  // Tcl 8.4.5 unless Tcl_FindExecutable() is called first.
  //
  Tcl_FindExecutable(argv[0]);

  wsp = new WinSysP(*this, NULL);

  //
  // Need the following so Tk sets classname to Sparky.
  // Without it resources specified as Sparky*... are not found.
  // tk appname command below apparently does not fix classname.
  //
  tcl_command(wsp, "set argv0 %s", appname);

  //
  // Set Tcl argc and argv so standard X options like -display are processed.
  //
  tcl_command(wsp, "set argc %s", Stringy(*argc - 1));
  char *args = Tcl_Merge(*argc - 1, argv + 1);
  tcl_command(wsp, "set argv %s", args);
  Tcl_Free(args);

  if (Tk_Init(wsp->tcl_interp) == TCL_ERROR)
    { debug_message(command_result(wsp) + "\n"); }

  update_argv(wsp, argc, argv);

  initialize_tk(wsp, classname, appname, resourcedir);
}

// ----------------------------------------------------------------------------
// Read Tcl argv variable and update C argc and argv to match.
// This is used after Tk removes standard X options from the command line.
//
static void update_argv(WinSysP *wsp, int *argc, char **argv)
{
  const char *tcl_argv = Tcl_GetVar(wsp->tcl_interp, "argv", TCL_GLOBAL_ONLY);
  if (tcl_argv == NULL)
    return;

  int ac;
  const char **tcl_args;
  if (Tcl_SplitList(wsp->tcl_interp, tcl_argv, &ac, &tcl_args) == TCL_ERROR)
    report_tcl_error(wsp, "Couldn't parse Tcl argv list.\n");

  if (ac < *argc - 1)
    {
      *argc = ac + 1;
      for (int a = 0 ; a < ac ; ++a)
	argv[a+1] = allocate_character_array(tcl_args[a]);
      argv[ac+1] = NULL;
    }
  Tcl_Free((char *) tcl_args);
}

// ----------------------------------------------------------------------------
// Use a Tcl interpretter created by someone else (eg by Python/Tkinter).
// The main window of the passed in Tcl interpretter will be used and
// destroyed when Sparky finishes.
//
WinSys::WinSys(const Stringy &classname, const Stringy &appname,
	       const Stringy &resourcedir, void *tcl_interpretter)
{
  wsp = new WinSysP(*this, (Tcl_Interp *) tcl_interpretter);

  initialize_tk(wsp, classname, appname, resourcedir);
}

// ----------------------------------------------------------------------------
//
static void initialize_tk(WinSysP *wsp, const Stringy &classname,
			  const Stringy &appname,
			  const Stringy &resourcedir)
{
  //
  // Tkinter disables Tk error message reporting by settings its own
  // tkerror function.  Remove this so Tk errors get reported.
  //
  if (tcl_command_exists(wsp, "tkerror"))
    tcl_command(wsp, "rename tkerror {}");

  //
  // Option database
  //
  tcl_command(wsp, "tk appname %s", appname);
  Stringy resources = file_path(resourcedir, classname);
  tcl_command(wsp, "option readfile %s startupFile", resources);

  // For debugging X errors
  // XSynchronize(widget_display(main_widget()), true);

  //
  // Configure main window
  //
  Widget w = main_widget(wsp);
  toplevel_geometry_option(wsp, w);
  add_tk_event_handler(w, StructureNotifyMask, iconify_all_cb, wsp);

  //
  // Main window was created before option database was read
  // so fix its background.
  //
  Stringy bg = wsp->ws.read_application_resource(w, "background",
						 "Background", "");
  if (!bg.is_empty())
    tcl_command(wsp, "%s configure -background %s", path(w), bg);

  //
  // Create named fonts
  //
  create_named_font(wsp, "normalFont");
  create_named_font(wsp, "fixedWidthFont");
  create_named_font(wsp, "scalableFont");
}

// ----------------------------------------------------------------------------
//
WinSys::~WinSys()
{
  delete wsp;
  wsp = NULL;
}

// ----------------------------------------------------------------------------
//
Widget WinSys::main_widget()
  { return ::main_widget(wsp); }
static Widget main_widget(WinSysP *wsp)
{
  Widget w = (Widget) Tk_MainWindow(wsp->tcl_interp);
  if (w == NULL)
    fatal_error("%s", command_result(wsp).cstring());
  return w;
}

// ----------------------------------------------------------------------------
//
void WinSys::set_keyboard_focus_child(Widget w)
{
  tcl_command(wsp, "focus %s", path(w));
}

// ----------------------------------------------------------------------------
//
void WinSys::event_loop()
{
  while (!wsp->exit_requested &&
	 Tcl_DoOneEvent(TCL_ALL_EVENTS))
    ;
}

// ----------------------------------------------------------------------------
//
void WinSys::exit_event_loop()
{
  wsp->exit_requested = true;
}

// ----------------------------------------------------------------------------
// Set a pointer grab on widget and process events until reply is given.
// A reply is detected when the replied variable is set to true.  This is
// done by callbacks typically on the modal dialog buttons.
// We also stop waiting for a reply if the window is destroyed.
// If a reply is given the return value is true, and if the window is
// destroyed the return value will be false.
//
// Even though a pointer grab has been performed, other windows can be
// destroyed by clicking the close button on window frame.  If this causes
// objects to be destroyed before the modal reply chaos can result.
// Currently this is handled by having the callbacks that handle query delete
// requests from the window manager refuse the delete if a grab is in effect.
//
// Warning, query and wait dialogs are full app modal.
// Warning and wait are just queries.  Unify them?
// No wait doesn't idle awaiting a button press.
//
bool WinSys::process_until_modal_reply(Widget w, bool *replied)
{
  map_widget(w);
  raise_widget(w);
  grab_events(w);

  bool destroyed = false;
  add_event_callback(destroy_event, w,
		     destroyed_modal_widget_cb, &destroyed);

  while (!destroyed && !*replied)
    if (Tcl_DoOneEvent(TCL_ALL_EVENTS) == 0)
      break;

  if (!destroyed)
    {
      remove_event_callback(destroy_event, w,
			    destroyed_modal_widget_cb, &destroyed);
      ungrab_events(w);
    }

  return *replied;
}

// ----------------------------------------------------------------------------
//
void WinSys::process_pending_events()
{
  while (Tcl_DoOneEvent(TCL_ALL_EVENTS | TCL_DONT_WAIT))
    ;
}

// ----------------------------------------------------------------------------
//
bool WinSys::more_motion_events()
{
  bool found_motion_event = false;
  ClientData previous_data, data;
  Tk_RestrictProc *previous =
    Tk_RestrictEvents(motion_filter, &found_motion_event, &previous_data);
  while (Tcl_DoOneEvent(TCL_WINDOW_EVENTS | TCL_DONT_WAIT) &&
	 !found_motion_event)
    ;
  Tk_RestrictEvents(previous, previous_data, &data);
  return found_motion_event;
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static Tk_RestrictAction motion_filter(ClientData foundp, XEvent *event)
{
  if (event->type == MotionNotify)
    *(bool *)foundp = true;
  return TK_DEFER_EVENT;
}
}

// ----------------------------------------------------------------------------
// Process all queued exposure events.
//
void WinSys::process_exposure_events(Widget w)
{
  if (window_exists(w))
    {
      //
      // Without this XSync() frequently miss exposure events.
      // This results in blank strips when scrolling contour plots.
      //
      XSync(widget_display(w), False);

      ClientData previous_data, data;
      Tk_RestrictProc *previous =
	Tk_RestrictEvents(exposure_filter, (ClientData) x_window(w),
			  &previous_data);
      while (Tcl_DoOneEvent(TCL_WINDOW_EVENTS | TCL_DONT_WAIT))
	;
      Tk_RestrictEvents(previous, previous_data, &data);
    }
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static Tk_RestrictAction exposure_filter(ClientData window, XEvent *event)
{
  if (event->type == Expose || event->type == GraphicsExpose)
    if (event->xexpose.window == (Window) window)
      return TK_PROCESS_EVENT;
  return TK_DEFER_EVENT;
}
}

// ----------------------------------------------------------------------------
//
void WinSys::repaint_widget(Widget w)
{
  process_exposure_events(w);
  tcl_command(wsp, "update idletasks");
}

// ----------------------------------------------------------------------------
//
class Work_Proc
{
public:
  Work_Proc(CB_Func fn, CB_Data cb_data, double priority);

  CB_Func fn;
  CB_Data cb_data;
  double priority;
};

// ----------------------------------------------------------------------------
//
class Work_Proc_Manager
{
public:
  Work_Proc_Manager();
  ~Work_Proc_Manager();
  void add_work_proc(CB_Func fn, CB_Data cb_data, double priority);
  void set_work_proc_priority(CB_Func fn, CB_Data cb_data, double priority);
  void remove_work_proc(CB_Func fn, CB_Data cb_data);
  void run_proc();

private:
  bool work_proc_registered;
  Tcl_TimerToken id;
  List waiting;					// Pointers to work_procs

  Work_Proc *find(CB_Func fn, CB_Data cb_data);
  void remove_work_proc(Work_Proc *wp);
  void register_idle_callback();
  Work_Proc *highest_priority();
};

// ----------------------------------------------------------------------------
//
Work_Proc_Manager::Work_Proc_Manager()
{
  work_proc_registered = false;
}

// ----------------------------------------------------------------------------
//
Work_Proc_Manager::~Work_Proc_Manager()
{
  List copy = waiting;
  for (int wi = 0 ; wi < copy.size() ; ++wi)
    remove_work_proc((Work_Proc *) copy[wi]);

  if (work_proc_registered)
    {
      Tcl_DeleteTimerHandler(id);
      Tcl_CancelIdleCall(work_proc_cb, this);
    }
}

// ----------------------------------------------------------------------------
//
void Work_Proc_Manager::add_work_proc(CB_Func fn, CB_Data cb_data,
				      double priority)
{
  if (!find(fn, cb_data))
    {
      Work_Proc *wp = new Work_Proc(fn, cb_data, priority);

      waiting.insert(wp, 0);
      if (!work_proc_registered)
	register_idle_callback();
    }
}

// ----------------------------------------------------------------------------
//
void Work_Proc_Manager::remove_work_proc(CB_Func fn, CB_Data cb_data)
{
  Work_Proc *wp = find(fn, cb_data);
  if (wp)
    remove_work_proc(wp);
}

// ----------------------------------------------------------------------------
//
void Work_Proc_Manager::remove_work_proc(Work_Proc *wp)
{
  waiting.erase(wp);
  delete wp;
}

// ----------------------------------------------------------------------------
//
void Work_Proc_Manager::set_work_proc_priority(CB_Func fn, CB_Data cb_data,
					       double priority)
{
  Work_Proc *wp = find(fn, cb_data);
  if (wp)
    wp->priority  = priority;
}

// ----------------------------------------------------------------------------
// Use Tcl timer which then registers an idle callback.
// This works around limitations of idle callbacks discussed in the
// Tcl_DoWhenIdle() man page.  If the idle callback is reregistered
// continuously then no window events will get processed.
//
void Work_Proc_Manager::register_idle_callback()
{
  id = Tcl_CreateTimerHandler(0, install_idle_callback_cb, this);
  work_proc_registered = true;
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static void install_idle_callback_cb(CB_Data cb_data)
  { Tcl_DoWhenIdle(work_proc_cb, cb_data); }
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static void work_proc_cb(CB_Data wpmanager)
{
  Work_Proc_Manager *wpm = (Work_Proc_Manager *) wpmanager;
  wpm->run_proc();
}
}

// ----------------------------------------------------------------------------
//
void Work_Proc_Manager::run_proc()
{
  if (!waiting.empty())
    {
      Work_Proc *wp = highest_priority();
      CB_Func fn = wp->fn;
      CB_Data cb_data = wp->cb_data;
      remove_work_proc(wp);
      fn(cb_data);
    }

  if (waiting.empty())
    work_proc_registered = false;
  else
    register_idle_callback();
}

// ----------------------------------------------------------------------------
//
Work_Proc *Work_Proc_Manager::highest_priority()
{
  Work_Proc *high_wp = NULL;
  for (int wpi = 0 ; wpi < waiting.size() ; ++wpi)
    {
      Work_Proc *wp = (Work_Proc *) waiting[wpi];
      if (!high_wp || wp->priority > high_wp->priority)
	high_wp = wp;
    }
  return high_wp;
}

// ----------------------------------------------------------------------------
//
Work_Proc *Work_Proc_Manager::find(CB_Func fn, CB_Data cb_data)
{
  for (int wpi = 0 ; wpi < waiting.size() ; ++wpi)
    {
      Work_Proc *wp = (Work_Proc *) waiting[wpi];
      if (wp->fn == fn && wp->cb_data == cb_data)
	return wp;
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
Work_Proc::Work_Proc(CB_Func fn, CB_Data cb_data, double priority)
{
  this->fn = fn;
  this->cb_data = cb_data;
  this->priority = priority;
}

// ----------------------------------------------------------------------------
//
void WinSys::add_work_procedure(CB_Func fn, CB_Data cb_data, double priority)
  { wsp->work_procs->add_work_proc(fn, cb_data, priority); }
void WinSys::remove_work_procedure(CB_Func fn, CB_Data cb_data)
  { wsp->work_procs->remove_work_proc(fn, cb_data); }
void WinSys::set_work_procedure_priority(CB_Func fn, CB_Data cb_data,
					 double priority)
  { wsp->work_procs->set_work_proc_priority(fn, cb_data, priority); }

// ----------------------------------------------------------------------------
//
static void add_tk_event_handler(Widget w, unsigned long mask,
				 Tk_EventProc cb, CB_Data cb_data)
  { Tk_CreateEventHandler(window(w), mask, cb, cb_data); }
static void remove_tk_event_handler(Widget w, unsigned long mask,
				    Tk_EventProc cb, CB_Data cb_data)
  { Tk_DeleteEventHandler(window(w), mask, cb, cb_data); }

// ----------------------------------------------------------------------------
//
static void add_tcl_command(WinSysP *wsp, const Stringy &name,
			    Tcl_CmdProc cb, CB_Data cb_data)
{
  Tcl_CreateCommand(wsp->tcl_interp, (char *)name.cstring(), cb, cb_data, NULL);
  if (wsp->debug_tcl)
    debug_message("proc " + name + " <c++-command>\n");
}
static void remove_tcl_command(WinSysP *wsp, const Stringy &name)
{
  Tcl_DeleteCommand(wsp->tcl_interp, (char *) name.cstring());
}

// ----------------------------------------------------------------------------
// Tk event handlers with added filtering.
//
// The C++ object remembers the extra event filtering to be done.
// The event callback is deleted when the widget is destroyed or when
// remove_event_callback(w, mask, filter, cb, cb_data) is called.
// There is a table to so that the C++ objects can be looked up using
// the (w, mask, filter, cb, cb_data) parameters.
//
// The Tk bind command provides a similar facility.
//
class Event_Callback
{
public:
  Event_Callback(WinSysP *, Widget, unsigned long mask,
		 Filter_Proc, CB_Proc, CB_Data);
  ~Event_Callback();

  void register_with_tk();

  bool operator==(const Event_Callback &ecb) const;
  unsigned long hash() const;

private:
  WinSysP *wsp;
  Widget w;
  unsigned long mask;
  Filter_Proc filter;
  CB_Proc cb;
  CB_Data cb_data;
  bool registered;
  class Event_Callback_Table *table;

  friend void ec_event_cb(CB_Data ecallback, XEvent *event);
};

// ----------------------------------------------------------------------------
//
class Event_Callback_Table
{
public:
  Event_Callback_Table();
  ~Event_Callback_Table();

  void add_callback(Event_Callback *ecb);
  void remove_callback(Event_Callback *ecb);
  Event_Callback *find_callback(const Event_Callback &key);

private:
  Table ecb_table;

  static bool ecb_equal(TableKey ecb1, TableKey ecb2);
  static unsigned long ecb_hash(TableKey ecb);
};

// ----------------------------------------------------------------------------
//
Event_Callback::Event_Callback(WinSysP *wsp, Widget w, unsigned long mask,
			       Filter_Proc filter, CB_Proc cb, CB_Data cb_data)
{
  this->wsp = wsp;
  this->w = w;
  this->mask = mask;
  this->filter = filter;
  this->cb = cb;
  this->cb_data = cb_data;
  this->registered = false;
  this->table = wsp->event_callbacks;
}

// ----------------------------------------------------------------------------
//
Event_Callback::~Event_Callback()
{
  if (registered)
    {
      remove_tk_event_handler(w, mask, ec_event_cb, this);
      remove_tk_event_handler(w, StructureNotifyMask, ec_destroy_cb, this);
      table->remove_callback(this);
    }

  w = NULL;
  cb = NULL;
}

// ----------------------------------------------------------------------------
// This is not part of the constructor because I need to be able to create
// table keys without registering an event handler.
//
void Event_Callback::register_with_tk()
{
  add_tk_event_handler(w, mask, ec_event_cb, this);
  add_tk_event_handler(w, StructureNotifyMask, ec_destroy_cb, this);
  registered = true;
  table->add_callback(this);
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static void ec_event_cb(CB_Data ecallback, XEvent *event)
{
  Event_Callback *ecb = (Event_Callback *) ecallback;

  if (ecb->filter == NULL || ecb->filter(ecb->wsp, event))
    ecb->cb(ecb->w, ecb->cb_data, event);
}
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static void ec_destroy_cb(ClientData ecb, XEvent *event)
{
  if (event->type == DestroyNotify)
    delete (Event_Callback *) ecb;
}
}

// ----------------------------------------------------------------------------
//
bool Event_Callback::operator==(const Event_Callback &ecb) const
{
  return (ecb.w == w && ecb.mask == mask && ecb.filter == filter &&
	  ecb.cb == cb && ecb.cb_data == cb_data);
}

// ----------------------------------------------------------------------------
//
unsigned long Event_Callback::hash() const
{
  return ((unsigned long) w ^ (unsigned long) mask ^ (unsigned long) filter ^
	  (unsigned long) cb ^ (unsigned long) cb_data);
}

// ----------------------------------------------------------------------------
//
Event_Callback_Table::Event_Callback_Table() : ecb_table(ecb_equal, ecb_hash)
{
}

// ----------------------------------------------------------------------------
//
Event_Callback_Table::~Event_Callback_Table()
{
  List cblist = ecb_table.keys();
  for (int ei = 0 ; ei < cblist.size() ; ++ei)
    delete (Event_Callback *) cblist[ei];
}

// ----------------------------------------------------------------------------
//
void Event_Callback_Table::add_callback(Event_Callback *ecb)
{
  TableData ecb2;
  if (ecb_table.find((TableKey) this, &ecb2))
    delete (Event_Callback *) ecb2;

  ecb_table.insert((TableKey) ecb, (TableData) ecb);
}

// ----------------------------------------------------------------------------
//
Event_Callback *Event_Callback_Table::find_callback(const Event_Callback &key)
{
  TableData ecb;
  if (ecb_table.find((TableKey) &key, &ecb))
    return (Event_Callback *) ecb;
  return NULL;
}

// ----------------------------------------------------------------------------
//
void Event_Callback_Table::remove_callback(Event_Callback *ecb)
{
  ecb_table.remove((TableKey) ecb);
}

// ----------------------------------------------------------------------------
// Used for hash tables.
//
bool Event_Callback_Table::ecb_equal(TableKey ecb1, TableKey ecb2)
  { return (*(Event_Callback *) ecb1 == *(Event_Callback *) ecb2); }
unsigned long Event_Callback_Table::ecb_hash(TableKey ecb)
  { return ((Event_Callback *) ecb)->hash(); }

// ----------------------------------------------------------------------------
//
static void add_event_handler(WinSysP *wsp, Widget w, unsigned long mask,
			      Filter_Proc filter, CB_Proc cb, CB_Data cb_data)
{
  Event_Callback *ecb = new Event_Callback(wsp, w, mask, filter, cb, cb_data);
  ecb->register_with_tk();
}

// ----------------------------------------------------------------------------
//
static void remove_event_handler(WinSysP *wsp, Widget w, unsigned long mask,
				 Filter_Proc filter,
				 CB_Proc cb, CB_Data cb_data)
{
  Event_Callback key(wsp, w, mask, filter, cb, cb_data);
  Event_Callback *ecb = wsp->event_callbacks->find_callback(key);
  if (ecb)
    delete ecb;
}

// ----------------------------------------------------------------------------
//
class Timer_Callback
{
public:
  Timer_Callback(Timer_Callback_List *list,
		 unsigned long msec, CB_Func, CB_Data);
  ~Timer_Callback();

  bool matches(unsigned long msec, CB_Func cb, CB_Data cb_data);
private:
  unsigned long msec;
  CB_Func cb;
  CB_Data cb_data;
  Tcl_TimerToken tcl_id;
  class Timer_Callback_List *list;

  friend void tc_timer_cb(CB_Data);
};

// ----------------------------------------------------------------------------
//
class Timer_Callback_List
{
public:
  ~Timer_Callback_List();

  void add_timer(Timer_Callback *tcb);
  void remove_timer(Timer_Callback *tcb);
  Timer_Callback *find(unsigned long msec, CB_Func, CB_Data);

private:
  List tcblist;
};

// ----------------------------------------------------------------------------
//
Timer_Callback::Timer_Callback(Timer_Callback_List *list,
			       unsigned long msec, CB_Func cb, CB_Data cb_data)
{
  this->msec = msec;
  this->cb = cb;
  this->cb_data = cb_data;
  this->tcl_id = Tcl_CreateTimerHandler((int) msec, tc_timer_cb, this);
  this->list = list;
  list->add_timer(this);
}

// ----------------------------------------------------------------------------
//
Timer_Callback::~Timer_Callback()
{
  list->remove_timer(this);
  Tcl_DeleteTimerHandler(tcl_id);
  this->cb = NULL;
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static void tc_timer_cb(CB_Data client_data)
{
  Timer_Callback *tcb = (Timer_Callback *) client_data;

  CB_Func cb = tcb->cb;
  CB_Data cb_data = tcb->cb_data;
  delete tcb;
  cb(cb_data);
}
}

// ----------------------------------------------------------------------------
//
bool Timer_Callback::matches(unsigned long msec, CB_Func cb, CB_Data cb_data)
{
  return (this->msec == msec && this->cb == cb && this->cb_data == cb_data);
}

// ----------------------------------------------------------------------------
//
Timer_Callback_List::~Timer_Callback_List()
{
  List copy = tcblist;
  for (int ti = 0 ; ti < copy.size() ; ++ti)
    delete (Timer_Callback *) copy[ti];
}

// ----------------------------------------------------------------------------
//
void Timer_Callback_List::add_timer(Timer_Callback *tcb)
  { tcblist.append(tcb); }
void Timer_Callback_List::remove_timer(Timer_Callback *tcb)
  { tcblist.erase(tcb); }

// ----------------------------------------------------------------------------
//
Timer_Callback *Timer_Callback_List::find(unsigned long msec,
					  CB_Func cb, CB_Data cb_data)
{
  for (int tci = 0 ; tci < tcblist.size() ; ++tci)
    {
      Timer_Callback *tcb = (Timer_Callback *) tcblist[tci];
      if (tcb->matches(msec, cb, cb_data))
	return tcb;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
void WinSys::add_timer(unsigned long msec, CB_Func cb, CB_Data cb_data)
{
  if (wsp->timer_callbacks->find(msec, cb, cb_data) == NULL)
    (void) new Timer_Callback(wsp->timer_callbacks, msec, cb, cb_data);
}

// ----------------------------------------------------------------------------
//
void WinSys::remove_timer(unsigned long msec, CB_Func cb, CB_Data cb_data)
{
  Timer_Callback *tcb = wsp->timer_callbacks->find(msec, cb, cb_data);
  if (tcb)
    delete tcb;
}

// ----------------------------------------------------------------------------
// Put a callback in the queue.  This is used to put a callback that will
// quit the session.  Since it will destory the WinSys object the callback
// invocation should not rely on WinSys data structures existing.
//
void WinSys::queue_callback(C_CB_Func cb, CB_Data cb_data)
{
  Tcl_CreateTimerHandler(0, cb, cb_data);
}

// ----------------------------------------------------------------------------
//
class Pointer_Pause
{
public:
  Pointer_Pause(WinSysP *, Widget w, CB_Proc cb, CB_Data cb_data);
  ~Pointer_Pause();

  bool matches(Widget w, CB_Proc cb, CB_Data cb_data);

private:
  WinSysP *wsp;
  Widget w;
  CB_Proc cb;
  CB_Data cb_data;
  bool timer_set, moved;

  friend void pp_motion_cb(CB_Data ppause, XEvent *);
  static void timer_cb(CB_Data ppause);
};

// ----------------------------------------------------------------------------
//
#define PAUSE_TIME 200		// milli-seconds

class Pointer_Pause_List
{
public:
  ~Pointer_Pause_List();

  void add_callback(Pointer_Pause *pcb);
  void remove_callback(Widget w, CB_Proc cb, CB_Data cb_data);
  void remove_callback(Pointer_Pause *pcb);

private:
  List pplist;
};

// ----------------------------------------------------------------------------
//
Pointer_Pause_List::~Pointer_Pause_List()
{
  List copy = pplist;
  for (int pi = 0 ; pi < copy.size() ; ++pi)
    delete (Pointer_Pause *) copy[pi];
}

// ----------------------------------------------------------------------------
//
void Pointer_Pause_List::add_callback(Pointer_Pause *pcb)
  { pplist.append(pcb); }
void Pointer_Pause_List::remove_callback(Pointer_Pause *pcb)
  { pplist.erase(pcb); }

// ----------------------------------------------------------------------------
//
void Pointer_Pause_List::remove_callback(Widget w, CB_Proc cb, CB_Data cb_data)
{
  for (int ppi = 0 ; ppi < pplist.size() ; ++ppi)
    {
      Pointer_Pause *pp = (Pointer_Pause *) pplist[ppi];
      if (pp->matches(w, cb, cb_data))
	{
	  delete pp;		// Careful! This modifies pplist
	  break;
	}
    }
}

// ----------------------------------------------------------------------------
//
Pointer_Pause::Pointer_Pause(WinSysP *wsp, Widget w,
			     CB_Proc cb, CB_Data cb_data)
{
  this->wsp = wsp;
  this->w = w;
  this->cb = cb;
  this->cb_data = cb_data;
  timer_set = false;
  moved = false;

  add_tk_event_handler(w, PointerMotionMask, pp_motion_cb, this);
  add_tk_event_handler(w, StructureNotifyMask, pp_destroy_cb, this);
  wsp->pause_callbacks->add_callback(this);
}

// ----------------------------------------------------------------------------
//
Pointer_Pause::~Pointer_Pause()
{
  wsp->pause_callbacks->remove_callback(this);
  remove_tk_event_handler(w, PointerMotionMask, pp_motion_cb, this);
  remove_tk_event_handler(w, StructureNotifyMask, pp_destroy_cb, this);
  if (timer_set)
    wsp->ws.remove_timer(PAUSE_TIME, timer_cb, this);
  w = NULL;
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static void pp_destroy_cb(ClientData pcb, XEvent *event)
{
  if (event->type == DestroyNotify)
    delete (Pointer_Pause *) pcb;
}
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static void pp_motion_cb(CB_Data ppause, XEvent *)
{
  Pointer_Pause *pp = (Pointer_Pause *) ppause;
  if (!pp->timer_set)
    {
      pp->wsp->ws.add_timer(PAUSE_TIME, Pointer_Pause::timer_cb, pp);
      pp->timer_set = true;
    }
  else
    pp->moved = true;
}
}

// ----------------------------------------------------------------------------
//
void Pointer_Pause::timer_cb(CB_Data ppause)
{
  Pointer_Pause *pp = (Pointer_Pause *) ppause;
  pp->timer_set = false;
  if (pp->moved)
    {
      pp->moved = false;
      pp->wsp->ws.add_timer(PAUSE_TIME, timer_cb, pp);
      pp->timer_set = true;
    }
  else
    pp->cb(pp->w, pp->cb_data, NULL);
}

// ----------------------------------------------------------------------------
//
bool Pointer_Pause::matches(Widget w, CB_Proc cb, CB_Data cb_data)
  { return (this->w == w && this->cb == cb && this->cb_data == cb_data); }

// ----------------------------------------------------------------------------
//
static void add_pointer_pause_callback(WinSysP *wsp, Widget w,
				       CB_Proc cb, CB_Data cb_data)
  { (void) new Pointer_Pause(wsp, w, cb, cb_data); }
static void remove_pointer_pause_callback(WinSysP *wsp, Widget w,
					  CB_Proc cb, CB_Data cb_data)
  { wsp->pause_callbacks->remove_callback(w, cb, cb_data); }

// ----------------------------------------------------------------------------
//
static void create_named_font(WinSysP *wsp, const Stringy &name)
{
  Stringy spec = wsp->ws.read_application_resource(main_widget(wsp), name,
						   "SparkyFont", "");
  if (!spec.is_empty())
    tcl_command(wsp, "font create " + name + " " + spec);
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_dialog(const Stringy &name, bool allow_destroy)
{
  Stringy path = unique_child_path(wsp, main_widget(), name);
  Stringy classname = capitalize(name);
  if (tcl_command(wsp, "toplevel %s -class %s", path, classname))
    {
      Widget shell = named_widget(path);
      tcl_command(wsp, "wm withdraw %s", path);
      if (!allow_destroy)
	set_delete_protocol(wsp, shell, unshow_cb, this);
      toplevel_geometry_option(wsp, shell);
      toplevel_title_option(wsp, shell);
      //      tcl_command(wsp, "wm group %s .", path);
      return shell;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
static Stringy capitalize(const Stringy &s)
{
  return (!s.is_empty() ?
	  formatted_string("%c%s", toupper(s[0]), s.cstring()+1) :
	  Stringy(""));
}

// ----------------------------------------------------------------------------
//
static void unshow_cb(Widget shell, CB_Data winsys, CB_Data)
{
  ((WinSys *) winsys)->unshow_dialog(shell);
}

// ----------------------------------------------------------------------------
//
static void toplevel_geometry_option(WinSysP *wsp, Widget t)
{
  Stringy geom = wsp->ws.read_application_resource(t, "geometry", "Geometry", "");
  if (! geom.is_empty())
    tcl_command(wsp, "wm geometry %s %s", path(t), geom);
}

// ----------------------------------------------------------------------------
//
static void toplevel_title_option(WinSysP *wsp, Widget t)
{
  Stringy title = wsp->ws.read_application_resource(t, "title", "Title", "");
  if (! title.is_empty())
    wsp->ws.set_dialog_title(t, title);
}

// ----------------------------------------------------------------------------
//
void WinSys::dialog_position(Widget w, int *x, int *y)
{
  *x = widget_x(w);
  *y = widget_y(w);
}
void WinSys::set_dialog_position(Widget w, int x, int y)
{
  move_onto_screen(wsp, w, &x, &y);
  Stringy geom = formatted_string("+%d+%d", x, y);
  tcl_command(wsp, "wm geometry %s %s", path(w), geom);
}

// ----------------------------------------------------------------------------
// If a dialog is positioned off screen there is no simple way for the user
// to find it an move it on screen.  This can happen if a Sparky session on
// a 1280x1024 screen (a unix machine) is later opened on an 1024x768 screen
// (a PC machine).
//
// This routine adjusts (x,y) if necessary so that at least part of the
// window will be on screen.
//
static void move_onto_screen(WinSysP *wsp, Widget w, int *x, int *y)
{
  int screen_w, screen_h;
  if (tcl_command(wsp, "winfo screenwidth %s", path(w)) &&
      command_result_integer(wsp, &screen_w) &&
      tcl_command(wsp, "winfo screenheight %s", path(w)) &&
      command_result_integer(wsp, &screen_h))
    {
      int widget_w = wsp->ws.requested_width(w);
      int widget_h = wsp->ws.requested_height(w);
      if (*x + widget_w <= 0)	*x = 0;
      else if (*x >= screen_w)	*x = screen_w - widget_w;
      if (*y + widget_h <= 0)	*y = 0;
      else if (*y >= screen_h)	*y = screen_h - widget_h;
    }
}

// ----------------------------------------------------------------------------
//
static void debug_message(const Stringy &msg)
{
  fputs(msg.cstring(), stderr);
}

// ----------------------------------------------------------------------------
//
static bool tcl_command(WinSysP *wsp, const Stringy &format,
			const Stringy &arg1)
{
  Stringy cmd = formatted_string(format.cstring(),
				 quoted_word(arg1).cstring());
  return tcl_command(wsp, cmd);
}

// ----------------------------------------------------------------------------
//
static bool tcl_command(WinSysP *wsp, const Stringy &format,
			const Stringy &arg1, const Stringy &arg2)
{
  Stringy cmd = formatted_string(format.cstring(),
				 quoted_word(arg1).cstring(),
				 quoted_word(arg2).cstring());
  return tcl_command(wsp, cmd);
}

// ----------------------------------------------------------------------------
//
static bool tcl_command(WinSysP *wsp, const Stringy &format,
			const Stringy &arg1, const Stringy &arg2,
			const Stringy &arg3)
{
  Stringy cmd = formatted_string(format.cstring(),
				 quoted_word(arg1).cstring(),
				 quoted_word(arg2).cstring(),
				 quoted_word(arg3).cstring());
  return tcl_command(wsp, cmd);
}

// ----------------------------------------------------------------------------
//
static bool tcl_command(WinSysP *wsp, const Stringy &format,
			const Stringy &arg1, const Stringy &arg2,
			const Stringy &arg3, const Stringy &arg4)
{
  Stringy cmd = formatted_string(format.cstring(),
				 quoted_word(arg1).cstring(),
				 quoted_word(arg2).cstring(),
				 quoted_word(arg3).cstring(),
				 quoted_word(arg4).cstring());
  return tcl_command(wsp, cmd);
}

// ----------------------------------------------------------------------------
//
static bool tcl_command(WinSysP *wsp, const Stringy &format,
			const Stringy &arg1,
			const Stringy &arg2, const Stringy &arg3,
			const Stringy &arg4, const Stringy &arg5)
{
  Stringy cmd = formatted_string(format.cstring(),
				 quoted_word(arg1).cstring(),
				 quoted_word(arg2).cstring(),
				 quoted_word(arg3).cstring(),
				 quoted_word(arg4).cstring(),
				 quoted_word(arg5).cstring());
  return tcl_command(wsp, cmd);
}

// ----------------------------------------------------------------------------
//
static bool tcl_command(WinSysP *wsp, const Stringy &cmd)
{
  bool error = (Tcl_GlobalEval(wsp->tcl_interp, (char *)cmd.cstring())
		== TCL_ERROR);

  if (wsp->debug_tcl)
    debug_message(cmd + "\n");
  if (error)
    report_tcl_error(wsp, cmd + "\n");

  return !error;
}

// ----------------------------------------------------------------------------
//
static Stringy command_result(WinSysP *wsp)
  { return Tcl_GetStringResult(wsp->tcl_interp); }

// ----------------------------------------------------------------------------
//
static bool command_result_integer(WinSysP *wsp, int *i)
{
  bool isint = (Tcl_GetIntFromObj(wsp->tcl_interp,
				  Tcl_GetObjResult(wsp->tcl_interp), i)
		== TCL_OK);
  if (!isint)
    report_tcl_error(wsp, "in command_result_integer(wsp, )");
  return isint;
}

// ----------------------------------------------------------------------------
//
static void report_tcl_error(WinSysP *wsp, const Stringy &msg)
{
  debug_message(msg);
  debug_message("Error: " + command_result(wsp) + "\n");
  Tcl_ResetResult(wsp->tcl_interp);
}

// ----------------------------------------------------------------------------
//
static bool command_result_double_pair(WinSysP *wsp, double *a, double *b)
{
  Tcl_Obj *ab = Tcl_GetObjResult(wsp->tcl_interp);
  Tcl_Obj *aobj, *bobj;
  if (Tcl_ListObjIndex(wsp->tcl_interp, ab, 0, &aobj)  == TCL_OK &&
      Tcl_GetDoubleFromObj(wsp->tcl_interp, aobj, a) == TCL_OK &&
      Tcl_ListObjIndex(wsp->tcl_interp, ab, 1, &bobj) == TCL_OK &&
      Tcl_GetDoubleFromObj(wsp->tcl_interp, bobj, b) == TCL_OK)
    return true;
  else
    {
      report_tcl_error(wsp, "in command_result_double_pair()\n");
      return false;
    }
}

// ----------------------------------------------------------------------------
//
static bool command_result_list(WinSysP *wsp, List *strings)
{
  Tcl_Obj *tlist = Tcl_GetObjResult(wsp->tcl_interp);
  int length;
  Tcl_Obj **elements;
  if (Tcl_ListObjGetElements(wsp->tcl_interp, tlist, &length, &elements)
      != TCL_OK)
    {
      report_tcl_error(wsp, "in command_result_list()\n");
      return false;
    }

  List slist;
  for (int k = 0 ; k < length ; ++k)
    slist.append(new Stringy(Tcl_GetStringFromObj(elements[k], NULL)));
  *strings = slist;

  return true;
}

// ----------------------------------------------------------------------------
//
static Stringy quoted_word(const Stringy &word)
{
  int argc = 1;
  char *argv[1];
  argv[0] = (char *) word.cstring();
  char *s = Tcl_Merge(argc, argv);
  Stringy quoted = s;
  Tcl_Free(s);
  return quoted;
}

// ----------------------------------------------------------------------------
//
static Stringy tcl_word_list(const List &words)
{
  int argc = words.size();
  char **argv = new char * [argc];
  for (int k = 0 ; k < words.size() ; ++k)
    argv[k] = (char *) ((Stringy *) words[k])->cstring();
  char *s = Tcl_Merge(argc, argv);
  Stringy list = s;
  Tcl_Free(s);
  delete [] argv;

  return list;
}

// ----------------------------------------------------------------------------
//
static Stringy path(Widget w)
  { return Tk_PathName(window(w)); }
Stringy WinSys::widget_name(Widget w)
  { return path(w); }
Widget WinSys::named_widget(const Stringy &path)
{
  Widget w = (Widget) Tk_NameToWindow(wsp->tcl_interp,
				      (char *)path.cstring(),
				      window(main_widget()));
  if (w == NULL)
    report_tcl_error(wsp, "in named_widget(" + path + ")\n");

  return w;
}
static bool widget_exists(WinSysP *wsp, const Stringy &path)
{
  bool exists = (Tk_NameToWindow(wsp->tcl_interp, (char *)path.cstring(),
				 window(main_widget(wsp))) != NULL);
  if (!exists)
    Tcl_ResetResult(wsp->tcl_interp);
  return exists;
}

// ----------------------------------------------------------------------------
//
static bool tcl_command_exists(WinSysP *wsp, const Stringy &name)
{
  Tcl_CmdInfo info;
  return Tcl_GetCommandInfo(wsp->tcl_interp, (char *) name.cstring(), &info);
}

// ----------------------------------------------------------------------------
//
static bool path_exists(WinSysP *wsp, const Stringy &path)
{
  return widget_exists(wsp, path) || tcl_command_exists(wsp, path);
}

// ----------------------------------------------------------------------------
//
static Stringy unique_child_path(WinSysP *wsp, Widget parent, const Stringy &name)
{
  Stringy n = name;
  if (n.length() > 0 && isupper(n[0]))
    {
      debug_message("Warning: Uppercase window name " + n + "\n");
      n = formatted_string("%c%s", tolower(n[0]), n.cstring() + 1);
    }

  Stringy path = child_path(parent, n);
  if (!path_exists(wsp, path))
    return path;
  return child_path(parent, unique_name(wsp, n));
}

// ----------------------------------------------------------------------------
//
static Stringy child_path(Widget parent, const Stringy &child)
{
  Stringy parent_path = path(parent);
  return (parent_path == "." ?
	  parent_path + child :
	  parent_path + "." + child);
}

// ----------------------------------------------------------------------------
//
static Stringy command_name(WinSysP *wsp, Widget parent, const Stringy &name)
{
  Stringy path = child_path(parent, name);
  if (!tcl_command_exists(wsp, path))
    return path;
  return child_path(parent, unique_name(wsp, name));
}

// ----------------------------------------------------------------------------
//
static Stringy unique_name(WinSysP *wsp, const Stringy &name)
{
  Stringy suffix = wsp->unique_name_count;
  wsp->unique_name_count += 1;
  return name + suffix;
}

// ----------------------------------------------------------------------------
//
static Stringy get_option(WinSysP *wsp, Widget w, const Stringy &option)
{
  if (tcl_command(wsp, "%s cget %s", path(w), option))
    return command_result(wsp);
  return "";
}

// ----------------------------------------------------------------------------
//
class Command_Callback
{
public:
  Command_Callback(WinSysP *wsp, Widget w, CB_Proc cb, CB_Data cb_data,
		   bool registered = true);
  virtual ~Command_Callback();

  virtual int process_args(int argc, const char *argv[]);

  Stringy name();
  void rename(const Stringy &);

  bool operator==(const Command_Callback &ccb) const;
  unsigned long hash() const;

protected:
  WinSysP *wsp;
  Widget w;
  Stringy cb_name;

private:
  bool registered;
  CB_Proc cb;
  CB_Data cb_data;

  friend int tk_cb(ClientData self, Tcl_Interp *, int argc, const char *argv[]);
};

// ----------------------------------------------------------------------------
//
Command_Callback::Command_Callback(WinSysP *wsp, Widget w,
				   CB_Proc cb, CB_Data cb_data,
				   bool registered)
{
  this->wsp = wsp;
  this->w = w;
  this->cb = cb;
  this->cb_data = cb_data;
  this->registered = registered;

  if (registered)
    {
      this->cb_name = command_name(wsp, w, "callback");
      add_tcl_command(wsp, cb_name, tk_cb, this);
      add_tk_event_handler(w, StructureNotifyMask, cc_destroy_cb, this);
    }
}

// ----------------------------------------------------------------------------
// Destructor called when owning widget is deleted.
//
Command_Callback::~Command_Callback()
{
  if (registered)
    {
      remove_tk_event_handler(w, StructureNotifyMask, cc_destroy_cb, this);
      remove_tcl_command(wsp, cb_name);
    }

  w = NULL;
  cb = NULL;
}

// ----------------------------------------------------------------------------
//
Stringy Command_Callback::name()
  { return cb_name; }

// ----------------------------------------------------------------------------
//
void Command_Callback::rename(const Stringy &name)
{
  if (name != cb_name)
    {
      tcl_command(wsp, "rename %s %s", cb_name, name);
      this->cb_name = name;
    }
}

// ----------------------------------------------------------------------------
//
int Command_Callback::process_args(int, const char *[])
{
  return TCL_OK;
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static int tk_cb(ClientData self, Tcl_Interp *, int argc, const char *argv[])
{
  Command_Callback *cc = (Command_Callback *) self;
  int status = ((Command_Callback *) self)->process_args(argc, argv);
  if (cc->cb)
    cc->cb(cc->w, cc->cb_data, cc);
  return status;
}
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static void cc_destroy_cb(ClientData ccb, XEvent *event)
{
  if (event->type == DestroyNotify)
    delete (Command_Callback *) ccb;
}
}

// ----------------------------------------------------------------------------
//
bool Command_Callback::operator==(const Command_Callback &ccb) const
{
  return (ccb.w == w && ccb.cb == cb && ccb.cb_data == cb_data);
}

// ----------------------------------------------------------------------------
//
unsigned long Command_Callback::hash() const
{
  return ((unsigned long) w ^ (unsigned long) cb ^ (unsigned long) cb_data);
}

// ----------------------------------------------------------------------------
//
class Variable_Callback : public Command_Callback
{
public:
  Variable_Callback(WinSysP *wsp, const Stringy &var,
		    Widget w, CB_Proc cb, CB_Data cb_data);
  ~Variable_Callback();
private:
  Stringy varname;
  class Variable_Callback_Manager *manager;
  friend char *variable_set_cb(ClientData, Tcl_Interp *,
			       const char *, const char *, int);
};

// ----------------------------------------------------------------------------
//
class Variable_Callback_Manager
{
public:
  Variable_Callback_Manager();
  ~Variable_Callback_Manager();

  void add_callback(WinSysP *, const Stringy &var, Widget, CB_Proc, CB_Data);
  void remove_callback(WinSysP *, Widget w, CB_Proc cb, CB_Data cb_data);
  void remove_callback(Variable_Callback *vcb);

private:
  Table ccb_table;

  static bool ccb_equal(TableKey ccb1, TableKey ccb2);
  static unsigned long ccb_hash(TableKey ccb);
};

// ----------------------------------------------------------------------------
//
Variable_Callback_Manager::Variable_Callback_Manager() :
  ccb_table(ccb_equal, ccb_hash)
{
}

// ----------------------------------------------------------------------------
//
Variable_Callback_Manager::~Variable_Callback_Manager()
{
  List cblist = ccb_table.keys();
  for (int vi = 0 ; vi < cblist.size() ; ++vi)
    remove_callback((Variable_Callback *) cblist[vi]);
}

// ----------------------------------------------------------------------------
//
void Variable_Callback_Manager::add_callback(WinSysP *wsp, const Stringy &var,
					     Widget w, CB_Proc cb, CB_Data cb_data)
{
  Variable_Callback *vcb = new Variable_Callback(wsp, var, w, cb, cb_data);
  Command_Callback *ccb = (Command_Callback *) vcb;
  ccb_table.insert((TableKey) ccb, (TableData) ccb);
}

// ----------------------------------------------------------------------------
// Nothing is done if a command for (w, cb, cb_data) does not exist.
// Note that if the same command is created twice it is deleted by the
// first attempt to delete it.  That's dangerous.
//
void Variable_Callback_Manager::remove_callback(WinSysP *wsp, Widget w,
						CB_Proc cb, CB_Data cb_data)
{
  Command_Callback key(wsp, w, cb, cb_data, false);
  TableData vcb;
  if (ccb_table.find((TableKey) &key, &vcb))
    delete (Variable_Callback *) vcb;
}

// ----------------------------------------------------------------------------
//
void Variable_Callback_Manager::remove_callback(Variable_Callback *vcb)
{
  ccb_table.remove((TableKey) (Command_Callback *) vcb);
}

// ----------------------------------------------------------------------------
// Used for hash tables.
//
bool Variable_Callback_Manager::ccb_equal(TableKey ccb1, TableKey ccb2)
  { return (*(Command_Callback *) ccb1 == *(Command_Callback *) ccb2); }
unsigned long Variable_Callback_Manager::ccb_hash(TableKey ccb)
  { return ((Command_Callback *) ccb)->hash(); }

// ----------------------------------------------------------------------------
//
Variable_Callback::Variable_Callback(WinSysP *wsp, const Stringy &var,
				     Widget w, CB_Proc cb, CB_Data cb_data)
  : Command_Callback(wsp, w, cb, cb_data)
{
  this->varname = var;
  this->manager = wsp->variable_callbacks;

  if (Tcl_TraceVar(wsp->tcl_interp, (char *) var.cstring(),
		   TCL_GLOBAL_ONLY | TCL_TRACE_WRITES,
		   variable_set_cb, this) == TCL_ERROR)
    report_tcl_error(wsp, "in variable_changed_callback(" + var + ")\n");
}

// ----------------------------------------------------------------------------
//
Variable_Callback::~Variable_Callback()
{
  this->manager->remove_callback(this);
  Tcl_UntraceVar(wsp->tcl_interp, (char *) varname.cstring(),
		 TCL_GLOBAL_ONLY | TCL_TRACE_WRITES, variable_set_cb, this);
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static char *variable_set_cb(ClientData cmdcb, Tcl_Interp *,
			     const char *, const char *, int)
{
  Variable_Callback *vcb = (Variable_Callback *) cmdcb;
  tcl_command(vcb->wsp, "%s", vcb->name());
  return NULL;
}
}

// ----------------------------------------------------------------------------
//
static void variable_changed_callback(WinSysP *wsp, Widget w, CB_Proc cb, CB_Data cb_data)
  { wsp->variable_callbacks->add_callback(wsp, widget_variable(w), w,
					  cb, cb_data); }
static void remove_variable_changed_callback(WinSysP *wsp, Widget w,
					     CB_Proc cb, CB_Data cb_data)
  { wsp->variable_callbacks->remove_callback(wsp, w, cb, cb_data); }

// ----------------------------------------------------------------------------
//
static Stringy widget_variable(Widget w)
  { return formatted_string("%s.variable", path(w).cstring()); }

// ----------------------------------------------------------------------------
//
static Stringy get_variable(WinSysP *wsp, Widget w)
{
  Stringy vname = widget_variable(w);
  return Tcl_GetVar(wsp->tcl_interp, (char *) vname.cstring(),
		    TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG);
}

// ----------------------------------------------------------------------------
//
static void set_variable(WinSysP *wsp, Widget w, const Stringy &value)
{
  Stringy vname = widget_variable(w);
  if (Tcl_SetVar(wsp->tcl_interp,
		 (char *) vname.cstring(), (char *) value.cstring(),
		 TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL)
    report_tcl_error(wsp, "in set_variable(" + vname + ", " + value + ")\n");
}

// ----------------------------------------------------------------------------
//
void WinSys::set_dialog_title(Widget shell, const Stringy &title)
{
  tcl_command(wsp, "wm title %s %s", path(shell), title);
}

// ----------------------------------------------------------------------------
// Tk top levels will resize automatically in response to changes in child
// widget sizes unless either a "wm geometry" command is executed or the user
// manually adjusts the window size.  This is documented in the wm manual
// page.
//
void WinSys::resize_dialog(Widget shell, int width, int height)
{
  Stringy geom = formatted_string("%dx%d", width, height);
  tcl_command(wsp, "wm geometry %s %s", path(shell), geom);
}

// ----------------------------------------------------------------------------
//
void WinSys::allow_dialog_resize(Widget w, bool allow)
{
  if (allow)
    tcl_command(wsp, "wm geometry %s {}", path(w));
  else
    {
      tcl_command(wsp, "update idletasks");	// finish geometry calculation
      int width = Tk_ReqWidth(w), height = Tk_ReqHeight(w);
      Stringy geom = formatted_string("%dx%d", width, height);
      tcl_command(wsp, "wm geometry %s %s", path(w), geom);
    }
}

// ----------------------------------------------------------------------------
//
Stringy WinSys::label_text(Widget label)
  { return get_option(wsp, label, "-text"); }
void WinSys::set_label_text(Widget label, const Stringy &text)
{
  tcl_command(wsp, "%s configure -text %s", path(label), text);
}

// ----------------------------------------------------------------------------
//
class Scroll_Callback: public Command_Callback
{
public:
  Scroll_Callback(WinSysP *, Widget w, CB_Proc cb, CB_Data cb_data);
  ~Scroll_Callback();

  void set_step_size(double step);

private:
  double step_size;
  Scroll_Callback_Table *table;

  virtual int process_args(int argc, const char *argv[]);
  void scroll_by_amount(double delta, double a, double b);
};

// ----------------------------------------------------------------------------
//
class Scroll_Callback_Table
{
public:
  Scroll_Callback_Table();
  ~Scroll_Callback_Table();

  void add_callback(const Stringy *name, Scroll_Callback *scb);
  void remove_callback(const Stringy &cb_name);
  Scroll_Callback *find_callback(const Stringy &cb_name);

private:
  Table scb_name_table;
};

// ----------------------------------------------------------------------------
//
Scroll_Callback_Table::Scroll_Callback_Table() :
  scb_name_table(equal_strings, hash_string)
{
}

// ----------------------------------------------------------------------------
//
Scroll_Callback_Table::~Scroll_Callback_Table()
{
  List cblist = scb_name_table.values();
  for (int si = 0 ; si < cblist.size() ; ++si)
    delete (Scroll_Callback *) cblist[si];
}

// ----------------------------------------------------------------------------
//
void Scroll_Callback_Table::add_callback(const Stringy *name,
					 Scroll_Callback *scb)
{
  scb_name_table.insert((TableKey) name, (TableData) scb);
}

// ----------------------------------------------------------------------------
//
void Scroll_Callback_Table::remove_callback(const Stringy &cb_name)
{
  scb_name_table.remove((TableKey) &cb_name);
}

// ----------------------------------------------------------------------------
//
Scroll_Callback *Scroll_Callback_Table::find_callback(const Stringy &cb_name)
{
  TableData scb;
  if (scb_name_table.find((TableKey) &cb_name, &scb))
    return (Scroll_Callback *) scb;
  return NULL;
} 

// ----------------------------------------------------------------------------
//
Scroll_Callback::Scroll_Callback(WinSysP *wsp, Widget w,
				 CB_Proc cb, CB_Data cb_data)
  : Command_Callback(wsp, w, cb, cb_data)
{
  this->table = wsp->scroll_callbacks;
  this->step_size = 1.0 / 1024;
  table->add_callback(&this->cb_name, this);
}

// ----------------------------------------------------------------------------
//
Scroll_Callback::~Scroll_Callback()
{
  table->remove_callback(cb_name);
}

// ----------------------------------------------------------------------------
// Argv example: widget-path view scroll 1 unit
//
int Scroll_Callback::process_args(int argc, const char *argv[])
{
  double a, b;
  if (!scrollbar_position(wsp, w, &a, &b))
    return TCL_ERROR;

  int move_units;
  double move_to;
  switch (Tk_GetScrollInfo(wsp->tcl_interp, argc, argv, &move_to, &move_units))
    {
    case TK_SCROLL_MOVETO:
      scroll_by_amount(move_to - a, a, b);
      break;
    case TK_SCROLL_PAGES:
      scroll_by_amount(move_units * (b - a), a, b);
      break;
    case TK_SCROLL_UNITS:
      scroll_by_amount(move_units * this->step_size, a, b);
      break;
    case TK_SCROLL_ERROR:
      return TCL_ERROR;
    }

  return TCL_OK;
}

// ----------------------------------------------------------------------------
//
void Scroll_Callback::scroll_by_amount(double delta, double a, double b)
{
  if (a + delta < 0)		delta = -a;
  else if (b + delta > 1)	delta = 1 - b;
  double start = a + delta;
  double end = b + delta;
  tcl_command(wsp, "%s set %s %s", path(w),
	      formatted_string("%f", start),
	      formatted_string("%f", end));
}

// ----------------------------------------------------------------------------
//
void Scroll_Callback::set_step_size(double step_size)
{
  this->step_size = step_size;
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_scrollbar(Widget parent, const Stringy &name, bool horz,
				CB_Proc cb, CB_Data cb_data)
{
  Stringy path = unique_child_path(wsp, parent, name);
  if (tcl_command(wsp, "scrollbar %s -orient %s",
		   path, (horz ? "horizontal" : "vertical")))
    {
      Widget w = named_widget(path);
      Scroll_Callback *scb = new Scroll_Callback(wsp, w, cb, cb_data);
      Stringy cmd = scb->name() + " view";
      tcl_command(wsp, "%s configure -command %s", path, cmd);
      return w;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
// Return the position of the front edge of the scrollbar.
//
double WinSys::scrollbar_position(Widget sbar)
{
  double a, b;
  if (::scrollbar_position(wsp, sbar, &a, &b))
    return a;
  return 0;
}

// ----------------------------------------------------------------------------
// Return the position of the scrollbar.
//
static bool scrollbar_position(WinSysP *wsp, Widget sbar, double *a, double *b)
{
  return (tcl_command(wsp, "%s get", path(sbar)) &&
	  command_result_double_pair(wsp, a, b));
}

// ----------------------------------------------------------------------------
// Example correspondence: step = 1
//
//        v1 = 1      v2 = 3
//           |-----|-----|
//  s1 = 0                                             s2 = 1023
//     |-----|-----|-----|--------------------------------|
//     0     1     2     3                              1023
//
//           |-----------|  slider
//     start = 1
//
// The scrollbar value changed callback is not invoked.
// (It is only called for user interactive manipulation of the scrollbar.)
//
void WinSys::set_scrollbar(Widget sbar, double v1, double v2,
			   double s1, double s2, double step)
{
  double start = (v1 - s1) / (s2 - s1);
  double end = (v2 - s1) / (s2 - s1);
  tcl_command(wsp, "%s set %s %s", path(sbar),
	      formatted_string("%f", start), formatted_string("%f", end));

  // Set the step size.
  Stringy cb_name = first_token(get_option(wsp, sbar, "-command"), NULL);
  Scroll_Callback *scb = wsp->scroll_callbacks->find_callback(cb_name);
  scb->set_step_size(fabs(step / (s2 - s1)));
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_form(Widget parent, const Stringy &name)
{
  Stringy path = unique_child_path(wsp, parent, name);
  if (tcl_command(wsp, "frame %s", path))
    return named_widget(path);
  return NULL;
}

// ----------------------------------------------------------------------------
//
void WinSys::column_attachments(Widget stretch_me, ...)
{
  va_list args;

  va_start(args, stretch_me);
  int count = 0;
  for (Widget w = va_arg(args, Widget) ;
       w != END_OF_WIDGETS ; w = va_arg(args, Widget))
    count += 1;

  Widget *widgets = new Widget [count+1];
  va_start(args, stretch_me);
  count = 0;
  for (Widget w = va_arg(args, Widget) ;
       w != END_OF_WIDGETS ; w = va_arg(args, Widget))
    widgets[count++] = w;
  va_end(args);
  widgets[count] = NULL;

  stack_widgets(wsp, true, widgets, stretch_me);
  delete [] widgets;
}

// ----------------------------------------------------------------------------
//
void WinSys::row_attachments(Widget stretch_me, ...)
{
  va_list args;

  va_start(args, stretch_me);
  int count = 0;
  for (Widget w = va_arg(args, Widget) ;
       w != END_OF_WIDGETS ; w = va_arg(args, Widget))
    count += 1;

  Widget *widgets = new Widget [count+1];
  va_start(args, stretch_me);
  count = 0;
  for (Widget w = va_arg(args, Widget) ;
       w != END_OF_WIDGETS ; w = va_arg(args, Widget))
    widgets[count++] = w;
  va_end(args);
  widgets[count] = NULL;

  stack_widgets(wsp, false, widgets, stretch_me);
  delete [] widgets;
}

// ----------------------------------------------------------------------------
//
static void stack_widgets(WinSysP *wsp, bool vertical, Widget widgets[], Widget stretch_me)
{
  for (int p = 0 ; widgets[p] != NULL ; ++p)
    tcl_command(wsp, (vertical ?
		 "grid configure %s -row %s -column 0 -sticky nws" :
		 "grid configure %s -row 0 -column %s -sticky new"),
		path(widgets[p]), Stringy(p));

  if (stretch_me)
    {
      tcl_command(wsp, "grid configure %s -sticky news", path(stretch_me));
      Widget master = parent_widget(stretch_me);
      tcl_command(wsp, "grid %s %s 0 -weight 1",
		  (vertical ? "columnconfigure" : "rowconfigure"),
		  path(master));
      for (int p = 0 ; widgets[p] != NULL ; ++p)
	if (widgets[p] == stretch_me)
	  tcl_command(wsp, "grid %s %s %s -weight 1",
		      (vertical ? "rowconfigure" : "columnconfigure"),
		      path(master), Stringy(p));
    }
}

// ----------------------------------------------------------------------------
//
void WinSys::detach_top_and_bottom(Widget w)
{
  tcl_command(wsp, "grid configure %s -sticky ew", path(w));
}

// ----------------------------------------------------------------------------
//
void WinSys::plot_form_layout(Widget form, Widget center,
			      const List &horz_edge_widgets,
			      const List &vert_edge_widgets)
{
  clear_grid(wsp, form);
  tcl_command(wsp, "grid rowconfigure %s 0 -weight 1", path(form));
  tcl_command(wsp, "grid columnconfigure %s 0 -weight 1", path(form));

  tcl_command(wsp, "grid configure %s -row 0 -column 0 -sticky news", path(center));

  int width = Tk_ReqWidth(center);
  int cnt = horz_edge_widgets.size();
  for (int wi = 0 ; wi < cnt ; ++wi)
    {
      Widget w = (Widget) horz_edge_widgets[wi];
      set_widget_width(w, width);
      tcl_command(wsp, "grid configure %s -row %s -column 0 -sticky news",
		  path(w), Stringy(cnt+1-wi));
    }

  int height = Tk_ReqHeight(center);
  cnt = vert_edge_widgets.size();
  for (int wi = 0 ; wi < cnt ; ++wi)
    {
      Widget w = (Widget) vert_edge_widgets[wi];
      set_widget_height(w, height);
      tcl_command(wsp, "grid configure %s -row 0 -column %s -sticky news",
		  path(w), Stringy(cnt+1-wi));
    }
}

// ----------------------------------------------------------------------------
//
static void clear_grid(WinSysP *wsp, Widget w)
{
  if (tcl_command(wsp, "grid slaves %s", path(w)))
    {
      Stringy slaves = command_result(wsp);
      if (! slaves.is_empty())
	tcl_command(wsp, "grid forget " + slaves);
    }
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_menu_bar(Widget parent, const Stringy &name)
{
  return create_form(parent, name);
}

// ----------------------------------------------------------------------------
// Return the menu pane.
//
Widget WinSys::add_menu(Widget menubar, const Stringy &button,
			const Stringy &pane)
{
  Widget menu = create_menu(menubar, button, pane);
  tcl_command(wsp, "pack %s -side left", path(menu));
  return menu_pane(menu);
}

// ----------------------------------------------------------------------------
// Menu attached to a button.
//
Widget WinSys::create_menu(Widget parent, const Stringy &button,
			   const Stringy &pane)
{
  Stringy bpath = unique_child_path(wsp, parent, button);
  if (tcl_command(wsp, "menubutton %s", bpath))
    {
      Widget b = named_widget(bpath);
      Widget p = create_menu_pane(b, pane);
      if (tcl_command(wsp, "%s configure -menu %s", bpath, path(p)))
	return b;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
Widget WinSys::menu_pane(Widget menu)
  { return named_widget(get_option(wsp, menu, "-menu")); }

// ----------------------------------------------------------------------------
//
class Menu_Callback: public Command_Callback
{
public:
  Menu_Callback(WinSysP *wsp, Widget pane, CB_Proc cb, CB_Data cb_data)
    : Command_Callback(wsp, pane, cb, cb_data) {}
  virtual ~Menu_Callback() {};

  Stringy command_cb(const Stringy &name, const Stringy &text);
  virtual int process_args(int argc, const char *argv[]);
  WinSysP *winsysp() { return wsp; }
  Stringy button_name, button_text;
};

// ----------------------------------------------------------------------------
//
Stringy Menu_Callback::command_cb(const Stringy &name, const Stringy &text)
{
  return this->name() + " " + quoted_word(name) + " " + quoted_word(text);
}

// ----------------------------------------------------------------------------
//
int Menu_Callback::process_args(int argc, const char *argv[])
{
  if (argc != 3)
    return TCL_ERROR;
  button_name = argv[1];
  button_text = argv[2];
  return TCL_OK;
}

// ----------------------------------------------------------------------------
//
Stringy WinSys::selected_menu_button(Widget, CB_Data call_data)
{
  Command_Callback *cc = (Command_Callback *) call_data;
  return ((Menu_Callback *) cc)->button_name;
}

// ----------------------------------------------------------------------------
//
static Stringy button_label_option(WinSysP *wsp, Widget pane, const Stringy &name)
{
  Stringy label = wsp->ws.read_application_resource(pane, name + "Label", "Text", "");
  if (! label.is_empty())
    return label;
  return name;
}

// ----------------------------------------------------------------------------
//
void WinSys::add_menu_button(Widget pane, const Stringy &name,
			     CB_Proc cb, CB_Data cb_data)
{
  Stringy label = button_label_option(wsp, pane, name);
  Menu_Callback *mc = new Menu_Callback(wsp, pane, cb, cb_data);
  tcl_command(wsp, "%s add command -label %s -command %s",
	      path(pane), label, mc->command_cb(name, label));
}

// ----------------------------------------------------------------------------
//
void WinSys::add_menu_button(Widget pane, const Stringy &name,
			     const Stringy &accel, CB_Proc cb, CB_Data cb_data)
{
  Stringy label = button_label_option(wsp, pane, name);
  Menu_Callback *mc = new Menu_Callback(wsp, pane, cb, cb_data);
  tcl_command(wsp, "%s add command -label %s -accelerator %s -command %s",
	      path(pane), label, accel, mc->command_cb(name, label));
}

// ----------------------------------------------------------------------------
//
void WinSys::add_menu_button_at_top(Widget pane, const Stringy &name,
				    CB_Proc cb, CB_Data cb_data)
{
  Stringy label = button_label_option(wsp, pane, name);
  Command_Callback *cc = new Command_Callback(wsp, pane, cb, cb_data);
  Stringy cmd = cc->name() + " " + quoted_word(name);
  tcl_command(wsp, "%s insert 0 command -label %s -command %s",
	      path(pane), label, cmd);
}

// ----------------------------------------------------------------------------
//
void WinSys::delete_menu_button(Widget pane, const Stringy &button)
{
  tcl_command(wsp, "%s delete %s", path(pane), button);
}

// ----------------------------------------------------------------------------
//
void WinSys::delete_menu_buttons(Widget pane)
{
  tcl_command(wsp, "%s delete 0 end", path(pane));
}

// ----------------------------------------------------------------------------
//
void WinSys::menu_separator(Widget pane, const Stringy &)
{
  tcl_command(wsp, "%s add separator", path(pane));
}

// ----------------------------------------------------------------------------
//
void WinSys::menu_label(Widget pane, const Stringy &name)
{
  Stringy label = button_label_option(wsp, pane, name);
  Stringy bg = get_option(wsp, pane, "-background");
  tcl_command(wsp, "%s add command -activebackground %s -label %s",
	      path(pane), bg, label);
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_menu_pane(Widget parent, const Stringy &name)
{
  Stringy path = unique_child_path(wsp, parent, name);
  if (tcl_command(wsp, "menu %s", path))
    return named_widget(path);

  return NULL;
}

// ----------------------------------------------------------------------------
//
void WinSys::popup_menu(Widget popup, CB_Data event)
{
  XEvent *e = (XEvent *) event;
  if (e->type == ButtonPress && e->xbutton.button == Button3 && popup)
    tcl_command(wsp, "tk_popup %s %s %s", path(popup),
		Stringy(e->xbutton.x_root), Stringy(e->xbutton.y_root));
}

// ----------------------------------------------------------------------------
//
Widget WinSys::cascade_pane(Widget pane, const Stringy &button_name)
{
  Stringy mpath = unique_child_path(wsp, pane, "pane");
  Stringy label = button_label_option(wsp, pane, button_name);
  if (tcl_command(wsp, "menu %s", mpath) &&
      tcl_command(wsp, "%s add cascade -label %s -menu %s",
		  path(pane), label, mpath))
    return named_widget(mpath);
  return NULL;
}

// ----------------------------------------------------------------------------
//
void WinSys::delete_cascade(Widget pane, const Stringy &button_name)
{
  Stringy label = button_label_option(wsp, pane, button_name);
  if (tcl_command(wsp, "%s entrycget %s -menu", path(pane), label))
    {
      Stringy pane_path = command_result(wsp);
      Widget p = named_widget(pane_path);
      delete_widget(p);
    }
  delete_menu_button(pane, button_name);
}

// ----------------------------------------------------------------------------
//
bool WinSys::is_pane_empty(Widget pane)
{
  return menu_entry_count(wsp, pane) == (has_tearoff_entry(wsp, pane) ? 1 : 0);
}

// ----------------------------------------------------------------------------
//
static int menu_entry_count(WinSysP *wsp, Widget pane)
{
  int last;
  if (tcl_command(wsp, "%s index last", path(pane)) &&
      command_result(wsp) != "none" &&
      command_result_integer(wsp, &last))
    return last + 1;
  return 0;
}

// ----------------------------------------------------------------------------
//
static bool has_tearoff_entry(WinSysP *wsp, Widget pane)
{
  int tearoff;
  if (tcl_command(wsp, "%s cget -tearoff", path(pane)) &&
      command_result_integer(wsp, &tearoff))
    return tearoff == 1;
  return false;
}

// ----------------------------------------------------------------------------
// Appears I have to scan all menu entries to check if a label exists.
//
Widget WinSys::find_cascade_pane(Widget parent, const Stringy &button_name)
{
  int entries = menu_entry_count(wsp, parent);
  Stringy label = button_label_option(wsp, parent, button_name);
  Stringy p = path(parent);
  for (int k = 0 ; k < entries ; ++k)
    {
      Stringy index = k;
      if (tcl_command(wsp, "%s type %s", p, index) &&
	  command_result(wsp) == "cascade" &&
	  tcl_command(wsp, "%s entrycget %s -label", p, index) &&
	  command_result(wsp) == label &&
	  tcl_command(wsp, "%s entrycget %s -menu", p, index))
	return named_widget(command_result(wsp));
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
void WinSys::square_up_menu(Widget m)
{
  Widget pane = menu_pane(m);
  int last;
  if (tcl_command(wsp, "%s index end", path(pane)) &&
      command_result_integer(wsp, &last))
    {
      int rows = 5;			// at least 5 rows
      while (last > rows * rows)
	rows += 1;
      for (int k = rows ; k <= last ; k += rows)
	tcl_command(wsp, "%s entryconfigure %s -columnbreak 1",
		    path(pane), Stringy(k));
    }
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_separator(Widget parent, const Stringy &name)
{
  Stringy path = unique_child_path(wsp, parent, name);
  if (tcl_command(wsp, "frame %s -height 8 -borderwidth 2 -relief groove", path))
    return named_widget(path);
  return NULL;
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_scrolled_list(Widget parent, const Stringy &name,
				    Widget *list)
{
  Widget f = create_form(parent, name);
  Stringy lpath = unique_child_path(wsp, f, "list");
  Stringy spath = unique_child_path(wsp, f, "scrollbar");
  Widget h = create_label(f, "listHeading");

  if (tcl_command(wsp, "listbox %s -exportselection 0", lpath) &&
      tcl_command(wsp, "scrollbar %s -command %s", spath, lpath + " yview") &&
      tcl_command(wsp, "%s configure -yscrollcommand %s", lpath, spath + " set") &&
      tcl_command(wsp, "%s configure -anchor nw", path(h)) &&
      tcl_command(wsp, "grid rowconfigure %s 1 -weight 1", path(f)) &&
      tcl_command(wsp, "grid columnconfigure %s 0 -weight 1", path(f)) &&
      tcl_command(wsp, "grid %s -row 0 -column 0 -sticky news", path(h)) &&
      tcl_command(wsp, "grid %s -row 1 -column 0 -sticky news", lpath) &&
      tcl_command(wsp, "grid %s -row 1 -column 1 -sticky news", spath))
    {
      if (list)
	*list = named_widget(lpath);
      return f;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
Widget WinSys::scrolled_list_heading(Widget slist)
  { return child_widget(slist, "listHeading"); }

// ----------------------------------------------------------------------------
//
void WinSys::add_list_selection_callback(Widget list,
					 CB_Proc cb, CB_Data cb_data)
  { add_event_handler(wsp, list, ButtonReleaseMask, button_1_filter, cb, cb_data); }
void WinSys::remove_list_selection_callback(Widget list,
					    CB_Proc cb, CB_Data cb_data)
  { remove_event_handler(wsp, list, ButtonReleaseMask,
			 button_1_filter, cb, cb_data); }

// ----------------------------------------------------------------------------
// Middle mouse button has same effect as double click.
//
void WinSys::add_list_double_click_callback(Widget list,
					    CB_Proc cb, CB_Data cb_data)
{
  add_event_handler(wsp, list, ButtonReleaseMask,
		    double_button_1_filter, cb, cb_data);
  add_event_handler(wsp, list, ButtonPressMask,
		    button_2_filter, list_select_cb, wsp);
  add_event_handler(wsp, list, ButtonReleaseMask, button_2_filter, cb, cb_data);
}

// ----------------------------------------------------------------------------
//
void WinSys::remove_list_double_click_callback(Widget list, CB_Proc cb,
					       CB_Data cb_data)
{
  remove_event_handler(wsp, list, ButtonReleaseMask,
		       double_button_1_filter, cb, cb_data);
  remove_event_handler(wsp, list, ButtonPressMask,
		       button_2_filter, list_select_cb, wsp);
  remove_event_handler(wsp, list, ButtonReleaseMask, button_2_filter, cb, cb_data);
}

// ----------------------------------------------------------------------------
//
static void list_select_cb(Widget list, CB_Data winsysp, CB_Data event)
{
  XEvent *e = (XEvent *) event;
  if (e->type == ButtonPress || e->type == ButtonRelease)
    {
      WinSysP *wsp = (WinSysP *) winsysp;
      tcl_command(wsp, "%s selection clear 0 end", path(list));
      tcl_command(wsp, "%s selection set @%s,%s",
		  path(list), Stringy(e->xbutton.x), Stringy (e->xbutton.y));
    }
}

// ----------------------------------------------------------------------------
//
void WinSys::multiple_selection_list(Widget list)
  { tcl_command(wsp, "%s configure -selectmode multiple", path(list)); }
void WinSys::extended_selection_list(Widget list)
  { tcl_command(wsp, "%s configure -selectmode extended", path(list)); }

// ----------------------------------------------------------------------------
//
void WinSys::add_list_item(Widget list, const Stringy &text)
{
  tcl_command(wsp, "%s insert end %s", path(list), text);
}

// ----------------------------------------------------------------------------
//
void WinSys::replace_list_item(Widget list, int pos, const Stringy &text)
{
  tcl_command(wsp, "%s insert %s %s", path(list), Stringy(pos), text);
  tcl_command(wsp, "%s delete %s", path(list), Stringy(pos+1));
}

// ----------------------------------------------------------------------------
//
void WinSys::set_list_items(Widget list, const List &strings)
{
  delete_list_items(list);
  Stringy cmd = Stringy(path(list)) + " insert end " + tcl_word_list(strings);
  tcl_command(wsp, cmd);
}

// ----------------------------------------------------------------------------
//
void WinSys::delete_list_items(Widget list)
  { tcl_command(wsp, "%s delete 0 end", path(list)); }
void WinSys::delete_list_position(Widget list, int pos)
  { tcl_command(wsp, "%s delete %s", path(list), Stringy(pos)); }

// ----------------------------------------------------------------------------
//
bool WinSys::selected_list_position(Widget list, int *position)
{
  int num, *positions;
  selected_list_positions(list, &num, &positions);
  if (num > 0)
    *position = positions[0];
  free_list_positions(positions);

  return (num > 0);
}

// ----------------------------------------------------------------------------
//
void WinSys::selected_list_positions(Widget list, int *num, int **positions)
{
  *num = 0;
  *positions = NULL;

  List index_list;
  if (tcl_command(wsp, "%s curselection", path(list)) &&
      command_result_list(wsp, &index_list))
    {
      int *pos = new int[index_list.size()];
      for (int k = 0 ; k < index_list.size() ; ++k)
	pos[k] = atoi(((Stringy *)index_list[k])->cstring());
      *num = index_list.size();;
      free_string_list_entries(index_list);
      *positions = pos;
    }
}

// ----------------------------------------------------------------------------
//
void WinSys::free_list_positions(int *positions)
  { delete [] positions; }

// ----------------------------------------------------------------------------
//
Stringy WinSys::selected_list_item(Widget list)
{
  int pos;

  if (selected_list_position(list, &pos))
    return list_item(wsp, list, pos);

  return "";
}

// ----------------------------------------------------------------------------
//
List WinSys::selected_list_items(Widget list)
{
  int num, *positions;
  selected_list_positions(list, &num, &positions);
  List slist;
  for (int k = 0 ; k < num ; ++k)
    slist.append(new Stringy(list_item(wsp, list, positions[k])));
  free_list_positions(positions);

  return slist;
}

// ----------------------------------------------------------------------------
//
void WinSys::select_list_position(Widget list, int pos, bool select)
{
  if (select)
    tcl_command(wsp, "%s selection set %s", path(list), Stringy(pos));
  else
    tcl_command(wsp, "%s selection clear %s", path(list), Stringy(pos));
}
 
// ----------------------------------------------------------------------------
//
void WinSys::deselect_list_items(Widget list)
  { tcl_command(wsp, "%s selection clear 0 end", path(list)); }

// ----------------------------------------------------------------------------
//
bool WinSys::find_list_substring(Widget list, int start_pos,
				 const Stringy &string, int *pos)
{
  if (string.is_empty())
    return false;

  List lines = list_items(wsp, list);
  bool found = false;
  for (int k = start_pos ; k < lines.size() && !found ; ++k)
    if (strstr(((Stringy *) lines[k])->cstring(), string.cstring()))
      {
	found = true;
	*pos = k;
      }
  free_string_list_entries(lines);

  return found;
}
 
// ----------------------------------------------------------------------------
// The listbox -height option returns the requested height instead of the
// actual height.  So I have to compute the real height.
//
static int visible_lines(WinSysP *wsp, Widget list)
{
  int h = wsp->ws.widget_height(list);
  int hfont;
  Stringy font = get_option(wsp, list, "-font");
  if (tcl_command(wsp, "font metrics %s -linespace", font) &&
      command_result_integer(wsp, &hfont))
    return h / hfont;
  return 0;
}

// ----------------------------------------------------------------------------
//
int WinSys::top_visible_list_position(Widget list)
{
  int top;
  if (tcl_command(wsp, "%s nearest 0", path(list)) &&
      command_result_integer(wsp, &top) &&
      top >= 0)				// Can return -1
    return top;
  return 0;
}
void WinSys::set_top_visible_list_position(Widget list, int position)
  { tcl_command(wsp, "%s yview %s", path(list), Stringy(position)); }

// ----------------------------------------------------------------------------
//
void WinSys::set_visible_list_length(Widget list, int length)
  { tcl_command(wsp, "%s configure -height %s", path(list), Stringy(length)); }
 
// ----------------------------------------------------------------------------
//
static Stringy list_item(WinSysP *wsp, Widget list, int position)
{
  if (tcl_command(wsp, "%s get %s", path(list), Stringy(position)))
    return command_result(wsp);
  return "";
}
 
// ----------------------------------------------------------------------------
//
static List list_items(WinSysP *wsp, Widget list)
{
  List lines;
  if (tcl_command(wsp, "%s get 0 end", path(list)))
    command_result_list(wsp, &lines);
  return lines;
}

// ----------------------------------------------------------------------------
//
void WinSys::center_list_item(Widget list, const Stringy &item, bool selected)
{
  List lines = list_items(wsp, list);
  for (int k = 0 ; k < lines.size() ; ++k)
    if (*(Stringy *)lines[k] == item)
      {
	center_list_position(list, k, selected);
	break;
      }
  free_string_list_entries(lines);
}

// ----------------------------------------------------------------------------
//
void WinSys::center_list_position(Widget list, int pos, bool selected)
{
  int visible = visible_lines(wsp, list);
  int top = max(pos - visible / 2, 0);
  set_top_visible_list_position(list, top);

  if (selected)
    select_list_position(list, pos, true);
}

// ----------------------------------------------------------------------------
// Omits blank lines.
//
bool WinSys::save_list_lines(Widget list, const Stringy &heading,
			     const Stringy &path)
{
  FILE *fp = fopen(path.cstring(), "w");	// Widget list to file
  if (fp)
    {
      if (!heading.is_empty())
	fprintf(fp, "%s\n\n", heading.cstring());
      List lines = list_items(wsp, list);
      for (int li = 0 ; li < lines.size() ; ++li)
	{
	  Stringy *line = (Stringy *) lines[li];
	  if (!line->is_white())
	    fprintf(fp, "%s\n", line->cstring());
	}
      free_string_list_entries(lines);
      fclose(fp);
    }

  return fp != NULL;
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_scrolled_text(Widget parent, const Stringy &name,
				    Widget *text)
{
  Widget f = create_form(parent, name);
  Stringy tpath = unique_child_path(wsp, f, "text");
  Stringy spath = unique_child_path(wsp, f, "scrollbar");

  if (tcl_command(wsp, "text %s", tpath) &&
      tcl_command(wsp, "scrollbar %s -command %s", spath, tpath + " yview") &&
      tcl_command(wsp, "%s configure -yscrollcommand %s", tpath, spath + " set") &&
      tcl_command(wsp, "grid rowconfigure %s 0 -weight 1", path(f)) &&
      tcl_command(wsp, "grid columnconfigure %s 0 -weight 1", path(f)) &&
      tcl_command(wsp, "grid %s -row 0 -column 0 -sticky news", tpath) &&
      tcl_command(wsp, "grid %s -row 0 -column 1 -sticky news", spath))
    {
      if (text)
	*text = named_widget(tpath);
      return f;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
void WinSys::replace_text(Widget t, const Stringy &text)
{
  tcl_command(wsp, "%s delete 1.0 end", path(t));
  tcl_command(wsp, quoted_word(path(t)) + " insert end " + quoted_word(text));
}

// ----------------------------------------------------------------------------
//
void WinSys::append_text(Widget t, const Stringy &text)
{
  tcl_command(wsp, "%s insert end %s", path(t), text);
  tcl_command(wsp, "%s see end", path(t));
  tcl_command(wsp, "%s mark set insert end", path(t));
}

// ----------------------------------------------------------------------------
//
void WinSys::add_text_position_callback(Widget t, CB_Proc cb, CB_Data cb_data)
  { add_event_handler(wsp, t, ButtonReleaseMask, button_1_filter, cb, cb_data); }
void WinSys::remove_text_position_callback(Widget t,
					   CB_Proc cb, CB_Data cb_data)
  { remove_event_handler(wsp, t, ButtonReleaseMask, button_1_filter, cb, cb_data); }
void WinSys::selected_text_position(Widget t, CB_Data, int *row, int *column)
{
  if (tcl_command(wsp, "%s index insert", path(t)) &&
      sscanf(command_result(wsp).cstring(), "%d.%d", row, column) == 2)
    *row -= 1; // row numbers start from 1
  else
    { *row = *column = 0; }
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_text_table(Widget parent, const Stringy &name,
				 Widget *row_heading, Widget *column_heading,
				 Widget *values)
{
  Widget f = create_form(parent, name);
  Stringy xhpath = unique_child_path(wsp, f, "xheading");
  Stringy yhpath = unique_child_path(wsp, f, "yheading");
  Stringy tpath = unique_child_path(wsp, f, "text");
  Stringy xspath = unique_child_path(wsp, f, "xscrollbar");
  Stringy yspath = unique_child_path(wsp, f, "yscrollbar");
  Stringy xscrollcmd = "ttscroll2 " + tpath + " " + xhpath + " xview";
  Stringy yscrollcmd = "ttscroll2 " + tpath + " " + yhpath + " yview";

  if (tcl_command(wsp, "text %s", tpath) &&
      tcl_command(wsp, "proc ttscroll2 {w1 w2 view args}"
		  " {eval $w1 $view $args ; eval $w2 $view $args}") &&
      tcl_command(wsp, "text %s", xhpath) &&
      tcl_command(wsp, "text %s", yhpath) &&
      tcl_command(wsp, "scrollbar %s -orient horizontal -command %s",
		  xspath, xscrollcmd) &&
      tcl_command(wsp, "%s configure -xscrollcommand %s", tpath, xspath + " set") &&
      tcl_command(wsp, "scrollbar %s -command %s", yspath, yscrollcmd) &&
      tcl_command(wsp, "%s configure -yscrollcommand %s", tpath, yspath + " set") &&
      tcl_command(wsp, "grid rowconfigure %s 1 -weight 1", path(f)) &&
      tcl_command(wsp, "grid columnconfigure %s 1 -weight 1", path(f)) &&
      tcl_command(wsp, "grid %s -row 1 -column 1 -sticky news", tpath) &&
      tcl_command(wsp, "grid %s -row 0 -column 1 -sticky news", xhpath) &&
      tcl_command(wsp, "grid %s -row 1 -column 0 -sticky news", yhpath) &&
      tcl_command(wsp, "grid %s -row 2 -column 1 -sticky news", xspath) &&
      tcl_command(wsp, "grid %s -row 1 -column 2 -sticky news", yspath))
    {
      if (row_heading)
	*row_heading = named_widget(yhpath);
      if (column_heading)
	*column_heading = named_widget(xhpath);
      if (values)
	*values = named_widget(tpath);
      return f;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
void WinSys::set_main_window_areas(Widget, Widget menubar,
				   Widget workwin, Widget statusline)
{
  column_attachments(workwin, menubar, workwin, statusline, END_OF_WIDGETS);
}

// ----------------------------------------------------------------------------
// Return the path of a user selected file that must already exist.
// Save the user chosen directory for this file type in the option database
// and have it override the directory argument.
//
Stringy WinSys::open_file_dialog(Widget parent, const Stringy &title,
				 const Stringy &directory,
				 const Stringy &file_type,
				 bool prefer_previous)
{
  Stringy types = "{Any *} {Assignments .save} {Project .proj} {Spectrum *}";
  Stringy dir = directory;
  if (prefer_previous)
    {
      Stringy p = default_path(wsp, file_type, file_path(directory, "file"));
      dir = file_directory(p);
    }

  //
  // File dialog fails to come up if initial directory doesn't exist.
  //
  if (!is_directory(dir))
    dir = current_directory();

  if (tcl_command(wsp, "tk_getOpenFile -parent %s -title %s "
		  " -initialdir %s -filetypes %s",
		  path(parent), // tk 8.0p1 bug prevents this from working
		  title, dir, types))
    {
      Stringy r = command_result(wsp);

      // following line fixes bug where Tk returns / separated path in Windows
      r = replace_character(r, '/', path_separator()[0]);

      set_default_path(wsp, file_type, r);

      return r;
    }
  return "";
}

// ----------------------------------------------------------------------------
// Return the path of a user selected file.  Ask on overwrite.
// Save the user chosen directory for this file type in the option database
// and have it override the directory argument.
//
Stringy WinSys::saveas_file_dialog(Widget parent, const Stringy &title,
				   const Stringy &path,
				   const Stringy &file_type,
				   bool prefer_previous)
{
  Stringy types = "{Any *} {Assignments .save} {Project .proj}";
  Stringy p = (prefer_previous ? default_path(wsp, file_type, path) : path);
  Stringy dir = file_directory(p);
  Stringy file = file_name(p);

  //
  // File dialog fails to come up if initial directory doesn't exist.
  //
  if (!is_directory(dir))
    dir = current_directory();

  if (tcl_command(wsp, "tk_getSaveFile -parent %s -title %s"
		  " -initialdir %s -initialfile %s -filetypes %s",
		  ::path(parent), // tk 8.0p1 bug prevents this from working
		  title, dir, file, types))
    {
      Stringy r = command_result(wsp);

      // following line fixes bug where Tk returns / separated path in Windows
      r = replace_character(r, '/', path_separator()[0]);

      set_default_path(wsp, file_type, r);

      return r;
    }
  return "";
}

// ----------------------------------------------------------------------------
//
static Stringy default_path(WinSysP *wsp, const Stringy &filetype, const Stringy &path)
{
  Stringy defaultdir = file_directory(path);
  Stringy dir = wsp->ws.read_application_resource(main_widget(wsp),
					  filetype + "Directory",
					  "Path", defaultdir);
  return file_path(dir, file_name(path));
}

// ----------------------------------------------------------------------------
//
static void set_default_path(WinSysP *wsp, const Stringy &filetype, const Stringy &path)
{
  if (! path.is_empty() && tcl_command(wsp, "tk appname"))
    {
      Stringy pattern = command_result(wsp) + "." + filetype + "Directory";
      tcl_command(wsp, "option add %s %s", pattern, file_directory(path));
    }
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_label(Widget parent, const Stringy &name)
{
  Stringy path = unique_child_path(wsp, parent, name);
  if (tcl_command(wsp, "label %s", path))
    return named_widget(path);
  return NULL;
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_frame(Widget parent, const Stringy &name)
{
  Stringy path = unique_child_path(wsp, parent, name);
  Stringy classname = capitalize(name);
  if (tcl_command(wsp, "frame %s -class %s", path, classname))
    return named_widget(path);
  return NULL;
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_text_field(Widget parent, const Stringy &name)
{
  Stringy path = unique_child_path(wsp, parent, name);
  if (tcl_command(wsp, "entry %s", path))
    return named_widget(path);
  return NULL;
}

// ----------------------------------------------------------------------------
//
Stringy WinSys::text_field_string(Widget tf)
{
  if (tcl_command(wsp, "%s get", path(tf)))
    return command_result(wsp);
  return "";
}

// ----------------------------------------------------------------------------
//
void WinSys::set_text_field(Widget tf, const Stringy &s)
{
  if (tcl_command(wsp, "%s delete 0 end", path(tf)) &&
      tcl_command(wsp, "%s insert end %s", path(tf), s))
    ;
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_toggle_button(Widget parent, const Stringy &name)
{
  Stringy bpath = unique_child_path(wsp, parent, name);
  if (tcl_command(wsp, "checkbutton %s -highlightthickness 0", bpath))
    {
      Widget b = named_widget(bpath);
      Stringy vname = widget_variable(b);
      tcl_command(wsp, "%s configure -variable %s", path(b), vname);
      return b;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
void WinSys::add_toggle_button_callback(Widget tb, CB_Proc cb, CB_Data cb_data)
{
  Command_Callback *cc = new Command_Callback(wsp, tb, cb, cb_data);
  tcl_command(wsp, "%s configure -command %s", path(tb), cc->name());
}

// ----------------------------------------------------------------------------
//
bool WinSys::toggle_button_state(Widget button)
{
  Stringy vname = get_option(wsp, button, "-variable");
  const char *value = Tcl_GetVar(wsp->tcl_interp, vname.cstring(),
				 TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG);
  int b;
  if (value && Tcl_GetBoolean(wsp->tcl_interp, value, &b) == TCL_OK)
    return b;

  return false;
}

// ----------------------------------------------------------------------------
//
void WinSys::set_toggle_button(Widget button, bool state, bool notify)
{
  tcl_command(wsp, "%s %s", path(button), (state ? "select" : "deselect"));
  if (notify)
    {
      Stringy cmd = get_option(wsp, button, "-command");
      if (!cmd.is_empty())
	tcl_command(wsp, cmd);
    }
}

// ----------------------------------------------------------------------------
//
void WinSys::set_toggle_button_text(Widget button, const Stringy &text)
  { tcl_command(wsp, "%s configure -text %s", path(button), text); }

// ----------------------------------------------------------------------------
//
Widget WinSys::option_menu(Widget parent, const Stringy &menu_name,
			   const char *entries[])
{
  Widget f = create_form(parent, menu_name);
  Widget h = create_label(f, "label");
  Widget b = create_menu(f, "button", "menu");
  tcl_command(wsp, "%s configure -indicatoron 1 -borderwidth 2 -relief raised",
	      path(b));
  Widget p = menu_pane(b);
  tcl_command(wsp, "%s configure -tearoff 0", path(p));
  row_attachments(NULL, h, b, END_OF_WIDGETS);

  if (entries)
    for (int k = 0 ; entries[k] ; ++k)
      add_menu_button(p, entries[k], option_selected_cb, f);

  return f;
}

// ----------------------------------------------------------------------------
//
static Widget option_button(WinSysP *wsp, Widget option_menu)
  { return wsp->ws.child_widget(option_menu, "button"); }
static Widget option_pane(WinSysP *wsp, Widget option_menu)
  { return wsp->ws.menu_pane(option_button(wsp, option_menu)); }

// ----------------------------------------------------------------------------
//
void WinSys::add_option(Widget option_menu, const Stringy &name)
{
  Widget pane = option_pane(wsp, option_menu);
  add_menu_button(pane, name, option_selected_cb, option_menu);
}

// ----------------------------------------------------------------------------
//
void WinSys::add_options(Widget option_menu, const char *names[])
{
  for (int k = 0 ; names[k] ; ++k)
    add_option(option_menu, names[k]);
}

// ----------------------------------------------------------------------------
//
void WinSys::delete_option(Widget option_menu, const Stringy &name)
{
  Widget pane = option_pane(wsp, option_menu);
  Stringy label = button_label_option(wsp, pane, name);
  tcl_command(wsp, "%s delete %s", path(pane), label);
}

// ----------------------------------------------------------------------------
//
void WinSys::delete_all_options(Widget option_menu)
{
  Widget choices = option_pane(wsp, option_menu);
  tcl_command(wsp, "%s delete 0 end", path(choices));
}

// ----------------------------------------------------------------------------
//
Stringy WinSys::option_selected(Widget option_menu)
  { return get_variable(wsp, option_button(wsp, option_menu)); }

// ----------------------------------------------------------------------------
// Set option menu button text and variable to named option
// Option menu callback is invoked if one exists.
//
void WinSys::set_option(Widget option_menu, const Stringy &name,
			bool call_callback)
{
  Widget pane = option_pane(wsp, option_menu);
  Stringy label = button_label_option(wsp, pane, name);

  Widget b = option_button(wsp, option_menu);
  set_variable(wsp, b, name);
  tcl_command(wsp, "%s configure -text %s", path(b), label);

  if (call_callback)
    {
      //
      // Invoke option menu callback if one exists.
      //
      Stringy cb_name = path(option_menu) + ".callback";
      if (tcl_command_exists(wsp, cb_name))
	tcl_command(wsp, "%s", cb_name);
    }
}

// ----------------------------------------------------------------------------
//
void WinSys::set_option(Widget option_menu, int num)
{
  Widget pane = option_pane(wsp, option_menu);
  tcl_command(wsp, "%s invoke %s", path(pane), Stringy(num));
}

// ----------------------------------------------------------------------------
//
void WinSys::sensitize_option(Widget option_menu, const Stringy &name,
			      bool sensitive)
{
  Widget pane = option_pane(wsp, option_menu);
  tcl_command(wsp, "%s entryconfigure %s -state %s",
	      path(pane), name, (sensitive ? "normal" : "disabled"));
}

// ----------------------------------------------------------------------------
//
void WinSys::option_callback(Widget option_menu, CB_Proc cb, CB_Data cb_data)
{
  Command_Callback *cc = new Command_Callback(wsp, option_menu, cb, cb_data);
  Stringy cb_name = path(option_menu) + ".callback";
  cc->rename(cb_name);
}

// ----------------------------------------------------------------------------
//
static void option_selected_cb(Widget, CB_Data omenu, CB_Data menu_cb)
{
  Widget option_menu = (Widget) omenu;
  Menu_Callback *mcb = (Menu_Callback *) menu_cb;
  mcb->winsysp()->ws.set_option(option_menu, mcb->button_name);
}

// ----------------------------------------------------------------------------
//
static void set_delete_protocol(WinSysP *wsp, Widget shell, CB_Proc cb, CB_Data cb_data)
{
  Command_Callback *cc = new Command_Callback(wsp, shell, cb, cb_data);
  tcl_command(wsp, "wm protocol %s WM_DELETE_WINDOW %s", path(shell), cc->name());
}

// ----------------------------------------------------------------------------
//
static void remove_delete_protocol(WinSysP *wsp, Widget shell, CB_Proc, CB_Data)
{
  tcl_command(wsp, "wm protocol %s WM_DELETE_WINDOW %s", path(shell), Stringy());
  // Tk reinstates default delete protocol (destroy window).
}

// ----------------------------------------------------------------------------
//
static bool map_filter(WinSysP *, XEvent *event)
  { return (event->type == MapNotify || event->type == UnmapNotify); }
static bool button_1_filter(WinSysP *, XEvent *event)
  { return event->xbutton.button == Button1; }
static bool button_2_filter(WinSysP *, XEvent *event)
  { return event->xbutton.button == Button2; }
static bool button_3_filter(WinSysP *, XEvent *event)
  { return event->xbutton.button == Button3; }
static bool move_filter(WinSysP *wsp, XEvent *event)
  { return ((event->xmotion.state & Button1Mask) == 0 &&
	    !wsp->ws.more_motion_events()); }
static bool drag_filter(WinSysP *wsp, XEvent *event)
  { return ((event->xmotion.state & Button1Mask) &&
	    !wsp->ws.more_motion_events()); }
static bool got_focus_filter(WinSysP *, XEvent *event)
  { return event->type == FocusIn; }
static bool destroy_filter(WinSysP *, XEvent *event)
  { return event->type == DestroyNotify; }
static bool resize_filter(WinSysP *, XEvent *event)
  { return event->type == ConfigureNotify; }

// ----------------------------------------------------------------------------
//
static bool enter_key_filter(WinSysP *wsp, XEvent *event)
{
  char c;
  return (wsp->ws.key_pressed(event, &c) && c == '\r');
}

// ----------------------------------------------------------------------------
//
static bool key_press_filter(WinSysP *wsp, XEvent *event)
{
  char c;
  int f;
  bool shift;
  return (wsp->ws.key_pressed(event, &c) ||
	  wsp->ws.function_key_pressed(event, &f, &shift));
}

// ----------------------------------------------------------------------------
//
static bool double_button_1_filter(WinSysP *wsp, XEvent *event)
{
  if (event->xbutton.button == Button1)
    {
      const unsigned long MAX_DOUBLE_CLICK_DELAY = 500;	// milliseconds
      if (wsp->last_double_click_time &&
	  event->xbutton.time - wsp->last_double_click_time
	  < MAX_DOUBLE_CLICK_DELAY)
	{
	  wsp->last_double_click_time = 0;
	  return true;
	}
      else
	wsp->last_double_click_time = event->xbutton.time;
    }
  return false;
}

// ----------------------------------------------------------------------------
//
void WinSys::add_event_callback(Event_Type type, Widget w,
				CB_Proc cb, CB_Data cb_data)
{
  switch (type)
    {
    case destroy_query_event:
      set_delete_protocol(wsp, w, cb, cb_data);
      break;
    case destroy_event:
      add_event_handler(wsp, w, StructureNotifyMask, destroy_filter, cb, cb_data);
      break;
    case expose_event:
      add_event_handler(wsp, w, ExposureMask, NULL, cb, cb_data);
      break;
    case resize_event:
      add_event_handler(wsp, w, StructureNotifyMask, resize_filter, cb, cb_data);
      break;
    case mapping_event:
      add_event_handler(wsp, w, StructureNotifyMask, map_filter, cb, cb_data);
      break;
    case button_1_press_event:
      add_event_handler(wsp, w, ButtonPressMask, button_1_filter, cb, cb_data);
      break;
    case button_1_release_event:
      add_event_handler(wsp, w, ButtonReleaseMask, button_1_filter, cb, cb_data);
      break;
    case button_3_press_event:
      add_event_handler(wsp, w, ButtonPressMask, button_3_filter, cb, cb_data);
      break;
    case pointer_move_event:
      add_event_handler(wsp, w, PointerMotionMask, move_filter, cb, cb_data);
      break;
    case pointer_drag_event:
      add_event_handler(wsp, w, PointerMotionMask, drag_filter, cb, cb_data);
      break;
    case pointer_pause_event:
      add_pointer_pause_callback(wsp, w, cb, cb_data);
      break;
    case enter_window_event:
      add_event_handler(wsp, w, EnterWindowMask, NULL, cb, cb_data);
      break;
    case leave_window_event:
      add_event_handler(wsp, w, LeaveWindowMask, NULL, cb, cb_data);
      break;
    case got_focus_event:
      add_event_handler(wsp, w, FocusChangeMask, got_focus_filter, cb, cb_data);
      break;
    case key_press_event:
      add_event_handler(wsp, w, KeyPressMask, key_press_filter, cb, cb_data);
      break;
    case enter_pressed_event:
      add_event_handler(wsp, w, KeyPressMask, enter_key_filter, cb, cb_data);
      break;
    };
}

// ----------------------------------------------------------------------------
//
void WinSys::remove_event_callback(Event_Type type, Widget w,
				   CB_Proc cb, CB_Data cb_data)
{
  switch (type)
    {
    case destroy_query_event:
      remove_delete_protocol(wsp, w, cb, cb_data);
      break;
    case destroy_event:
      remove_event_handler(wsp, w, StructureNotifyMask,
			   destroy_filter, cb, cb_data);
      break;
    case expose_event:
      remove_event_handler(wsp, w, ExposureMask, NULL, cb, cb_data);
      break;
    case resize_event:
      remove_event_handler(wsp, w, StructureNotifyMask, resize_filter, cb, cb_data);
      break;
    case mapping_event:
      remove_event_handler(wsp, w, StructureNotifyMask, map_filter, cb, cb_data);
      break;
    case button_1_press_event:
      remove_event_handler(wsp, w, ButtonPressMask, button_1_filter, cb, cb_data);
      break;
    case button_1_release_event:
      remove_event_handler(wsp, w, ButtonReleaseMask, button_1_filter, cb, cb_data);
      break;
    case button_3_press_event:
      remove_event_handler(wsp, w, ButtonPressMask, button_3_filter, cb, cb_data);
      break;
    case pointer_move_event:
      remove_event_handler(wsp, w, PointerMotionMask, move_filter, cb, cb_data);
      break;
    case pointer_drag_event:
      remove_event_handler(wsp, w, PointerMotionMask, drag_filter, cb, cb_data);
      break;
    case pointer_pause_event:
      remove_pointer_pause_callback(wsp, w, cb, cb_data);
      break;
    case enter_window_event:
      remove_event_handler(wsp, w, EnterWindowMask, NULL, cb, cb_data);
      break;
    case leave_window_event:
      remove_event_handler(wsp, w, LeaveWindowMask, NULL, cb, cb_data);
      break;
    case got_focus_event:
      remove_event_handler(wsp, w, FocusChangeMask, got_focus_filter, cb, cb_data);
      break;
    case key_press_event:
      remove_event_handler(wsp, w, KeyPressMask, key_press_filter, cb, cb_data);
      break;
    case enter_pressed_event:
      remove_event_handler(wsp, w, KeyPressMask, enter_key_filter, cb, cb_data);
      break;
    };
}

// ----------------------------------------------------------------------------
//
static Tk_Window window(Widget w)
  { return (Tk_Window) w; }
static Window x_window(Widget w)
{
  Window win = Tk_WindowId(window(w));
  if (win == NULL)
    fatal_error("x_window: widget not realized.\n");
  return win;
}
static bool window_exists(Widget w)
  { return Tk_WindowId(window(w)) != NULL; }
static void make_window_exist(Widget w)
  { Tk_MakeWindowExist(window(w)); }
static void map_widget(Widget w)
  { Tk_MapWindow(window(w)); }
static bool is_mapped(Widget w)
  { return Tk_IsMapped(window(w)); }
bool WinSys::is_top_level(Widget w)
  { return Tk_IsTopLevel(window(w)); }
void WinSys::delete_widget(Widget w)
  { tcl_command(wsp, "destroy %s", path(w)); }
static Widget parent_widget(Widget w)
  { return Tk_Parent(window(w)); }

// ----------------------------------------------------------------------------
//
Widget WinSys::push_button(Widget parent, const Stringy &name,
			   CB_Proc cb, CB_Data cb_data)
{
  Stringy path = unique_child_path(wsp, parent, name);
  if (tcl_command(wsp, "button %s", path))
    {
      default_button_text(wsp, path, name);
      Widget b = named_widget(path);
      Command_Callback *cc = new Command_Callback(wsp, b, cb, cb_data);
      tcl_command(wsp, "%s configure -command %s", path, cc->name());
      return b;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
static void default_button_text(WinSysP *wsp, const Stringy &path, const Stringy &text)
{
  if (tcl_command(wsp, "%s cget -text", path) && command_result(wsp).is_empty())
    tcl_command(wsp, "%s configure -text %s", path, text);
}

// ----------------------------------------------------------------------------
//
Widget WinSys::button_row(Widget parent, const Stringy &name, char *first, ...)
{
  Widget controls = create_form(parent, name);

  va_list args;
  va_start(args, first);
  for (char *bname = first ; bname != NULL ; bname = va_arg(args, char *))
    {
      CB_Proc cb = va_arg(args, CB_Proc);
      CB_Data client_data = va_arg(args, CB_Data);
      Widget control = push_button(controls, bname, cb, client_data);
      tcl_command(wsp, "pack %s -side left", path(control));
    }
  va_end(args);

  return controls;
}

// ----------------------------------------------------------------------------
//
Widget WinSys::create_radio_buttons(Widget parent, const Stringy &name)
{
  return create_form(parent, name);
}

// ----------------------------------------------------------------------------
//
void WinSys::add_radio_button(Widget rbuttons, const Stringy &name)
{
  Stringy path = unique_child_path(wsp, rbuttons, name);
  Stringy vname = widget_variable(rbuttons);
  if (tcl_command(wsp, "radiobutton %s -variable %s -value %s", path, vname, name))
    {
      default_button_text(wsp, path, name);
      tcl_command(wsp, "pack %s -side top -fill x", path);
    }
}

// ----------------------------------------------------------------------------
//
void WinSys::set_radio_button(Widget rbuttons, const Stringy &name)
  { set_variable(wsp, rbuttons, name); }

// ----------------------------------------------------------------------------
//
Stringy WinSys::radio_selection(Widget rbuttons)
  { return get_variable(wsp, rbuttons); }

// ----------------------------------------------------------------------------
//
void WinSys::radio_callback(Widget rbuttons, CB_Proc cb, CB_Data cb_data)
{
  variable_changed_callback(wsp, rbuttons, cb, cb_data);
}

// ----------------------------------------------------------------------------
//
Widget WinSys::edit_field(Widget parent, const Stringy &name, bool menu)
{
  Widget edit = create_form(parent, name);

  Widget title = create_label(edit, "title");
  Widget text = create_text_field(edit, "text");

  tcl_command(wsp, "pack %s -side left", path(title));
  tcl_command(wsp, "pack %s -side left", path(text));

  if (menu)
    {
      Widget b = create_menu(edit, "button", "menu");
      tcl_command(wsp, "%s configure -text V", path(b));
      tcl_command(wsp, "pack %s -side left", path(b));
    }

  return edit;
}

// ----------------------------------------------------------------------------
//
static Widget edit_field_title(WinSysP *wsp, Widget ef)
  { return wsp->ws.child_widget(ef, "title"); }
Widget WinSys::edit_field_menu(Widget ef)
  { return child_widget(ef, "button"); }
Widget WinSys::edit_field_text(Widget ef)
  { return child_widget(ef, "text"); }

// ----------------------------------------------------------------------------
//
Stringy WinSys::edit_value(Widget ef)
  { return text_field_string(edit_field_text(ef)); }

// ----------------------------------------------------------------------------
//
void WinSys::set_edit_title(Widget ef, const Stringy &title)
{
  set_label_text(edit_field_title(wsp, ef), title);
}

// ----------------------------------------------------------------------------
//
void WinSys::set_edit_value(Widget ef, const Stringy &value)
{
  Stringy old_value = edit_value(ef);
  if (value != old_value)
    set_text_field(edit_field_text(ef), value);
}

// ----------------------------------------------------------------------------
//
void WinSys::edit_field_editable(Widget ef, bool editable)
{
  tcl_command(wsp, "%s configure -state %s",
	      path(edit_field_text(ef)), (editable ? "normal" : "disabled"));
}

// ----------------------------------------------------------------------------
//
double WinSys::numeric_edit_value(Widget ef)
  { return numeric_text_field(edit_field_text(ef)); }

// ----------------------------------------------------------------------------
//
void WinSys::set_numeric_edit_value(Widget ef, const Stringy &format,
				    double value)
  { set_numeric_text_field(edit_field_text(ef), format, value); }

// ----------------------------------------------------------------------------
//
void WinSys::add_edit_field_menu_entry(Widget pane, const Stringy &text,
				       Widget ef)
{
  add_menu_button(pane, text, set_edit_field_cb, ef);
}

// ----------------------------------------------------------------------------
//
static void set_edit_field_cb(Widget, CB_Data edit_field, CB_Data menu_cb)
{
  Widget ef = (Widget) edit_field;
  Menu_Callback *mcb = (Menu_Callback *) menu_cb;
  mcb->winsysp()->ws.set_edit_value(ef, mcb->button_text);
}

// ----------------------------------------------------------------------------
//
double WinSys::numeric_text_field(Widget tf)
{
  double value;

  Stringy value_text = text_field_string(tf);
  if (sscanf(value_text.cstring(), "%lf", &value) != 1)
    value = 0;

  return value;
}

// ----------------------------------------------------------------------------
//
void WinSys::set_numeric_text_field(Widget tf, const Stringy &format,
				    double value)
{
  set_text_field(tf, formatted_string(format.cstring(), value));
}

// ----------------------------------------------------------------------------
//
bool WinSys::text_field_is_empty(Widget tf)
{
  return text_field_string(tf).is_empty();
}

// ----------------------------------------------------------------------------
//
void WinSys::add_text_field_changed_callback(Widget tf,
					     CB_Proc cb, CB_Data cb_data)
{
  Stringy vname = widget_variable(tf);
  tcl_command(wsp, "%s configure -textvariable %s", path(tf), vname);
  variable_changed_callback(wsp, tf, cb, cb_data);
}
void WinSys::remove_text_field_changed_callback(Widget tf,
						CB_Proc cb, CB_Data cb_data)
  { remove_variable_changed_callback(wsp, tf, cb, cb_data); }

// ----------------------------------------------------------------------------
//
Widget WinSys::switches(Widget parent, const Stringy &name,
			const char *names[], int columns)
{
  Widget f = create_form(parent, name);

  int count = 0;
  while (names[count] != NULL)
    count += 1;

  int column_length = (count + columns - 1) / columns;

  for (int k = 0 ; names[k] != NULL ; ++k)
    {
      Widget b = create_toggle_button(f, names[k]);
      int row = k % column_length;
      int column = k / column_length;
      tcl_command(wsp, "grid %s -row %s -column %s -sticky nw",
		  path(b), Stringy(row), Stringy(column));
    }

  return f;
}

// ----------------------------------------------------------------------------
//
void WinSys::set_switch(Widget switches, const Stringy &name, bool state)
{
  Widget w = child_widget(switches, name);
  tcl_command(wsp, "%s %s", path(w), (state ? "select" : "deselect"));
}

// ----------------------------------------------------------------------------
//
bool WinSys::switch_state(Widget switches, const Stringy &name)
{
  Widget w = child_widget(switches, name);
  return toggle_button_state(w);
}

// ----------------------------------------------------------------------------
//
void WinSys::switch_callback(Widget switches, const Stringy &name,
			     CB_Proc cb, CB_Data cb_data)
{
  Widget w = child_widget(switches, name);
  add_toggle_button_callback(w, cb, cb_data);
}

// ----------------------------------------------------------------------------
//
Widget WinSys::widget_table(Widget parent, const Stringy &name,
			    int rows, int columns, ...)
{
  Widget table = create_form(parent, name);

  va_list args;
  va_start(args, columns);
  for (int r = 0 ; r < rows ; ++r)
    for (int c = 0 ; c < columns ; ++c)
      {
	Table_Entry te = (Table_Entry) va_arg(args, int);
	Stringy name = table_element_name(table, r, c);
	switch (te)
	  {
	  case TABLE_LABEL: create_label(table, name); break;
	  case TABLE_TEXT_FIELD: create_text_field(table, name); break;
	  case TABLE_OPTION_MENU: option_menu(table, name, NULL); break;
	  case TABLE_TOGGLE_BUTTON: create_toggle_button(table, name); break;
	  };
      }
  va_end(args);

  manage_table_children(table, rows, columns);

  return table;
}

// ----------------------------------------------------------------------------
//
void WinSys::manage_table_children(Widget table, int rows, int columns)
{
  clear_grid(wsp, table);
  for (int r = 0 ; r < rows ; ++r)
    for (int c = 0 ; c < columns ; ++c)
      {
	Stringy path = table_element_path(table, r, c);
	if (widget_exists(wsp, path))
	  tcl_command(wsp, "grid configure %s -row %s -column %s -sticky w",
		      path, Stringy(r), Stringy(c));
      }
}

// ----------------------------------------------------------------------------
//
void WinSys::manage_table_row(Widget table, int r)
{
  for (int c = 0 ; path_exists(wsp, table_element_path(table, r, c)) ; ++c)
    tcl_command(wsp, "grid configure %s -row %s -column %s",
		 table_element_path(table, r, c), Stringy(r), Stringy(c));
}

// ----------------------------------------------------------------------------
//
Widget WinSys::table_element(Widget table, int row, int column)
{
  Stringy path = table_element_path(table, row, column);
  return (widget_exists(wsp, path) ? named_widget(path) : NULL);
}
Stringy WinSys::table_element_name(Widget, int row, int column)
  { return formatted_string("r%dc%d", row, column); }
static Stringy table_element_path(Widget table, int row, int column)
  { return formatted_string("%s.r%dc%d", path(table).cstring(), row, column); }

// ----------------------------------------------------------------------------
//
void WinSys::show_dialog(Widget w)
{
  tcl_command(wsp, "wm deiconify %s", path(w));
  // Tk_MapWindow(window(w));
}
void WinSys::unshow_dialog(Widget w)
{
  //
  // This is needed to work around the problem that window positions
  // are forgotten when Tk_UnmapWindow/Tk_MapWindow called.
  //
  tcl_command(wsp, "wm positionfrom %s user", path(w));

  tcl_command(wsp, "wm withdraw %s", path(w));
  //  Tk_UnmapWindow(window(w));
}

// ----------------------------------------------------------------------------
//
void WinSys::iconify_dialog(Widget w)
{
  tcl_command(wsp, "wm iconify %s", path(w));
}

// ----------------------------------------------------------------------------
// This routine looks for an child with the specified name.
//
Widget WinSys::child_widget(Widget parent, const Stringy &childname)
{
  Stringy cpath = child_path(parent, childname);
  return named_widget(cpath);
}

// ----------------------------------------------------------------------------
//
static List child_widgets(WinSysP *wsp, Widget w)
{
  List children;
  if (tcl_command(wsp, "winfo children %s", path(w)))
    {
      List child_paths;
      if (command_result_list(wsp, &child_paths))
	{
	  for (int k = 0 ; k < child_paths.size() ; ++k)
	    children.append(wsp->ws.named_widget(*(Stringy *) child_paths[k]));
	  free_string_list_entries(child_paths);
	}
    }
  return children;
}

// ----------------------------------------------------------------------------
//
void WinSys::set_icon_name(Widget w, const Stringy &name)
{
  tcl_command(wsp, "wm iconname %s %s", path(w), name);
}

// ----------------------------------------------------------------------------
//
static void destroyed_modal_widget_cb(Widget, CB_Data destroyed, CB_Data)
{
  *(bool *)destroyed = true;
}

// ----------------------------------------------------------------------------
//
static void raise_modal_widget_cb(Widget modal_dialog, CB_Data winsysp, CB_Data)
{
  WinSysP *wsp = (WinSysP *) winsysp;
  wsp->ws.show_dialog(modal_dialog);
  wsp->ws.raise_widget(modal_dialog);
}

// ----------------------------------------------------------------------------
//
void WinSys::raise_widget(Widget w)
{
  tcl_command(wsp, "after idle raise %s", path(toplevel(wsp, w)));
}

// ----------------------------------------------------------------------------
//
static Widget toplevel(WinSysP *wsp, Widget w)
{
  if (tcl_command(wsp, "winfo toplevel %s", path(w)))
    return wsp->ws.named_widget(command_result(wsp));
  return NULL;
}

// ----------------------------------------------------------------------------
//
bool WinSys::is_viewable(Widget w)
{
  //
  // Optimization:  Instead of using "winfo viewable path" tcl command
  //		    directly check if window or any parent is unmapped.
  //
  for (Widget v = w ; v != NULL ; v = parent_widget(v))
    if (!is_mapped(v))
      return false;

  return true;
}

// ----------------------------------------------------------------------------
// Unmap all child toplevels when when main window is unmapped.
// When the main window is mapped, remap all previously unmapped
// child toplevels.  This could be done with the Tk "wm group"
// command but it doesn't appear to work with the CDE window manager
// on DEC Alpha Unix 4.0b.  Even if it did work it would probably
// also insist the main window stay on top of all other windows,
// and I do not want this.
//
extern "C"
{
static void iconify_all_cb(ClientData winsysp, XEvent *event)
{
  WinSysP *wsp = (WinSysP *) winsysp;
  if (event->type == UnmapNotify)
    {
      free_string_list_entries(wsp->hidden_dialogs);
      List children = child_widgets(wsp, main_widget(wsp));
      for (int k = 0 ; k < children.size() ; ++k)
	{
	  Widget w = (Widget) children[k];
	  if (wsp->ws.is_top_level(w) && is_mapped(w))
	    {
	      wsp->hidden_dialogs.append(new Stringy(path(w)));
	      wsp->ws.unshow_dialog(w);
	    }
	}
    }
  else if (event->type == MapNotify)
    {
      for (int k = 0 ; k < wsp->hidden_dialogs.size() ; ++k)
	{
	  Stringy path = *(Stringy *) wsp->hidden_dialogs[k];
	  if (widget_exists(wsp, path))
	    wsp->ws.show_dialog(wsp->ws.named_widget(path));
	}
      free_string_list_entries(wsp->hidden_dialogs);
    }
}
}

// ----------------------------------------------------------------------------
// Used right before data structures are all cleaned up at shutdown.
//
void WinSys::unshow_all_dialogs()
{
  List children = child_widgets(wsp, main_widget());
  for (int k = 0 ; k < children.size() ; ++k)
    {
      Widget w = (Widget) children[k];
      if (is_top_level(w) && is_mapped(w))
	unshow_dialog(w);
    }
  unshow_dialog(main_widget());
  tcl_command(wsp, "update idletasks");	// make sure windows get unmapped
}

// ----------------------------------------------------------------------------
//
void WinSys::delete_dialogs_except_main()
{
  List children = child_widgets(wsp, main_widget());
  for (int k = 0 ; k < children.size() ; ++k)
    {
      Widget w = (Widget) children[k];
      if (is_top_level(w))
	delete_widget(w);
    }
}

// ----------------------------------------------------------------------------
//
static Display *widget_display(Widget w)
  { return Tk_Display(window(w)); }
int WinSys::widget_width(Widget w)
  { return ::widget_width(w); }
static int widget_width(Widget w)
  { return Tk_Width(window(w)); }
int WinSys::widget_height(Widget w)
  { return ::widget_height(w); }
static int widget_height(Widget w)
  { return Tk_Height(window(w)); }
int WinSys::requested_width(Widget w)
  { return Tk_ReqWidth(window(w)); }
int WinSys::requested_height(Widget w)
  { return Tk_ReqHeight(window(w)); }
int WinSys::widget_x(Widget w)
  { return Tk_X(window(w)); }
int WinSys::widget_y(Widget w)
  { return Tk_Y(window(w)); }

// ----------------------------------------------------------------------------
//
void WinSys::set_widget_size(Widget w, int width, int height)
  { Tk_GeometryRequest(window(w), width, height); }
void WinSys::set_widget_width(Widget w, int width)
  { set_widget_size(w, width, Tk_ReqHeight(window(w))); }
void WinSys::set_widget_height(Widget w, int height)
  { set_widget_size(w, Tk_ReqWidth(window(w)), height); }
void WinSys::place_widget(Widget w, int x, int y)
  { tcl_command(wsp, "place %s -x %s -y %s", path(w), Stringy(x), Stringy(y)); }
void WinSys::unplace_widget(Widget w)
  { tcl_command(wsp, "place forget %s", path(w)); }

// ----------------------------------------------------------------------------
//
Widget WinSys::choice_dialog(const Stringy &name, bool allow_destroy,
			     const char *question,
			     // const char *choice, CB_Proc, CB_Data
			     ...)
{
  Widget dialog = create_dialog(name, allow_destroy);
  Widget q = create_label(dialog, "question");
  tcl_command(wsp, "%s configure -anchor nw -justify left -text %s",
	      path(q), question);
  Widget sep = create_separator(dialog, "separator");
  Widget buttons = create_form(dialog, "choices");

  va_list args;
  va_start(args, question);
  int count = 0;
  const char *btext;
  while ((btext = va_arg(args, const char *)))
    {
      count += 1;
      CB_Proc cb = va_arg(args, CB_Proc);
      void *cb_data = va_arg(args, void *);
      Stringy bname = "choice" + Stringy(count);
      Widget b = push_button(buttons, bname, cb, cb_data);
      tcl_command(wsp, "%s configure -text %s", path(b), btext);
      tcl_command(wsp, "pack %s -side left", path(b));
      add_event_callback(button_1_release_event, b, unshow_dialog_cb, wsp);
    }
  va_end(args);

  if (count > 0)
    column_attachments(q, q, sep, buttons, END_OF_WIDGETS);
  else
    column_attachments(q, q, END_OF_WIDGETS);

  return dialog;
}

// ----------------------------------------------------------------------------
//
static void unshow_dialog_cb(Widget button, CB_Data winsysp, CB_Data)
{
  WinSysP *wsp = (WinSysP *) winsysp;
  wsp->ws.unshow_dialog(toplevel(wsp, button));
}

// ----------------------------------------------------------------------------
//
void WinSys::new_choice_question(Widget cd, const Stringy &question)
{
  set_label_text(child_widget(cd, "question"), question);
}

// ----------------------------------------------------------------------------
//
void WinSys::delete_choice_dialog(Widget cd)
{
  delete_widget(cd);
}

// ----------------------------------------------------------------------------
//
bool WinSys::modal_dialog_shown()
{
  return (tcl_command(wsp, "grab current") && !command_result(wsp).is_empty());
}

// ----------------------------------------------------------------------------
// Application local grab.
//
// Careful.  Tk 8.0.5 behaviour.
//
// 1) The grab is not done until events in the Tcl event queue are
//    processed.  Other windows can hold the focus until the grab dialog
//    is mapped for the first time.
//
// 2) Does not grab key events.  Also the keyboard focus can't be given
//    to any window of the application except the one with the grab,
//    unless the Tk focus -force command is used.
//    The window manager will highlight other windows but they do not
//    get the Tk focus or any key events.
//    If the grabbing window is destroyed and another window already has
//    been given the focus by the window manager, that window still will not
//    get key events until you explicitly give the focus to it again.
//
void WinSys::grab_events(Widget dialog)
{
  tcl_command(wsp, "grab set %s", path(dialog));

  //
  // Would like dialog with a grab to stay on top.
  // Haven't found a satisfactory way to do this in Tk.
  // The following handler at least raises the grab window
  // if the user clicks on any Sparky window.
  //
  add_event_handler(wsp, dialog, ButtonPressMask, NULL,
		    raise_modal_widget_cb, wsp);
}

// ----------------------------------------------------------------------------
//
void WinSys::ungrab_events(Widget dialog)

{
  remove_event_handler(wsp, dialog, ButtonPressMask, NULL,
		       raise_modal_widget_cb, wsp);
  tcl_command(wsp, "grab release %s", path(dialog));

  //
  // If the focus has been switched to a non-grabbing top level window in the
  // application by clicking on it, the window manager will highlight that
  // window.  But Tk will not have given the keyboard focus to it as noted
  // in the comments on grab_events.  The Tk focus command with no arguments
  // will report that no window has the focus.  After the grab is released
  // Tk still doesn't give any window the keyboard focus even though the
  // window manager shows a top level highlighted indicating it has the focus.
  // This is a Tk 8.0.5 bug.  To work around it I tried giving the grabbing
  // window the focus as we release the grab.  That failed because Tk doesn't
  // believe the application has the window manager focus, so it will not
  // tell the window manager to switch focus to the grab dialog.
  //
}

// ----------------------------------------------------------------------------
//
void WinSys::set_sensitive(Widget w, bool sensitive)
{
  Stringy type = Tk_Class(window(w));
  if (type == "Button" ||
      type == "Checkbutton" ||
      type == "Radiobutton" ||
      type == "Menubutton" ||
      type == "Entry")
    tcl_command(wsp, "%s configure -state %s",
		path(w), (sensitive ? "normal" : "disabled"));
  else
    {
      List children = child_widgets(wsp, w);
      for (int k = 0 ; k < children.size() ; ++k)
	set_sensitive((Widget) children[k], sensitive);
    }
}

// ----------------------------------------------------------------------------
//
Stringy WinSys::read_application_resource(Widget w,
					  const Stringy &resource_name,
					  const Stringy &class_name,
					  const Stringy &default_value)
{
  if (tcl_command(wsp, "option get %s %s %s",
		  path(w), resource_name, class_name))
    {
      Stringy value = command_result(wsp);
      if (! value.is_empty())
	return value;
    }
  return default_value;
}

// ----------------------------------------------------------------------------
//
#define MAX_RGB_COLOR 65535

bool WinSys::color_rgb(const Color &color, double *r, double *g, double *b)
{
  Tk_Uid cname = Tk_GetUid(color.name().cstring());
  Widget w = main_widget();
  XColor *clr = Tk_GetColor(wsp->tcl_interp, window(w), cname);
  if (clr == NULL)
    {
      Tcl_ResetResult(wsp->tcl_interp);
      return false;
    }
  *r = (double) clr->red / MAX_RGB_COLOR;
  *g = (double) clr->green / MAX_RGB_COLOR;
  *b = (double) clr->blue / MAX_RGB_COLOR;
  Tk_FreeColor(clr);
  return true;
}

// ----------------------------------------------------------------------------
//
class Input_Callback
{
public:
  Input_Callback(FILE *fp, CB_Func, CB_Data);
  ~Input_Callback();

  FILE *fp;
  CB_Func cb;
  CB_Data cb_data;

  friend void ic_input_cb(ClientData, int);
};

// ----------------------------------------------------------------------------
//
class Input_Callback_Manager
{
public:
  ~Input_Callback_Manager();

  void add_callback(FILE *fp, CB_Func cb, CB_Data cb_data);
  void remove_callback(FILE *fp, CB_Func cb, CB_Data cb_data);

private:
  List icblist;

  Input_Callback *find(FILE *fp, CB_Func, CB_Data);
};

// ----------------------------------------------------------------------------
//
Input_Callback_Manager::~Input_Callback_Manager()
{
  List copy = icblist;
  for (int ii = 0 ; ii < copy.size() ; ++ii)
    delete (Input_Callback *) copy[ii];
  icblist.erase();
}

// ----------------------------------------------------------------------------
//
void Input_Callback_Manager::add_callback(FILE *fp,
					  CB_Func cb, CB_Data cb_data)
{
  if (find(fp, cb, cb_data) == NULL)
    {
      Input_Callback *icb = new Input_Callback(fp, cb, cb_data);
      icblist.append(icb);
    }
}

// ----------------------------------------------------------------------------
//
void Input_Callback_Manager::remove_callback(FILE *fp,
					     CB_Func cb, CB_Data cb_data)
{
  Input_Callback *icb = find(fp, cb, cb_data);
  if (icb)
    {
      icblist.erase(icb);
      delete icb;
    }
}

// ----------------------------------------------------------------------------
//
Input_Callback::Input_Callback(FILE *fp, CB_Func cb, CB_Data cb_data)
{
  this->fp = fp;
  this->cb = cb;
  this->cb_data = cb_data;

  create_tcl_file_handler(fp, TCL_READABLE, ic_input_cb, this);
}

// ----------------------------------------------------------------------------
//
Input_Callback::~Input_Callback()
{
  remove_tcl_file_handler(fp, ic_input_cb, this);
  fp = NULL;
  cb = NULL;
  cb_data = NULL;
}

// ----------------------------------------------------------------------------
//
extern "C"
{
static void ic_input_cb(ClientData self, int)
{
  Input_Callback *icb = (Input_Callback *) self;

  if (is_data_available(icb->fp))
    icb->cb(icb->cb_data);
}
}

// ----------------------------------------------------------------------------
//
Input_Callback *Input_Callback_Manager::find(FILE *fp,
					     CB_Func cb, CB_Data cb_data)
{
  for (int ici = 0 ; ici < icblist.size() ; ++ici)
    {
      Input_Callback *icb = (Input_Callback *) icblist[ici];
      if (icb->fp == fp && icb->cb == cb && icb->cb_data == cb_data)
	return icb;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
void WinSys::add_input_callback(FILE *fp, CB_Func cb, CB_Data cb_data)
  { wsp->input_callbacks->add_callback(fp, cb, cb_data); }
void WinSys::remove_input_callback(FILE *fp, CB_Func cb, CB_Data cb_data)
  { wsp->input_callbacks->remove_callback(fp, cb, cb_data); }

// ----------------------------------------------------------------------------
//
bool WinSys::shift_modifier(CB_Data e)
{
  XEvent *event = (XEvent *) e;
  return ((event->type == ButtonPress && (event->xbutton.state & ShiftMask)) ||
	  (event->type == MotionNotify && (event->xmotion.state & ShiftMask)));
}

// ----------------------------------------------------------------------------
//
bool WinSys::event_xy(CB_Data e, int *x, int *y)
{
  XEvent *event = (XEvent *) e;

  if (event->type == MotionNotify)
    { *x = event->xmotion.x; *y = event->xmotion.y; }
  else if (event->type == EnterNotify || event->type == LeaveNotify)
    { *x = event->xcrossing.x; *y = event->xcrossing.y; }
  else if (event->type == ButtonPress || event->type == ButtonRelease)
    { *x = event->xbutton.x; *y = event->xbutton.y; }
  else
    return false;

  return true;
}

// ----------------------------------------------------------------------------
//
class Scaled_Font
{
public:
  Scaled_Font(const Stringy &family, int pixel_height, Tk_Font font);
  ~Scaled_Font();

  Stringy family;
  int pixel_height;
  Tk_Font font;
  int reference_count;
};

// ----------------------------------------------------------------------------
//
class Scaled_Font_Manager
{
public:
  Scaled_Font_Manager(WinSysP *);
  ~Scaled_Font_Manager();

  Tk_Font find(const Stringy &family, int pixel_height);
  void release(Tk_Font font);

private:
  WinSysP *wsp;
  List fonts;

  Scaled_Font *lookup(Tk_Font font);
  Scaled_Font *lookup(const Stringy &family, int pixel_height);
  Scaled_Font *load(const Stringy &family, int pixel_height);
};

// ----------------------------------------------------------------------------
//
Scaled_Font_Manager::Scaled_Font_Manager(WinSysP *wsp)
{
  this->wsp = wsp;
}

// ----------------------------------------------------------------------------
//
Scaled_Font_Manager::~Scaled_Font_Manager()
{
  for (int fi = 0 ; fi < fonts.size() ; ++fi)
    delete (Scaled_Font *) fonts[fi];
  fonts.erase();
}

// ----------------------------------------------------------------------------
//
void Scaled_Font_Manager::release(Tk_Font font)
{
  Scaled_Font *sf = lookup(font);
  if (sf == NULL)
    fatal_error("Scaled_Font_Manager::release(): Bad font.\n");

  sf->reference_count -= 1;
  if (sf->reference_count == 0)
    {
      fonts.erase(sf);
      delete sf;
    }
}

// ----------------------------------------------------------------------------
//
Tk_Font Scaled_Font_Manager::find(const Stringy &family, int pixel_height)
{
  Scaled_Font *sf = lookup(family, pixel_height);

  if (sf)
    sf->reference_count += 1;
  else
    sf = load(family, pixel_height);

  return (sf ? sf->font : NULL);
}

// ----------------------------------------------------------------------------
//
Scaled_Font *Scaled_Font_Manager::lookup(const Stringy &family,
					 int pixel_height)
{
  for (int fi = 0 ; fi < fonts.size() ; ++fi)
    {
      Scaled_Font *sf = (Scaled_Font *) fonts[fi];
      if (sf->pixel_height == pixel_height && sf->family == family)
	return sf;
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
Scaled_Font *Scaled_Font_Manager::lookup(Tk_Font font)
{
  for (int fi = 0 ; fi < fonts.size() ; ++fi)
    {
      Scaled_Font *sf = (Scaled_Font *) fonts[fi];
      if (sf->font == font)
	return sf;
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
Scaled_Font *Scaled_Font_Manager::load(const Stringy &family, int pixel_height)
{
  Widget w = main_widget(wsp);
  Stringy name = quoted_word(family) + " -" + Stringy(pixel_height);
  Tk_Font font = Tk_GetFont(wsp->tcl_interp, window(w), name.cstring());
  if (font == NULL)
    report_tcl_error(wsp, "in Scaled_Font_Manager::load()");

  Scaled_Font *sf = lookup(font);
  if (sf)
    sf->reference_count += 1;
  else if (font)
    {
      sf = new Scaled_Font(family, pixel_height, font);
      fonts.append(sf);
    }

  return sf;
}

// ----------------------------------------------------------------------------
//
Scaled_Font::Scaled_Font(const Stringy &family, int pixel_height, Tk_Font font)
{
  this->family = family;
  this->pixel_height = pixel_height;
  this->font = font;
  reference_count = 1;
}

// ----------------------------------------------------------------------------
//
Scaled_Font::~Scaled_Font()
{
  Tk_FreeFont(font);
}

// ----------------------------------------------------------------------------
// Set the window's bit gravity to ForgetGravity
//
static void forget_bit_gravity(Widget w)
{
  XSetWindowAttributes attributes;

  attributes.bit_gravity = ForgetGravity;

  Tk_ChangeWindowAttributes(window(w), CWBitGravity, &attributes);
}

// ----------------------------------------------------------------------------
//
bool WinSys::exposure_region(CB_Data xevent, int *x, int *y, int *w, int *h)
{
  return exposed_region((XEvent *) xevent, x, y, w, h);
}
  
// ----------------------------------------------------------------------------
//
static bool exposed_region(XEvent *event, int *x, int *y, int *w, int *h)
{
  //
  // No repaint for NoExpose events
  //
  if (event->type == Expose || event->type == GraphicsExpose)
    {
      XExposeEvent *eevent = &event->xexpose;
      *x = eevent->x;
      *y = eevent->y;
      *w = eevent->width;
      *h = eevent->height;
      return true;
    }

  return false;
}

// ----------------------------------------------------------------------------
//
bool WinSys::key_pressed(CB_Data e, char *c)
  { return x_key((XEvent *) e, c); }
bool WinSys::function_key_pressed(CB_Data e, int *f, bool *shifted)
  { return x_function_key((XEvent *) e, f, shifted); }

// ----------------------------------------------------------------------------
//
static void xor_gc_mode(Display *display, GC gc,
			unsigned long *prev_foreground, int *prev_gc_mode)
{
  XGCValues gc_values;
  XGetGCValues(display, gc, GCForeground | GCBackground | GCFunction,
	       &gc_values);
  *prev_foreground = gc_values.foreground;
  *prev_gc_mode = gc_values.function;

  unsigned long fg = gc_values.foreground;
  unsigned long bg = gc_values.background;
  XSetForeground(display, gc, fg ^ bg);
  XSetFunction(display, gc, GXxor);
}

// ----------------------------------------------------------------------------
//
static void restore_gc_mode(Display *display, GC gc,
			    unsigned long foreground, int gc_mode)
{
  XSetForeground(display, gc, foreground);
  XSetFunction(display, gc, gc_mode);
}

// ----------------------------------------------------------------------------
//
static GC create_gc(Display *display, Window w)
  { return XCreateGC(display, w, 0, NULL); }
static void free_gc(Display *display, GC gc)
  { XFreeGC(display, gc); }

// ----------------------------------------------------------------------------
// Private member for Drawing_Window class that I want to hide in interface.
//
class DrawWinP
{
public:
  DrawWinP(WinSysP *, Widget parent, const Stringy &name);
  ~DrawWinP();

  WinSysP *wsp;
  Widget w;
  Display *display;
  Window win;
  GC gc;
  Color gc_color;	// Avoid consecutive duplicate color lookups.
  Tk_Font font;
  int font_pixel_height;
};

// ----------------------------------------------------------------------------
//
DrawWinP::DrawWinP(WinSysP *wsp, Widget parent, const Stringy &name)
{
  this->wsp = wsp;
  this->w = wsp->ws.create_form(parent, name);

  //
  // Set background to empty string no background painting is done.
  // Otherwise Tk repaints background (maybe with an idle callback)
  // after I redraw window.
  //
  tcl_command(wsp, "%s configure -background %s", path(w), Stringy());

  forget_bit_gravity(w);		// clear entire window on resize

  //
  // Make sure window exists in case client tries to draw in window
  // before exposure event received.  This happens for edge panels where
  // a translation is done immediately w/o going through X event queue.
  // The DrawWinP class constructor makes the window exists but
  // only one is used for all edge panels.
  //
  make_window_exist(w);
  this->display = widget_display(w);
  this->win = x_window(w);

  //
  // Create the graphics context
  //
  this->gc = create_gc(display, win);
  XColor *bg_color = drawing_area_background(wsp, w);
  if (bg_color)
    XSetBackground(display, gc, bg_color->pixel);

  XGCValues gcvalues;
  gcvalues.graphics_exposures = true;
  XChangeGC(display, gc, GCGraphicsExposures, &gcvalues);

  set_background_erase(w, gc, (bg_color ? bg_color->pixel : 0));

  this->gc_color = "";
  this->font = NULL;
  this->font_pixel_height = 0;
}

// ----------------------------------------------------------------------------
// Canvas widget must be deleted by owning dialog.
//
DrawWinP::~DrawWinP()
{
  if (font_pixel_height != 0)
    wsp->fonts->release(font);
  else if (font != NULL)
    Tk_FreeFont(font);
  free_gc(display, gc);
}

// ----------------------------------------------------------------------------
//
Drawing_Window::Drawing_Window(WinSys &ws, Widget parent,
				       const Stringy &name)
{
  this->p = new DrawWinP(ws.winsysp(), parent, name);
}

// ----------------------------------------------------------------------------
//
Drawing_Window::~Drawing_Window()
{
  delete this->p;
}

// ----------------------------------------------------------------------------
//
Widget Drawing_Window::widget() const
  { return p->w; }

// ----------------------------------------------------------------------------
//
void Drawing_Window::draw_string(int x, int y, const Stringy &string)
{
  if (p->font != NULL)
    Tk_DrawChars(p->display, p->win, p->gc, p->font,
		 string.cstring(), string.length(), x, y);
}

// ----------------------------------------------------------------------------
//
void Drawing_Window::draw_line(int x1, int y1, int x2, int y2,
				       bool xor_mode)
{
  if (xor_mode)
    {
      unsigned long prev_foreground;
      int prev_gc_mode;
      xor_gc_mode(p->display, p->gc, &prev_foreground, &prev_gc_mode);
      XDrawLine(p->display, p->win, p->gc, x1, y1, x2, y2);
      restore_gc_mode(p->display, p->gc, prev_foreground, prev_gc_mode);
    }
  else
    XDrawLine(p->display, p->win, p->gc, x1, y1, x2, y2);
}

// ----------------------------------------------------------------------------
//
void Drawing_Window::draw_point(int x, int y)
{
  //
  // Tk for MS Windows doesn't support XDrawPoint()
  // Here I use that the end point of lines are not drawn.
  //
  XDrawLine(p->display, p->win, p->gc, x, y, x+1, y);
}

// ----------------------------------------------------------------------------
//
void Drawing_Window::draw_rectangle(int x, int y,
					    unsigned int w, unsigned int h,
					    bool xor_mode)
{
  if (xor_mode)
    {
      unsigned long prev_foreground;
      int prev_gc_mode;
      xor_gc_mode(p->display, p->gc, &prev_foreground, &prev_gc_mode);
      XDrawRectangle(p->display, p->win, p->gc, x, y, w, h);
      restore_gc_mode(p->display, p->gc, prev_foreground, prev_gc_mode);
    }
  else
    XDrawRectangle(p->display, p->win, p->gc, x, y, w, h);
}

// ----------------------------------------------------------------------------
//
void Drawing_Window::fill_rectangle(int x, int y,
				    unsigned int w, unsigned int h)
{
  XFillRectangle(p->display, p->win, p->gc, x, y, w, h);
}

// ----------------------------------------------------------------------------
//
void Drawing_Window::fill_triangle(int x1, int y1, int x2, int y2,
				   int x3, int y3)
{
  XPoint points[3];

  points[0].x = x1;
  points[0].y = y1;
  points[1].x = x2;
  points[1].y = y2;
  points[2].x = x3;
  points[2].y = y3;

  XFillPolygon(p->display, p->win, p->gc,
	       points, 3, Convex, CoordModeOrigin);
}

// ----------------------------------------------------------------------------
//
void Drawing_Window::draw_arc(int x, int y,
			      unsigned int w, unsigned int h,
			      int a1, int a2)
{
  XDrawArc(p->display, p->win, p->gc, x, y, w, h, a1, a2);
}

// ----------------------------------------------------------------------------
// This is a very inefficient text rotation routine.
// It draws horizontal text into a pixmap on the server.
// Then brings it into an XImage on the client.
// Rotates it and sends it back to the server for display.
//
void Drawing_Window::draw_vertical_string(int x, int y,
					  const Stringy &string,
					  bool up)
{
  draw_vertical_text(p->w, p->gc, p->font, x, y, string, up);
}

// ----------------------------------------------------------------------------
//
bool Drawing_Window::set_font_height(int pixel_height)
{
  if (pixel_height != p->font_pixel_height)
    {
      tcl_command(p->wsp, "font actual scalableFont -family");
      Stringy family = command_result(p->wsp);
      Tk_Font font = p->wsp->fonts->find(family, pixel_height);
      if (font)
	{
	  if (p->font_pixel_height != 0)
	    p->wsp->fonts->release(p->font);
	  p->font = font;
	  p->font_pixel_height = pixel_height;
	  XSetFont(p->display, p->gc, Tk_FontId(font));
	  return true;
	}
    }
  return false;
}

// ----------------------------------------------------------------------------
//
void Drawing_Window::text_size(const Stringy &string,
			       int *w, int *ascent, int *descent)
{
  ::text_size(p->font, string, w, ascent, descent);
}

// ----------------------------------------------------------------------------
//
static void text_size(Tk_Font font, const Stringy &string,
		      int *w, int *ascent, int *descent)
{
  if (font == NULL)
    *w = *ascent = *descent = 0;
  else
    {
      Tk_FontMetrics metrics;
      Tk_GetFontMetrics(font, &metrics);
      *ascent = metrics.ascent;
      *descent = metrics.descent;
      *w = Tk_TextWidth(font, string.cstring(), string.length());
    }
}

// ----------------------------------------------------------------------------
//
int Drawing_Window::font_height()
{
  if (p->font == NULL)
    return 0;

  Tk_FontMetrics metrics;
  Tk_GetFontMetrics(p->font, &metrics);
  return metrics.linespace;
}

// ----------------------------------------------------------------------------
//
bool Drawing_Window::use_label_font()
{
  Tk_Font font = label_font(p->wsp, widget());
  if (font == NULL)
    return false;

  if (p->font_pixel_height != 0)
    p->wsp->fonts->release(p->font);
  XSetFont(p->display, p->gc, Tk_FontId(font));
  p->font = font;
  p->font_pixel_height = 0;
  return true;
}

// ----------------------------------------------------------------------------
//
static Tk_Font label_font(WinSysP *wsp, Widget w)
{
  Widget label = wsp->ws.create_label(w, "findFont");
  Stringy font_spec = get_option(wsp, label, "-font");
  wsp->ws.delete_widget(label);

  Tk_Font font = Tk_GetFont(wsp->tcl_interp, window(w), font_spec.cstring());
  if (font == NULL)
    report_tcl_error(wsp, "getting font " + font_spec + "\n");

  return font;
}

// ----------------------------------------------------------------------------
//
class Color_Manager
{
public:
  Color_Manager(WinSysP *);
  ~Color_Manager();
  XColor *x_color(const Color &color);
private:
  WinSysP *wsp;
  Table string_to_xcolor;
  void free_allocated_colors();
};

// ----------------------------------------------------------------------------
//
Color_Manager::Color_Manager(WinSysP *wsp) :
  string_to_xcolor(equal_strings, hash_string)
{
  this->wsp = wsp;
}

// ----------------------------------------------------------------------------
//
Color_Manager::~Color_Manager()
{
  free_allocated_colors();
}

// ----------------------------------------------------------------------------
// Find pixel and rgb values for a color.  A colormap allocation for the main
// window is done if needed.  The colormap entries are never freed.
//
XColor *Color_Manager::x_color(const Color &color)
{
  TableData xcolor;
  if (string_to_xcolor.find((TableKey) &color.name(), &xcolor))
    return (XColor *) xcolor;

  Tk_Uid cname_id = Tk_GetUid(color.name().cstring());
  XColor *c = Tk_GetColor(wsp->tcl_interp, window(main_widget(wsp)), cname_id);
  if (c == NULL)
    {
      report_tcl_error(wsp, "getting color " + color.name() + "\n");
      return NULL;
    }

  string_to_xcolor.insert((TableKey) new Stringy(color.name()), c);

  return c;
}

// ----------------------------------------------------------------------------
//
void Color_Manager::free_allocated_colors()
{
  List cnames = string_to_xcolor.keys();
  free_string_list_entries(cnames);

  List xcolors = string_to_xcolor.values();
  for (int k = 0 ; k < xcolors.size() ; ++k)
    Tk_FreeColor((XColor *) xcolors[k]);

  string_to_xcolor.erase();
}

// ----------------------------------------------------------------------------
//
static XColor *drawing_area_background(WinSysP *wsp, Widget w)
{
  Stringy colorname = wsp->ws.read_application_resource(w, "drawingBackground",
						"DrawingBackground", "");
  if (colorname.is_empty())
    colorname = wsp->ws.read_application_resource(w, "background", "Background", "");
  if (colorname.is_empty())
    return NULL;

  return wsp->colors->x_color(colorname);
}

// ----------------------------------------------------------------------------
//
void Drawing_Window::set_foreground(const Color &color)
{
  if (color != p->gc_color)
    {
      XColor *c = p->wsp->colors->x_color(color);
      if (c)
	{
	  p->gc_color = color;
	  XSetForeground(p->display, p->gc, c->pixel);
	}
    }
}

// ----------------------------------------------------------------------------
//
void Drawing_Window::translate_contents(int dx, int dy)
{
  if (dx == 0 && dy == 0)
    return;

  translate_window_contents(p->w, p->gc, dx, dy);
}

// ----------------------------------------------------------------------------
// Clear window and generate exposure events.
//
void Drawing_Window::clear_window()
  { clear_area(0, 0, 0, 0); }
void Drawing_Window::clear_area(int x, int y, int w, int h)
  { eventually_redraw(p->w, x, y, w, h); }
void Drawing_Window::draw_background(int x, int y, int w, int h)
{
  ::draw_background(p->display, p->win, p->gc, x, y, w, h);
}

// ----------------------------------------------------------------------------
//
void Drawing_Window::request_backing_store()
{
  XSetWindowAttributes attr;

  attr.backing_store = Always;

  Tk_ChangeWindowAttributes(window(p->w), CWBackingStore, &attr);
}

// ----------------------------------------------------------------------------
//
WinSysP *WinSys::winsysp()
  { return wsp; }

// ----------------------------------------------------------------------------
//
WinSysP::WinSysP(WinSys &winsys, Tcl_Interp *tcl_interp) : ws(winsys)
{
  this->own_tcl_interp = (tcl_interp == NULL);

  if (tcl_interp == NULL)
    {
      tcl_interp = Tcl_CreateInterp();
      if (Tcl_Init(tcl_interp) == TCL_ERROR)
	fatal_error("WinSysP::WinSysP(): %s\n",
		    Tcl_GetStringResult(tcl_interp));
    }

  this->tcl_interp = tcl_interp;
  this->debug_tcl = (getenv("SPARKY_DEBUG_TCL") != NULL);
  this->unique_name_count = 1;
  this->last_double_click_time = 0;
  this->exit_requested = false;

  this->work_procs = new Work_Proc_Manager();
  this->event_callbacks = new Event_Callback_Table();
  this->timer_callbacks = new Timer_Callback_List();
  this->pause_callbacks = new Pointer_Pause_List();
  this->variable_callbacks = new Variable_Callback_Manager();
  this->scroll_callbacks = new Scroll_Callback_Table();
  this->input_callbacks = new Input_Callback_Manager();
  this->fonts = new Scaled_Font_Manager(this);
  this->colors = new Color_Manager(this);
}

// ----------------------------------------------------------------------------
//
WinSysP::~WinSysP()
{
  ws.delete_widget(main_widget(this));

  delete work_procs;		work_procs = NULL;
  delete event_callbacks;	event_callbacks = NULL;
  delete timer_callbacks;	timer_callbacks = NULL;
  delete pause_callbacks;	pause_callbacks = NULL;
  delete variable_callbacks;	variable_callbacks = NULL;
  delete scroll_callbacks;	scroll_callbacks = NULL;
  delete input_callbacks;	input_callbacks = NULL;
  delete fonts;			fonts = NULL;
  delete colors;		colors = NULL;

  if (own_tcl_interp)
    {
      Tcl_DeleteInterp(tcl_interp);	tcl_interp = NULL;
      // Tcl_Finalize();
    }
}

