// ----------------------------------------------------------------------------
// Mapping of axes of one spectrum to axes of another spectrum.
//

#include "axismap.h"
#include "spectrum.h"		// Use Spectrum
#include "utility.h"		// Use fatal_error()

// ----------------------------------------------------------------------------
//
Axis_Map::Axis_Map() {}

// ----------------------------------------------------------------------------
//
int Axis_Map::map(int axis) const
{
  if (axis < 0 || axis >= to_axis.dimension())
    fatal_error("Axis_Map::map(): axis out of range.\n");

  return to_axis[axis];
}

// ----------------------------------------------------------------------------
//
int Axis_Map::invert(int axis) const
{
  for (int a = 0 ; a < to_axis.dimension() ; ++a)
    if (to_axis[a] == axis)
      return a;

  fatal_error("Axis_Map::invert(): axis out of range.\n");

  return -1;	// Never reached.  Supresses compilation warning.
}

// ----------------------------------------------------------------------------
//
void Axis_Map::set_map(const IPoint &to_axis)
{
  this->to_axis = to_axis;
}

// ----------------------------------------------------------------------------
//
SPoint Axis_Map::map(const SPoint &p) const
{
  SPoint top(p.dimension());

  for (int a = 0 ; a < p.dimension() ; ++a)
    top[map(a)] = p[a];

  return top;
}

// ----------------------------------------------------------------------------
//
SPoint Axis_Map::invert(const SPoint &p) const
{
  SPoint fromp(p.dimension());

  for (int a = 0 ; a < p.dimension() ; ++a)
    fromp[a] = p[map(a)];

  return fromp;
}

// ----------------------------------------------------------------------------
//
SRegion Axis_Map::map(const SRegion &r) const
  { return SRegion(map(r.min), map(r.max)); }
SRegion Axis_Map::invert(const SRegion &r) const
  { return SRegion(invert(r.min), invert(r.max)); }

// ----------------------------------------------------------------------------
//
bool identity_axis_map(Spectrum *from, Spectrum *to, Axis_Map *axismap)
{
  if (from->dimension() != to->dimension())
    return false;

  IPoint to_axis(from->dimension());
  for (int a = 0 ; a < from->dimension() ; ++a)
    if (from->nucleus_type(a) != to->nucleus_type(a))
      return false;
    else
      to_axis[a] = a;

  if (axismap)
    axismap->set_map(to_axis);

  return true;
}

// ----------------------------------------------------------------------------
//
bool unique_axis_map(Spectrum *from, Spectrum *to, Axis_Map *axismap)
{
  if (from->dimension() != to->dimension())
    return false;

  IPoint to_axis(from->dimension());
  for (int a = 0 ; a < from->dimension() ; ++a)
    {
      int from_axis;
      Stringy nucleus = from->nucleus_type(a);
      if (!from->is_nucleus_unique(nucleus, &from_axis) ||
	  !to->is_nucleus_unique(nucleus, &to_axis[a]))
	return false;
    }

  if (axismap)
    axismap->set_map(to_axis);

  return true;
}
