/*
 * integrate.c:		Both linefitting and boxing integration methods.
 *
 * The linefitting methods are implemented mainly in calc/Linefit.C, with
 * the linefitting routine itself in LinefitVolume::minimize().
 *
 */
#include <float.h>		// use FLT_MAX
#include <math.h>		// use sqrt(), fabs()

#include "dataregion.h"		// Use Marked_Region
#include "integrate.h"
#include "linefit.h"
#include "memalloc.h"		// use new()
#include "num.h"
#include "ornament.h"
#include "peak.h"
#include "peakgp.h"
#include "reporter.h"		// use Reporter
#include "spectrum.h"		// Use Spectrum
#include "undo.h"
#include "wait.h"		// use Wait

#define MAXPEAKS	100

// ----------------------------------------------------------------------------
// Minimum positive and negative data heights to use in fitting.
//
struct Integration_Thresholds
{
  double negative;
  double positive;
};

static bool peak_grouping(Peak *peak, const List &peaks,
			  bool contour_grouping, bool distance_grouping,
			  const Integration_Thresholds &threshold,
			  Marked_Region *mr, List *overlapped, Wait *wait,
			  Reporter &);
static List peaks_above_threshold(const List &peaks,
				  const Integration_Thresholds &threshold);
static bool peak_above_threshold(Peak *pk, const Integration_Thresholds &t);
static List peaks_in_region(const List &peaks, const SRegion &region);

// ----------------------------------------------------------------------------
//
const char *Integration_Method_Short_Names[] =
  {"", "ga", "lo", "bx", "el", "ma", NULL};
const char *Integration_Method_Names[] =
  {"none", "gaussian", "lorentzian", "box", "ellipse", "manual", NULL};

// ----------------------------------------------------------------------------
//
Integration_Parameters::Integration_Parameters(SpectrumData *sp)
{
  integration_method = INTEGRATE_GAUSSIAN;
  contour_grouping = true;
  distance_grouping = false;
  maxiterations = 10000;
  tolerance = .001;
  allow_motion = true;
  adjust_linewidths = true;
  contoured_data = true;
  rectangle_data = false;
  fit_baseline = false;
  subtract_peaks = false;

  int dim = (sp == NULL ? 2 : sp->dimension());
  grouping_dist = SPoint(dim);
  motion_range = SPoint(dim);
  linewidth_range = SRegion(dim);

  if (sp)
    for (int a = 0 ; a < dim ; ++a)
      {
	double f = sp->nucleus_ppm_factor(a);
	grouping_dist[a] = sp->scale(30.0 * f, a, HZ, PPM);
	motion_range[a] = .01 * f;
	linewidth_range.min[a] = sp->scale(2.0, a, HZ, PPM);
	linewidth_range.max[a] = sp->scale(200.0, a, HZ, PPM);
      }
  else
    for (int a = 0 ; a < dim ; ++a)
      {
	grouping_dist[a] = .1;
	motion_range[a] = .01;
	linewidth_range.min[a] = 0;
	linewidth_range.max[a] = 1;
      }
}

/*
 * Fill weights array <wt> with constant fill value <fill>.
 */
static void
_fill_weights(
	double		*wt,			/* array of weights */
	int	w,
	int	h,			/* array size */
	double		fill)			/* fill value */
{
	int	col;

	for ( ; h > 0; h--)
		for (col = 0; col < w; col++)
			*wt++ = fill;
}

#define DY	0.1
#define IDY	10

/*
 * Fill the weights array <wt> with a mask that is the fractional area that
 * each data element is within the *elliptical* region
 * (<cx> - <w>/2, <cy> - <h>/2), (<cx> + <w>/2, <cy> + <h>/2).
 */
static void
_set_weights_ellipse(
	double		*wt,		/* weight array */
	int		iw,
	int		ih,		/* size of data box */
	double		cx,
	double		cy,		/* center of ellipse in the data box */
	double		w,
	double		h)		/* size of actual box */
{
	int	row, col, iy;
	double		x, y;
	double		h2, det;
	double		len;

	/*
	 * Initialized the weight array with zero.
	 */
	_fill_weights(wt, iw, ih, 0.0);

	if (h <= DY)
		return;

	h2 = 4.0 / (h * h);	/* h2 == (1 / b)^2 == 2 / h * 2 / h */

	/*
	 * Step up the rows by a fraction of the cell size, evaluating
	 * the endpoints of the ellipse.  If there are valid endpoints,
	 * increment along them, incrementing the contents of each
	 * cell by the amount of the ellipse in the cell.
	 */
	for (iy = 0; iy < h * IDY; iy++) {
		row = iy / IDY;

		/*
		 * Set y to be half way up the fractional cell, and
		 * compute the argument of the square root.  If it is
		 * positive, there are two endpoints to the ellipse at
		 * this height.
		 */
		y = (double) iy / IDY + 0.5;
		det = 1 - (y - cy) * (y - cy) * h2;
		if (det >= 0) {
			len = w * sqrt(det);
			x = cx - len / 2;
			col = (int) x;

			/*
			 * Set the first data point.
			 */
			wt[row * iw + col] += DY * min(col + 1 - x, len);
			len -= min(col + 1 - x, 1.0);

			/*
			 * Set the rest of the data points.
			 */
			for ( ; len >= 0; ) {
				wt[row * iw +  ++col] += DY * min(1.0, len);
				len -= 1.0;
			}
		}
	}
}

/*
 * Fill the weights array <wt> with a mask that is the fractional area that
 * each data element is within the *rectangular* region
 * (<cx> - <w>/2, <cy> - <h>/2), (<cx> + <w>/2, <cy> + <h>/2).
 */
static void
_set_weights_box(
	double		*wt,
	int		iw,
	int		ih,	// size of data box
	double		cx,
	double		cy,	// center of actual box (0,0) is data box origin
	double		w,
	double		h)	// size of actual box
{
	int	row, col;
	double		dblx, dbly, dtrx, dtry;
	double		dy;

	/*
	 * Compute the border around the box.
	 */
	dblx = .5 + cx - w/2;
	dbly = .5 + cy - h/2;
	dtrx = .5 + (iw-1) - (cx + w/2);
	dtry = .5 + (ih-1) - (cy + h/2);

	/*
	 * Fill in the weights.
	 */
	for (row = 0; row < ih; row++) {
		dy = (row == 0) ? dbly : (row == ih - 1) ? dtry : 1.0;
		*wt++ = dy * dblx;
		for (col = 1; col < iw - 1; col++)
			*wt++ = dy;
		*wt++ = dy * dtrx;
	}
}


/*
 * Integrate overlapped peaks using linefitting, returning true if successful
 */
static bool integrate_by_fitting(const List &		peaks,
				 const Marked_Region &	data_mask,
				 Integration_Method	method,
				 Wait *			wait,
				 Reporter &		rr)
{
  Spectrum *sp = data_mask.spectrum();
  sp->mIntegrate.integration_method = method;

  return linefit(peaks, data_mask, sp->mIntegrate, wait, rr);
}


/*
 * Integrate an isolated peak by one of the boxing methods, returning true
 * if successful.
 */
static bool integrate_by_boxing(Peak *peak,
				const SRegion &region,
				Integration_Method method,
				Reporter &rr)
{
	if (region.no_interior())
	  {
	    rr.message("Can't integrate by box or ellipse method without\n"
		       "dragging a rectangle around the peak.\n");
	    return false;
	  }

	Spectrum *sp = peak->spectrum();
	if (sp->dimension() != 2)
	  {
	    rr.message("Can't integrate %d-D peak by boxing.\n",
		       sp->dimension());
	    return false;
	  }

	/*
	 * Allocate memory for the data and the weights.
	 */
	SRegion index_region = sp->map(region, PPM, INDEX);
	IRegion grid_region = index_region.rounded();
	int w = grid_region.size(0);
	int h = grid_region.size(1);
	int volume = w * h;
	float	*dat = new float [volume];
	double	*wts = new double [volume];

	/*
	 * Get the data from the data file
	 */
	if (!sp->heights_for_region(grid_region, dat)) {
	  rr.message("Can't read data file\n");
	  return false;
	}

	/*
	 * Set up the appropriate weights array
	 */

	if (method == INTEGRATE_BOX)
	  _set_weights_box(wts, w, h,
			   index_region.center(0) - grid_region.min[0],
			   index_region.center(1) - grid_region.min[1],
			   index_region.size(0), index_region.size(1));
	else if (method == INTEGRATE_ELLIPSE)
	  _set_weights_ellipse(wts, w, h,
			       index_region.center(0) - grid_region.min[0],
			       index_region.center(1) - grid_region.min[1],
			       index_region.size(0), index_region.size(1));
	else
	  _fill_weights(wts, w, h, 1.0);

	/*
	 * Integrate the region by summing the intensity (height) at each
	 * point weighted by the weight array at each point.
	 *
	 * Also compute the center of mass, which is the absolute intensity
	 * weighted by the weight array and the column or row number.
	 */
	float *dp = dat;
	double *wp = wts;
	double abs_integral = 0;
	double integral = 0;
	double comx = 0, comy = 0;
	for (int row = 0; row < h; row++) {
		for (int col = 0; col < w; col++) {
			double weighted_height = (*dp++) * (*wp++);
			double abs_height = fabs(weighted_height);
			integral += weighted_height;
			abs_integral += abs_height;
			comx += col * abs_height;
			comy += row * abs_height;
		}
	}
	comx = (abs_integral == 0 ? 0 : comx / abs_integral);
	comy = (abs_integral == 0 ? 0 : comy / abs_integral);

	/*
	 * Compute the new center of mass and update the peak
	 */
	comx += grid_region.min[0];
	comy += grid_region.min[1];
	double com[2];
	com[0] = comx;
	com[1] = comy;
	SPoint c = sp->map(SPoint(2, com), INDEX, PPM);

	if (peak) {
		if (sp->mIntegrate.allow_motion && ! peak->IsLocked()) {
			peak->SetLocation(c);
		}
		peak->box_volume(method, integral);

		rr.message("Volume by %s method = %.2e\n",
			index_name(Integration_Method_Names, method).cstring(),
			integral);
	}

	delete [] wts;
	delete [] dat;

	return true;
}

/*
 * Integration helper -- integrates each peak on the peak <list>.
 */

  /*
   * For each peak on the list, look for peaks it overlaps. If there is
   * an overlap, use the overlapped method to determine the integrals.
   * Otherwise, use the isolated method.
   */
static void _integratePeaks(Spectrum *sp,
			    const List &peaks,		// list of peaks
			    const SRegion &region,	// integration region
			    const Integration_Thresholds &threshold,
			    Wait *wait, Reporter &rr)
{
  const Integration_Parameters &ip = sp->mIntegrate;
  bool use_region_data = (ip.rectangle_data && !region.no_interior());
  SRegion bounds = (use_region_data ? region : sp->ppm_region());

  List peak_list = peaks_in_region(peaks, bounds);
  int peak_count = peak_list.size();
  int peaks_done = 0;
  while (peak_list.size() > 0)
    {
      Peak		*pp = (Peak *) peak_list[0];

      Integration_Method method = ip.integration_method;
      if (method == INTEGRATE_LORENTZIAN || method == INTEGRATE_GAUSSIAN)
	{
	  Marked_Region data_mask(sp, bounds);
	  List overlapped;
	  if (peak_grouping(pp, peak_list, ip.contour_grouping,
			    ip.distance_grouping, threshold,
			    &data_mask, &overlapped, wait, rr))
	    {
	      if (overlapped.size() > 1 && wait)
		wait->progress_report("Integrated %d peaks of %d\n"
				      "Integrating group of %d peaks\n",
				      peaks_done, peak_count,
				      overlapped.size());

	      if (use_region_data && !ip.contoured_data)
		data_mask.mark_region(bounds);

	      if (ip.subtract_peaks)
		{
		  add_fit_peaks(sp->peaklets(), -1.0,
				data_mask.region(), data_mask.data());
		  add_fit_peaks(overlapped, 1.0,
				data_mask.region(), data_mask.data());
		}

	      integrate_by_fitting(overlapped, data_mask, method, wait, rr);
	    }
	  peak_list.erase(overlapped);
	  peaks_done += overlapped.size();
	}
      else if (method == INTEGRATE_BOX || method == INTEGRATE_ELLIPSE)
	{
	  integrate_by_boxing(pp, region, method, rr);
	  peak_list.erase((void *) pp);
	  peaks_done += 1;
	}

      if (wait)
	{
	  wait->progress_report("Integrated %d peaks of %d\n"
				"Finished integrating %s\n",
				peaks_done, peak_count,
				pp->CrossPeak::assignment_name().cstring());
	  if (wait->was_stop_requested())
	    break;
	}
    }

}

/*
 * Integrate all peaks on the <list>, using Rectangle <r> in case we
 * are integrating by boxing or ellipsing.
 */
void integrate_list(Spectrum *sp, const List &peaks, const SRegion &region,
		    double positive_threshold, double negative_threshold,
		    Wait *wait, Reporter &rr)
{
  if (wait)
    wait->begin_waiting("Integrating peaks.", true);

  Integration_Thresholds threshold;
  threshold.positive = positive_threshold;
  threshold.negative = negative_threshold;

  _integratePeaks(sp, peaks, region, threshold, wait, rr);

  if (wait)
    wait->end_waiting();
}

/*
 * Mark all the data region connected to the peaks.
 */
static bool find_threshold_region(const List &peaks,
				  const Integration_Thresholds &threshold,
				  Marked_Region *mr, Wait *wait,
				  Reporter &rr)
{
	int below = 0;
	int volume_limit = 300000;

	for (int pi = 0 ; pi < peaks.size() ; ++pi) {
		double lothresh, hithresh;
		Peak *pk = (Peak *) peaks[pi];
		if (pk->DataHeight() <= threshold.negative) {
			lothresh = -FLT_MAX;
			hithresh = threshold.negative;
		}
		else if (pk->DataHeight() >= threshold.positive) {
			lothresh = threshold.positive;
			hithresh = FLT_MAX;
		}
		else {
			below += 1;
			continue;	// Peak below contour level.
		}

		Spectrum *sp = pk->spectrum();
		IPoint i = sp->map(pk->position(), PPM, INDEX).rounded();
		if (!mr->mark_connected(i,
					lothresh, hithresh,
					volume_limit, wait))
		  {
		    rr.message("Bounding contour too large.\n");
		    return false;
		  }
	}

	if (below == peaks.size())
	  {
	    rr.message("All peaks below lowest contour.\n");
	    return false;
	  }

	return true;
}


/*
 * Add a peak to a list of overlapped peaks, but only if it isn't already on
 * the list.  Return 0 if the peak is already
 * on the list, otherwise return 1 if the peak was added to the list.
 */
static int _add_overlapped(List &peaks, Peak *newpk)
{
	if (peaks.contains(newpk))
	  return 0;

	peaks.append(newpk);

	return 1;
}

//
// For each peak in peaks, see if the peak falls within
// a visited region of the contour. If it does, add this peak as
// being overlapped by way of sharing a common contour.
//
static void _add_contour_overlapped(const List &peaks, List &overlapped,
				    Marked_Region *mr)
{
  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    {
      Peak *pk = (Peak *) peaks[pi];
      Spectrum *sp = pk->spectrum();
      IPoint i = sp->map(pk->position(), PPM, INDEX).rounded();
      if (mr->is_marked(i))
	_add_overlapped(overlapped, pk);
    }
}


/*
 * Examine selected peaks in the spectrum, adding peaks within the distance
 * specified in the integrate parameters.
 */
static bool _add_distance_overlapped(const List &peaks, List &overlapped)
{
	List inrange;
	for (int pi1 = 0 ; pi1 < peaks.size() ; ++pi1)
	  {
	    Peak *pk1 = (Peak *) peaks[pi1];
	    Spectrum *sp = pk1->spectrum();
	    for (int pi2 = 0 ; pi2 < overlapped.size() ; ++pi2)
	      {
		Peak *pk2 = (Peak *) overlapped[pi2];
		bool close = true;
		for (int a = 0 ; a < sp->dimension() ; ++a)
		  if (fabs(pk1->position()[a] - pk2->position()[a])
		      > sp->mIntegrate.grouping_dist[a])
		    close = false;
		if (close)
		  inrange.append(pk1);
	      }
	  }

	int	done = 0;
	for (int pi = 0 ; pi < inrange.size() ; ++pi)
	  done += _add_overlapped(overlapped, (Peak *) inrange[pi]);

	return done > 0;
}


/*
 * Find peaks in the list that are overlapped with a specified peak.
 * The contour and contour_allocated fields of the overlapped peaks
 * are set.
 *
 * The determinator of overlap may be either "contained within the same
 * base contour" method and/or "search through distance space".
 */
/*
 * This routine locates peaks overlapped with this
 * one. If there are overlaps the method also finds
 * the baseline contour.
 */
static bool peak_grouping(Peak *peak, const List &peaks,
			  bool contour_grouping, bool distance_grouping,
			  const Integration_Thresholds &threshold,
			  Marked_Region *mr, List *overlapped, Wait *wait,
			  Reporter &rr)
{
	overlapped->append(peak);
	if (!peak_above_threshold(peak, threshold))
	  return false;

	/*
	 * Loop until the contour surrounds all overlapped peaks. If
	 * we do not search for peaks with the distance method, this
	 * loop will only be executed once. Otherwise, if we add a
	 * peak to the current list of peaks with the distance method,
	 * we must recompute a bounding contour.
	 */
	bool mark_succeeded;
	List peak_list = peaks_above_threshold(peaks, threshold);
	do
	  {
	    mark_succeeded = find_threshold_region(*overlapped, threshold,
						   mr, wait, rr);
	    if (!mark_succeeded)
	      break;

	    /*
	     * This adds the peaks overlapped with this peaks,
	     * and sharing a common contour.
	     */
	    if (contour_grouping)
	      _add_contour_overlapped(peak_list, *overlapped, mr);
	  }
	while (distance_grouping
	       && _add_distance_overlapped(peak_list, *overlapped));

	if (mark_succeeded)
	  mr->enscribe_marked();  // Shrink region to enclose marked points

	return mark_succeeded;
}

// ----------------------------------------------------------------------------
//
static List peaks_above_threshold(const List &peaks, const Integration_Thresholds &t)
{
  List above;
  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    if (peak_above_threshold((Peak *) peaks[pi], t))
      above.append(peaks[pi]);
  return above;
}

// ----------------------------------------------------------------------------
//
static bool peak_above_threshold(Peak *pk, const Integration_Thresholds &t)
{
  double h = pk->DataHeight();
  return (h > t.positive || h < t.negative);
}

// ----------------------------------------------------------------------------
//
static List peaks_in_region(const List &peaks, const SRegion &region)
{
  List in_region;
  for (int pi = 0 ; pi < peaks.size() ; ++pi)
    if (region.contains(((Peak *) peaks[pi])->position()))
      in_region.append(peaks[pi]);
  return in_region;
}
