/*
 * Linefit.C:
 *
 * Peak integration is done by fitting known lineshapes to the
 * integration volume. The lineshapes can be either Gaussian or
 * Lorentzian. 
 *
 * To do the linefitting we use temporary peaks that are the
 * theoretical lineshape and that have modifiable parameters.
 * These are called LinefitPeaks.
 *
 * We also need a volume of data to fit over, and a mask on that
 * data. The data comes from the spectrum. The mask limits the linefit
 * to a portion of that data.
 *
 * In this file are the implementations for the classes:
 *
 *	LinefitPeak	- peak used during integration.
 *	LinefitVolume	- a volume that gets filled with the intensity
 *			  computed from the LinefitPeaks it is holding.
 */
#include <math.h>		// Use sqrt(), exp(), fabs()
#include <float.h>		// Use DBL_MAX
#include <stdlib.h>		// use size_t
#include <string.h>		// Use memset()

#include "dataregion.h"		// Use Marked_Region
#include "linefit.h"
#include "lusolve.h"
#include "mathconst.h"		// use MC_PI, MC_LN2
#include "memalloc.h"		// use new()
#include "num.h"		// Use abs()
#include "peak.h"
#include "reporter.h"		// use Reporter
#include "simplex.h"
#include "spectrum.h"
#include "spoint.h"		// Use SPoint
#include "subarray.h"		// use Subarray_Iterator
#include "utility.h"		// Use fatal_error()
#include "wait.h"		// use Wait

static void add_rank_one_multiple(float **OneDFactor, float scale,
				  const IRegion &r, float *d);
static void add_baseline_heights(double offset, const SPoint &slope,
				 const IRegion &region, float *data);
static double width_to_linewidth_factor(Integration_Method m);
static double height_range(const Marked_Region &data);
static double square_difference(float *d, const Marked_Region &mr);

static void compute_fit_heights(const List &peaks, const Marked_Region &r,
				float *heights);
static void compute_gaussian_factors(float *factorp, double x, int n, double ax);
static void compute_lorentzian_factors(float *factorp, double x, int n, double ax);
static bool stop_requested(void *lfvolume);
static bool peak_near_region(Peak *pk, const SRegion &region);
static void add_linefit(Peak *pk, double scale,
			const IRegion &region, float *data);
static void add_fit_heights(const SPoint &center, const SPoint &width,
			    double height, Integration_Method method,
			    const IRegion &region, float *data);

//
// The LinefitPeak is a modifiable peak used during linefitting.
//
class LinefitPeak {

public:
  LinefitPeak(Peak *, const Integration_Parameters &ip);

  int		dimension() const;
  Spectrum *	spectrum() const;

  bool		VaryPosition() const;
  const SPoint &position() const;
  void		set_position(int axis, double freq);

  bool		VaryWidth(int axis) const;
  const SPoint &width() const;
  void		set_width(int axis, double w);

  double	height() const;
  void		height(double h);

  bool		update_peak(double residual, int npoints, Reporter &);
  int		GetNumberOfVariables() const;
  const Integration_Parameters &integration_parameters() const;

  /*
   * Mark this peak as having failed in the integration.
   */
  void		MarkFailed() const;

  //
  // Return and set the state of the "heightIsComputed". The height
  // is computed if it comes from a previous linefit.
  //
  int		HeightIsComputed() const { return mHeightIsComputed; }
  void		HeightIsComputed(int s) { mHeightIsComputed = s; }

private:
  double		mHeight;
  SPoint		mPosition;	// position in index units
  SPoint		mWidth;		// width parameter in index units
  // width parameter 'a', Gausian exp(-x*x/a*a), Lorentzian 1/(1+x*x/a*a)

  Peak		*mPeak;
  bool		mVaryPosition;
  IPoint	mVaryWidth;		// boolean vary linewidth for each axis
  int		mHeightIsComputed;
  const Integration_Parameters &mIntegrate;

  double	fit_volume() const;
};

// ----------------------------------------------------------------------------
//
class LinefitVolume
{
public:

  LinefitVolume(const List &peaks,
		const Marked_Region &data_mask,
		const Integration_Parameters &ip, Wait *wait);
  virtual ~LinefitVolume();

  const List &	peak_list() const { return mPeakList; }

  double	GetSquaredResidual(double *peakparams);

  bool		stop_requested();

  bool		minimize();
  int		UpdatePeaks(Reporter &);

  /*
   * Mark each peak as having failed in the integration;
   */
  void		MarkFailed() const;

private:

  Integration_Parameters	mIntegrate;	// Integration parameters
  const Marked_Region &		mDataVolume;	// spectrum data

  List		mPeakList;		// list of LinefitPeaks
  double	mBaselineOffset;
  SPoint	mBaselineSlope;

  int		mNumOfParams;		// number of adjustable parameters
  double	*mBestParams;		// deterimined by minimize()
  float		*mFitData;		// peak fit data for data volume
  double	mResidual;		// residual for best parameters

  Wait *	mWait;
  int		mSteps;			// Count steps for progress report

  int	dimension() const;
  void	parameter_vector(double *) const;
  void	parameter_variation(double *) const;
  void	parameter_bounds(double *pmin, double *pmax) const;
  void	use_these_parameters(double *);
  void	EstimateHeights();
};

// ----------------------------------------------------------------------------
//
bool linefit(const List &peaks, const Marked_Region &data_mask,
	     const Integration_Parameters &ip, Wait *wait, Reporter &rr)
{
	/*
	 * Create a linefit block, initting it from the peaks.
	 */
	LinefitVolume linefit(peaks, data_mask, ip, wait);
	if (linefit.minimize())
	  {
	    if (! linefit.UpdatePeaks(rr))
	      return false;			// Linewidths out of bounds
	  }
	else if (wait->was_stop_requested())
	  {
	    rr.message("Linefit cancelled.\n");
	    return false;
	  }
	else					// Too many iterations
	  {
	    linefit.MarkFailed();
	    rr.message("Linefit failed to converge\n");
	    return false;
	  }

	return true;
}

// ----------------------------------------------------------------------------
// Convert between width parameter in fit function to half height full width.
// Gausian exp(-x*x/a*a), Lorentzian 1/(1+x*x/a*a)
//
static double width_to_linewidth_factor(Integration_Method m)
{
	if (m == INTEGRATE_GAUSSIAN)
		return 2 * sqrt(MC_LN2);
	else if (m == INTEGRATE_LORENTZIAN)
		return 2;

	fatal_error("width_to_linewidth(): Bad integration method\n");
	return 0;
}

int LinefitPeak::dimension() const
  { return spectrum()->dimension(); }
Spectrum *LinefitPeak::spectrum() const
  { return mPeak->spectrum(); }
bool LinefitPeak::VaryPosition() const
  { return mVaryPosition; }
const SPoint &LinefitPeak::position() const
  { return mPosition; }
void LinefitPeak::set_position(int axis, double freq)
  { mPosition[axis] = freq; }
bool LinefitPeak::VaryWidth(int axis) const
  { return mVaryWidth[axis]; }
const SPoint &LinefitPeak::width() const
  { return mWidth; }
void LinefitPeak::set_width(int axis, double w)
  { mWidth[axis] = w; }
double LinefitPeak::height() const
  { return mHeight; }
void LinefitPeak::height(double h)
  { mHeight = h; }

// ----------------------------------------------------------------------------
//
double LinefitPeak::fit_volume() const
{
  double vol = mHeight;
  for (int i = 0; i < dimension(); i++) {
    if (mIntegrate.integration_method == INTEGRATE_GAUSSIAN)
      vol *= mWidth[i] * sqrt(MC_PI);
    else
      vol *= mWidth[i] * MC_PI;
  }
  return vol;
}

/*
 * Update the peak that corresponds to this LinefitPeak.
 */
bool LinefitPeak::update_peak(double residual, int npoints, Reporter &rr)
{
	Spectrum *sp = mPeak->spectrum();

	double	vol = fit_volume();
	double fe = (vol != 0 ?
		     npoints * sqrt(residual / npoints) / fabs(vol) :
		     0);

	SPoint	pos = sp->map(position(), INDEX, PPM);
	double f = width_to_linewidth_factor(mIntegrate.integration_method);
	SPoint	lw = sp->scale(width() * f, INDEX, PPM);

	/*
	 * If the peak isn't locked, change its location.
	 */
	if (mVaryPosition && ! mPeak->IsLocked()) {
	  mPeak->IncrementLocation(pos - mPeak->position());
	}

	/*
	 * Now update its integration information.
	 */
	mPeak->FitHeight(mHeight);
	mPeak->fit_volume(mIntegrate.integration_method, vol, fe);
	mPeak->linewidth("fit", lw);

	Stringy pos_string = point_format(pos, "%6.3f", false);
	SPoint lw_hz = sp->scale(lw, PPM, HZ);
	Stringy lw_string = point_format(lw_hz, "%6.3f", false);
	rr.message("peak @ %s lw %s vol %.3e rms %.1f%%\n",
		   pos_string.cstring(), lw_string.cstring(), vol, 100 * fe);

	return true;
}

/*
 * Create a LinefitPeak from a Peak. The linefit peak is used while
 * doing a linefit. It is initalized from the <pp> and, later, it
 * finalizes the <pp> with the new fit parameters.
 */
LinefitPeak::LinefitPeak(Peak *pp, const Integration_Parameters &ip)
  : mIntegrate(ip)
{
	Spectrum	*sp = pp->spectrum();
	double f = 1 / width_to_linewidth_factor(ip.integration_method);

	mPeak = pp;
	mVaryPosition = (ip.allow_motion && !pp->IsLocked());
	mPosition = sp->map(pp->position(), PPM, INDEX);

	int dim = sp->dimension();
	mVaryWidth = IPoint(dim);
	for (int a = 0 ; a < dim ; ++a)
	  mVaryWidth[a] = (ip.adjust_linewidths ||
			   !pp->HasLinewidth() ||
			   pp->linewidth(a) <= 0);

	if (!pp->HasLinewidth())
	  pp->estimate_linewidth();
	mWidth = sp->scale(pp->linewidth(), PPM, INDEX) * f;

	mHeightIsComputed = pp->IntegrateByFit();
	mHeight = pp->FitHeight();
	if (mHeight == 0.0)
		mHeight = pp->DataHeight();
}

//
// Return the number of variables in this linefit peak. This number is
// dependent on the number free variables per peak.
//
int LinefitPeak::GetNumberOfVariables() const
{
  int dim = dimension();
  int c = 1 + (VaryPosition() ? dim : 0);	// height + position
  for (int a = 0 ; a < dim ; ++a)		// + linewidth
    c += (VaryWidth(a) ? 1 : 0);
  return c;
}

// ----------------------------------------------------------------------------
//
const Integration_Parameters &LinefitPeak::integration_parameters() const
  { return mIntegrate; }

/*
 * Compute the intensity for the peaks, given their current parameters.
 *
 * The computed intensity fills the volume <mDat>
 */
static void compute_fit_heights(const List &peaks, const Marked_Region &r,
				float *heights)
{
  memset(heights, 0, r.region().volume() * sizeof(float));

  for (int k = 0 ; k < peaks.size() ; ++k)
    {
      LinefitPeak *lp = (LinefitPeak *) peaks[k];
      add_fit_heights(lp->position(), lp->width(), lp->height(),
		      lp->integration_parameters().integration_method,
		      r.region(), heights);
    }
}

// ----------------------------------------------------------------------------
//
LinefitVolume::LinefitVolume(const List &peaks,
			     const Marked_Region &data,
			     const Integration_Parameters &ip,
			     Wait *wait)
  : mIntegrate(ip), mDataVolume(data)
{
  //
  // Make a LinefitPeak for each Peak.
  //
  for (int k = 0 ; k < peaks.size() ; ++k)
    mPeakList.append(new LinefitPeak((Peak *) peaks[k], ip));

  mBaselineOffset = 0;
  mBaselineSlope = SPoint(dimension());

  mNumOfParams = (ip.fit_baseline ? 1 + dimension() : 0);
  for (int k = 0 ; k < mPeakList.size() ; ++k)
    mNumOfParams += ((LinefitPeak *) mPeakList[k])->GetNumberOfVariables();

  mBestParams = new double [ mNumOfParams];
  mFitData = new float [data.region().volume()];
  mResidual = 0;

  mWait = wait;
  mSteps = 0;

  //
  // Now estimate the peak heights. For peaks without much overlap
  // this will be about the same as the data height.  For peaks
  // with overlap, the height will be reduced from the data height.
  //
  if (mPeakList.size() > 1 && mPeakList.size() < 100)
    EstimateHeights();
}

// ----------------------------------------------------------------------------
//
LinefitVolume::~LinefitVolume()
{
	delete [] mFitData;
	delete [] mBestParams;

	const List &peaks = peak_list();
	for (int k = 0 ; k < peaks.size() ; ++k)
		delete (LinefitPeak *) peaks[k];
}
// ----------------------------------------------------------------------------
//
int LinefitVolume::dimension() const
  { return mDataVolume.spectrum()->dimension(); }

// ----------------------------------------------------------------------------
//
void LinefitVolume::parameter_vector(double *pp) const
{
  int c = 0;

  if (mIntegrate.fit_baseline)
    {
      pp[c] = mBaselineOffset; c += 1;
      for (int k = 0 ; k < dimension() ; ++k)
	{ pp[c] = mBaselineSlope[k]; c += 1; }
    }

  const List &peaks = peak_list();
  for (int k = 0 ; k < peaks.size() ; ++k)
    {
      LinefitPeak *lp = (LinefitPeak *) peaks[k];

      pp[c] = lp->height(); c += 1;

      if (lp->VaryPosition())
	for (int i = 0; i < lp->dimension(); i++)
	  { pp[c] = lp->position()[i]; c += 1; }

      for (int a = 0; a < lp->dimension(); ++a)
	if (lp->VaryWidth(a))
	  { pp[c] = lp->width()[a]; c += 1; }
    }
}

//
// Unpack the fit parameters from the vector <params>
//
void LinefitVolume::use_these_parameters(double *params)
{
  int c = 0;
  int dim = dimension();

  if (mIntegrate.fit_baseline)
    {
      mBaselineOffset = params[c]; c += 1;
      for (int a = 0 ; a < dim ; ++a, ++c)
	mBaselineSlope[a] = params[c];
    }

  const List &peaks = peak_list();
  for (int k = 0 ; k < peaks.size() ; ++k)
    {
      LinefitPeak *lp = (LinefitPeak *) peaks[k];

      lp->height(params[c]); c += 1;

      if (lp->VaryPosition())
	for (int a = 0 ; a < dim ; ++a, ++c)
	  lp->set_position(a, params[c]);

      for (int a = 0 ; a < dim ; ++a)
	if (lp->VaryWidth(a))
	  { lp->set_width(a, params[c]); c += 1; }
    }
}

// ----------------------------------------------------------------------------
//
void LinefitVolume::parameter_variation(double *pp) const
{
  int c = 0;

  if (mIntegrate.fit_baseline)
    {
      double h = height_range(mDataVolume);
      const IRegion &r = mDataVolume.region();
      pp[c] = .1 * h; c += 1;
      for (int k = 0 ; k < dimension() ; ++k)
	{ pp[c] = .1 * h / r.size(k); c += 1; }
    }

  const List &peaks = peak_list();
  for (int k = 0 ; k < peaks.size() ; ++k)
    {
      LinefitPeak *lp = (LinefitPeak *) peaks[k];
      Spectrum *sp = lp->spectrum();

      pp[c] = lp->height() * .1; c += 1;

      if (lp->VaryPosition())
	for (int i = 0; i < lp->dimension(); i++)
	  { pp[c] = sp->scale(2.0, i, HZ, INDEX); c += 1; }

      for (int a = 0; a < lp->dimension(); ++a)
	if (lp->VaryWidth(a))
	  { pp[c] = sp->scale(2.0, a, HZ, INDEX); c += 1; }
    }
}

// ----------------------------------------------------------------------------
//
void LinefitVolume::parameter_bounds(double *pmin, double *pmax) const
{
  int c = 0;
  const Integration_Parameters &ip = mIntegrate;

  if (ip.fit_baseline)
    {
      // baseline offset
      pmin[c] = -DBL_MAX; pmax[c] = DBL_MAX; c += 1;

      // baseline slope
      for (int k = 0 ; k < dimension() ; ++k)
	{ pmin[c] = -DBL_MAX; pmax[c] = DBL_MAX; c += 1; }
    }

  const List &peaks = peak_list();
  for (int k = 0 ; k < peaks.size() ; ++k)
    {
      LinefitPeak *lp = (LinefitPeak *) peaks[k];
      Spectrum *sp = lp->spectrum();

      // height bounds
      pmin[c] = -DBL_MAX; pmax[c] = DBL_MAX; c += 1;

      // position bounds
      if (lp->VaryPosition())
	{
	  SPoint pos = lp->position();
	  SPoint range = sp->scale(ip.motion_range, PPM, INDEX);
	  for (int i = 0; i < lp->dimension(); i++)
	    {
	      pmin[c] = pos[i] - range[i];
	      pmax[c] = pos[i] + range[i];
	      c += 1;
	    }
	}

      // linewidth bounds
      SRegion range = sp->scale(ip.linewidth_range, PPM, INDEX);
      double f = 1 / width_to_linewidth_factor(ip.integration_method);
      for (int a = 0; a < lp->dimension(); ++a)
	if (lp->VaryWidth(a))
	  {
	    pmin[c] = range.min[a] * f;
	    pmax[c] = range.max[a] * f;
	    c += 1;
	  }
    }
}

/*
 * Compute the square difference between theoretical and actual data
 * over the contoured data.
 */
double LinefitVolume::GetSquaredResidual(double *peakparams)
{
	use_these_parameters(peakparams);
	compute_fit_heights(peak_list(), mDataVolume, mFitData);
	add_baseline_heights(mBaselineOffset, mBaselineSlope,
			     mDataVolume.region(), mFitData);
	return square_difference(mFitData, mDataVolume);
}

// ----------------------------------------------------------------------------
//
static double height_range(const Marked_Region &data)
{
  double hmin = 0, hmax = 0;
  const IRegion &r = data.region();
  IPoint p = r.first_point();
  do
    if (data.is_marked(p))
      {
	double h = data.height(p);
	if (h > hmax) hmax = h;
	else if (h < hmin) hmin = h;
      }
  while (r.next_point(&p));

  return hmax - hmin;
}

// ----------------------------------------------------------------------------
//
bool LinefitVolume::stop_requested()
{
  if (mWait)
    {
      mWait->progress_report("%d simplex steps", mSteps);
      mSteps += 1;
      return mWait->was_stop_requested();
    }
  return false;
}

// ----------------------------------------------------------------------------
//
static double square_difference(float *d, const Marked_Region &dr)
{
  double sum = 0;
  const float *d2 = dr.data();
  Subarray_Iterator p(dr.region(), dr.region(), true);
  for ( ; !p.finished() ; p.next())
    if (dr.is_marked(p.position()))
      {
	offset_type k = p.index();
	double diff = d[k] - d2[k];
	sum += diff * diff;
      }
  return sum;
}

//
// When the linefit succeeds, the peaks are update with the linefit
// information for the best linefit (lowest residual). This returns
// true if all peaks are successfully updated.
//
int LinefitVolume::UpdatePeaks(Reporter &rr)
{
	if (mPeakList.size() > 1)
	  rr.message("Fit group of %d peaks.\n", mPeakList.size());
	else
	 rr.message("Isolated ");

	use_these_parameters(mBestParams);
	double	residual = mResidual;
	int n = mDataVolume.marked_volume();
	int status = true;
	const List &peaks = peak_list();
	for (int k = 0 ; k < peaks.size() ; ++k)
	  status &= ((LinefitPeak *) peaks[k])->update_peak(residual, n, rr);

	if (mIntegrate.fit_baseline)
	  rr.message("baseline offset %.3e slope %s\n",
		     mBaselineOffset,
		     point_format(mBaselineSlope, "%.3e", false));

	return status;
}

/*
 * Using the current linewidths and measured heights, estimate the computed
 * heights of the peaks.
 *
 * Each peak's measured height is the sum of its actual height and the
 * contribution to its height resulting from other peaks near it. We
 * use LU decomposition to turn the measured heights into actual
 * heights by estimating the contribution of the neighboring peaks.
 */
void LinefitVolume::EstimateHeights()
{
  const List &peaks = peak_list();
  int size = peaks.size();

  /*
   * We have to pass an array of pointers to lusolve(), so
   * cook up the array here.
   */
  double **factor = new double * [size];
  for (int k = 0 ; k < size ; ++k)
    factor[k] = new double [size];
  double *measured = new double [size];

  /*
   * Fill the array of factors.  Foreach element i,j in the array,
   * the value is the contribution of peak j to peak i's height.
   * We assume the peak is gaussian (exponential).
   *
   * f[i][j] = prod(dim) exp (- (fq_i(dim) - fq_j(dim))^2 / (2 * lwj^2))
   */
  for (int k = 0 ; k < size ; ++k)
    {
      LinefitPeak *pp = (LinefitPeak *) peaks[k];
      double *fp = factor[k];

      /*
       * For peaks that have a computed height, all diagonal factors
       * are 1.0, other factors are 0.0.
       */
      if (pp->HeightIsComputed()) {
	for (int j = 0 ; j < size ; ++j)
	  *fp++ = (LinefitPeak *)peaks[j] == pp ? 1.0 : 0.0;
      }
      else {
	for (int j = 0 ; j < size ; ++j)
	  {
	    LinefitPeak *pp2 = (LinefitPeak *) peaks[j];
	    double fact = 1.0;

	    /*
	     * The exponential factor is: 
	     *		exp - (delta_f / mWidth)^2
	     */
	    int dim = mDataVolume.spectrum()->dimension();
	    SPoint diff = pp->position() - pp2->position();
	    SPoint lw = pp2->width();
	    for (int d = 0; d < dim; d++) {
	      double tmp = 0.0;
	      if (lw[d] != 0.0) {
		tmp = diff[d] / lw[d];
		tmp = exp(- tmp * tmp);
	      }
	      fact *= tmp;
	    }
	    *fp++ = fact;
	  }
      }
    }


  /*
   * Copy the measured heights into the measured array.
   */
  for (int k = 0 ; k < size ; ++k)
    measured[k] = ((LinefitPeak *) peaks[k])->height();

  /*
   * Solve the equations and, if there is a solution, copy the
   * solutions back to the height.
   */
  if (lusolve(factor, measured, size)) {
    const List &peaks = peak_list();
    for (int k = 0 ; k < size ; ++k)
      {
	LinefitPeak *pp = (LinefitPeak *) peaks[k];
	pp->height(measured[k]);
	pp->HeightIsComputed(true);
      }
  }

  delete [] measured;
  for (int k = 0 ; k < size ; ++k)
    delete [] factor[k];
  delete [] factor;
}

/*
 * Mark each peak as having failed in the integration.
 */
void LinefitVolume::MarkFailed() const
{
	const List &peaks = peak_list();
	for (int k = 0 ; k < peaks.size() ; ++k)
		((LinefitPeak *) peaks[k])->MarkFailed();
}

/*
 * Mark this peak as having failed in the integration.
 */
void LinefitPeak::MarkFailed() const
{
  mPeak->no_volume();
}


/*
 * Evaluate the X^2 difference between the theoretical peak intensity
 * and the actual peak intensity, summed over the peak box.
 */
static double _getResidual(double *pp, void *lfvolume)
{
	LinefitVolume *lp = (LinefitVolume *) lfvolume;
	return lp->GetSquaredResidual(pp);
}

// ----------------------------------------------------------------------------
//
static bool stop_requested(void *lfvolume)
{
  LinefitVolume *lp = (LinefitVolume *) lfvolume;
  return lp->stop_requested();
}

// ----------------------------------------------------------------------------
//
bool LinefitVolume::minimize()
{
  double *start = mBestParams;
  parameter_vector(start);

  double *variation = new double [ mNumOfParams ];
  parameter_variation(variation);

  double *param_max = new double [ mNumOfParams ];
  double *param_min = new double [ mNumOfParams ];
  parameter_bounds(param_min, param_max);

  bool converged = simplex(start, variation, param_min, param_max,
			   mNumOfParams, mIntegrate.tolerance,
			   mIntegrate.maxiterations, _getResidual,
			   ::stop_requested, this);
  if (converged)
    mResidual = GetSquaredResidual(mBestParams);

  delete [] variation;
  delete [] param_max;
  delete [] param_min;

  return converged;
}

/*
 * Fill in the <factorp> array for a gaussian starting at <x> away from
 * the center, incrementing <x> <n> times, with 1/e-width <ax>
 */
static void compute_gaussian_factors(float *factorp, double x, int n, double ax)
{
	double	_explimit = log(DBL_MAX);
	double	ax2 = ax * ax;

	for ( ; n > 0; n--, x += 1.0) {
		double	tmp = (ax2 == 0 ? _explimit : x * x / ax2);
		*factorp++ = (tmp >= _explimit ? 0.0 : exp(- tmp));
	}
}

/*
 * Fill in the <factorp> array for a lorentzian starting at <x> away from
 * the center, incrementing <x> <n> times, with 1/2-width <ax>
 */
static void compute_lorentzian_factors(float *factorp, double x, int n, double ax)
{
	double	ax2 = ax * ax;

	for ( ; n > 0; n--, x += 1.0)
	  {
	    double x2 = x * x;
	    double denom = (ax2 + x2);
	    *factorp++ = (denom == 0 ? 1.0 : ax2 / denom);
	  }
}

// ----------------------------------------------------------------------------
//
void add_fit_peaks(const List &peaks, double scale,
		   const IRegion &region, float *data)
{
  if (peaks.size() == 0)
    return;

  Spectrum *sp = ((Peak *) peaks[0])->spectrum();
  SRegion ppm_region = sp->map(region, INDEX, PPM);
  for (int p = 0 ; p < peaks.size() ; ++p)
    {
      Peak *pk = (Peak *) peaks[p];

      if (pk->IntegrateByFit() && peak_near_region(pk, ppm_region))
	add_linefit(pk, scale, region, data);
    }
}

// ----------------------------------------------------------------------------
//
static bool peak_near_region(Peak *pk, const SRegion &region)
{
  double lwidths = pk->linewidth_cutoff_range();
  for (int a = 0 ; a < pk->dimension() ; ++a)
    {
      double center = pk->position()[a];
      double range = lwidths * pk->linewidth(a);
      if (center + range < region.min[a] || center - range > region.max[a])
	return false;
    }

  return true;
}

// ----------------------------------------------------------------------------
//
static void add_linefit(Peak *pk, double scale,
			const IRegion &region, float *data)
{
  Spectrum *sp = pk->spectrum();
  Integration_Method method = pk->GetIntegrateMethod();
  SPoint center = sp->map(pk->position(), PPM, INDEX);
  double f = 1 / width_to_linewidth_factor(method);
  SPoint width = sp->scale(pk->linewidth(), PPM, INDEX) * f;
  double height = pk->FitHeight() * scale;

  add_fit_heights(center, width, height, method, region, data);
}

/*
 * Add a Gaussian or Lorentzian to a lattice of data values.
 */
static void add_fit_heights(const SPoint &center, const SPoint &width,
			    double height, Integration_Method method,
			    const IRegion &region, float *data)
{
  int dim = region.dimension();
  float **OneDFactor = new float * [dim];
  for (int a = 0 ; a < dim ; ++a)
    OneDFactor[a] = new float [region.size(a)];

  if (method == INTEGRATE_GAUSSIAN) {
    for (int a = 0 ; a < dim ; ++a)
      compute_gaussian_factors(OneDFactor[a],
			       region.min[a] - center[a],
			       region.size(a), width[a]);
  }
  else if (method == INTEGRATE_LORENTZIAN) {
    for (int a = 0 ; a < dim ; ++a)
      compute_lorentzian_factors(OneDFactor[a],
				 region.min[a] - center[a],
				 region.size(a), width[a]);
  }
  else
    fatal_error("add_fit_heights(): Bad integration method %d\n", method);

  add_rank_one_multiple(OneDFactor, height, region, data);

  for (int a = 0 ; a < dim ; ++a)
    delete [] OneDFactor[a];
  delete [] OneDFactor;
}

// ----------------------------------------------------------------------------
// Add a multiple of a rank one matrix to a data array.
// The OneDFactor variable is dim vectors, each vector has length equal
// to the size of the region along the corresponding axis.
//
static void add_rank_one_multiple(float **OneDFactor, float scale,
				  const IRegion &r, float *d)
{
  if (r.dimension() == 2)	// Optimize 2-D case.
    {
      float *d0 = OneDFactor[0], *d1 = OneDFactor[1];
      int size0 = r.size(0), size1 = r.size(1);
      for (int k0 = 0 ; k0 < size0 ; ++k0)
	{
	  double c = scale * d0[k0];
	  for (int k1 = 0 ; k1 < size1 ; ++k1, ++d)
	    *d += c * d1[k1];
	}
    }
  else
    {
      Subarray_Iterator p(r, r, true);
      for ( ; !p.finished() ; p.next())
	{
	  float h = scale;
	  for (int a = 0 ; a < r.dimension() ; ++a)
	    h *= OneDFactor[a][p.position()[a] - r.min[a]];
	  d[p.index()] += h;
	}
    }
}

// ----------------------------------------------------------------------------
//
static void add_baseline_heights(double offset, const SPoint &slope,
				 const IRegion &region, float *data)
{
  if (region.dimension() == 2)	// Optimize 2-D case.
    {
      int size0 = region.size(0), size1 = region.size(1);
      double m0 = slope[0], m1 = slope[1];
      for (int k0 = 0 ; k0 < size0 ; ++k0)
	{
	  float h = offset + k0 * m0;
	  for (int k1 = 0 ; k1 < size1 ; ++k1, ++data, h += m1)
	    *data += h;
	}
    }
  else
    {
      Subarray_Iterator p(region, region, true);
      for ( ; !p.finished() ; p.next())
	{
	  float h = offset;
	  for (int a = 0 ; a < region.dimension() ; ++a)
	    h += (p.position()[a] - region.min[a]) * slope[a];
	  data[p.index()] += h;
	}
    }
}
