// ----------------------------------------------------------------------------
// Container class for pointers.
//

#include <stdlib.h>	// Use qsort()

#include "list.h"
#include "memalloc.h"		// use new()
#include "utility.h"	// use fatal_error()

// ----------------------------------------------------------------------------
//
List::List()
{
  used = 0;
  allocated = 0;
  pointers = NULL;
}

// ----------------------------------------------------------------------------
//
List::List(const List &a)
{
  used = 0;
  allocated = 0;
  pointers = NULL;
  *this = a;
}

// ----------------------------------------------------------------------------
//
List &List::operator=(const List &a)
{
  if (a.used > allocated)
    {
      allocated = a.used;
      delete [] pointers;
      pointers = new void * [allocated];
    }
  used = a.used;
  for (int k = 0 ; k < used ; ++k)
    pointers[k] = a.pointers[k];

  return *this;
}

// ----------------------------------------------------------------------------
//
List::~List()
{
  delete [] pointers;
  pointers = NULL;	// Safety measure.
}

// ----------------------------------------------------------------------------
//
bool List::empty() const
{
  return used == 0;
}

// ----------------------------------------------------------------------------
//
int List::size() const
{
  return used;
}

// ----------------------------------------------------------------------------
//
void *List::operator[](int k) const
{
  return pointers[k];
}

// ----------------------------------------------------------------------------
//
void *List::pop()
{
  if (empty())
    return NULL;
  int last = size() - 1;
  void *e = (*this)[last];
  erase(last);
  return e;
}

// ----------------------------------------------------------------------------
//
void List::append(void *e)
{
  insert(e, size());
}

// ----------------------------------------------------------------------------
//
void List::append(const List &elements)
{
  int s = elements.size();
  for (int e = 0 ; e < s ; ++e)
    append(elements[e]);
}

// ----------------------------------------------------------------------------
//
#define INITIAL_LIST_SIZE 16

void List::insert(void *e, int pos)
{
  if (pos < 0 || pos > used)
    fatal_error("List::insert(): Position %d (%d) out of range.\n",
		pos, size());

  if (used == allocated)
    {
      //
      // Reallocate
      //
      allocated = (allocated == 0 ? INITIAL_LIST_SIZE : 2 * allocated);
      void **p = new void * [allocated];
      for (int k = 0 ; k < used ; ++k)
	p[k] = pointers[k];
      delete [] pointers;
      pointers = p;
    }

  for (int k = used ; k > pos ; --k)
    pointers[k] = pointers[k-1];

  pointers[pos] = e;
  used += 1;
}

// ----------------------------------------------------------------------------
//
void List::swap(int p1, int p2)
{
  void *temp = pointers[p1];
  pointers[p1] = pointers[p2];
  pointers[p2] = temp;
}

// ----------------------------------------------------------------------------
//
void List::move(int f, int t)
{
  if (f < 0 || f >= used || t < 0 || t >= used)
    fatal_error("List::move(): iterator out of range.\n");

  void *p = pointers[f];
  if (f < t)
    for (int k = f ; k < t ; ++k)
      pointers[k] = pointers[k+1];
  else if (t < f)
    for (int k = f ; k > t ; --k)
      pointers[k] = pointers[k-1];
  pointers[t] = p;
}

// ----------------------------------------------------------------------------
//
bool List::erase(void *e)
{
  bool found = false;

  int s = size();
  for (int i = find(e) ; i < s ; i = find(e))
    {
      found = true;
      erase(i);
      s -= 1;
    }

  return found;
}

// ----------------------------------------------------------------------------
//
void List::erase(int i)
{
  erase(i, i+1);
}

// ----------------------------------------------------------------------------
//
void List::erase(int b, int e)
{
  int s = size();
  if (b < 0 || b > s || e < 0 || e > s)
    fatal_error("List::erase(%d,%d): Bad range.  Size = %d\n", b, e, s);

  if (b < e)
    {
      for ( ; e < s ; ++b, ++e)
	pointers[b] = pointers[e];

      used -= (e - b);

      //
      // I should shrink the allocated space when much more is
      // allocated than is used.
      //
    }
}

// ----------------------------------------------------------------------------
//
void List::erase(const List &remove)
{
  for (int ri = 0 ; ri < remove.size() ; ++ri)
    erase(remove[ri]);
}


// ----------------------------------------------------------------------------
//
void List::erase()
{
  erase(0, size());
}

// ----------------------------------------------------------------------------
//
void List::replace(void *f, void *t)
{
  for (int k = 0 ; k < size() ; ++k)
    if (pointers[k] == f)
      pointers[k] = t;
}

// ----------------------------------------------------------------------------
//
bool List::contains(const void *e) const
{
  int s = size();
  for (int i = 0 ; i < s ; ++i)
    if ((*this)[i] == e)
      return true;

  return false;
}

// ----------------------------------------------------------------------------
//
bool List::contains(const void *e, ListOrder compare) const
{
  for (int i = 0 ; i < size() ; ++i)
    if (compare((*this)[i], e) == 0)
      return true;

  return false;
}

// ----------------------------------------------------------------------------
//
bool List::find(const void *e, int *position) const
{
  int i = find(e);
  if (i < size())
    *position = i;
  return i < size();
}

// ----------------------------------------------------------------------------
//
int List::find(const void *e) const
{
  int i;

  int s = size();
  for (i = 0 ; i < s ; ++i)
    if ((*this)[i] == e)
      break;

  return i;
}

// ----------------------------------------------------------------------------
//
ListOrder::ListOrder()
{
  this->compare = NULL;
}

// ----------------------------------------------------------------------------
//
ListOrder::ListOrder(int (*compare)(const void *a, const void *b))
{
  this->compare = compare;
}

// ----------------------------------------------------------------------------
//
int ListOrder:: operator()(const void *a, const void *b) const
{
  return compare(a, b);
}

// ----------------------------------------------------------------------------
//
int compare_ints(int a, int b)
{
  if (a > b) return 1;
  else if (b > a) return -1;
  return 0;
}

// ----------------------------------------------------------------------------
//
int compare_doubles(double a, double b)
{
  if (a > b) return 1;
  else if (b > a) return -1;
  return 0;
}

// ----------------------------------------------------------------------------
//
int compare_pointers(const void *a, const void *b)
{
  if (a > b) return 1;
  else if (b > a) return -1;
  return 0;
}

// ----------------------------------------------------------------------------
//
List List::sublist(int b, int e)
{
  if (b < 0 || b > size() || e < 0 || e > size())
    fatal_error("List::sublist(%d,%d): Bad range.  Size = %d\n", b, e, size());
  List sub;
  for (int k = b ; k < e ; ++k)
    sub.append((*this)[k]);
  return sub;
}

// ----------------------------------------------------------------------------
//
void List::reverse()
{
  int s = size();
  for (int b = 0, e = s - 1 ; e > b ; ++b, --e)
    swap(b, e);
}

// ----------------------------------------------------------------------------
//
namespace
{
const ListOrder *list_compare_fn = NULL;		// state variable

extern "C" int list_compare(const void *a, const void *b)
{
  return (*list_compare_fn)(*(void **)a, *(void **)b);
}
}

// ----------------------------------------------------------------------------
//
void List::sort(const ListOrder &compare)
{
  if (list_compare_fn)
    fatal_error("List::sort(): Non-reentrant.\n");

  list_compare_fn = &compare;
  qsort(pointers, used, sizeof(void *), list_compare);
  list_compare_fn = NULL;
}

// ----------------------------------------------------------------------------
//
void List::sort_removing_duplicates(const ListOrder &compare)
{
  if (!empty())
    {
      sort(compare);
      int p = 0;
      for (int n = 1 ; n < size() ; ++n)
	if (compare(pointers[p], pointers[n]) != 0)
	  pointers[++p] = pointers[n];
      used = p + 1;
    }
}
