// ----------------------------------------------------------------------------
// NMR data file reading and writing.
//

#ifndef NMRDATA_HEADER_INCLUDED
#define NMRDATA_HEADER_INCLUDED

#include <iostream>		// use ostream

#include "list.h"		// Use List
#include "spoint.h"		// Use SPoint, IPoint, IRegion
#include "stringc.h"		// Use Stringy

class Block_File;

// ----------------------------------------------------------------------------
//
class NMR_Data
{
 public:
  NMR_Data(const Stringy &path,
	   const IPoint &size,
	   const IPoint &block_size,
	   const SPoint &spectrometer_freq,		// In MHz
	   const SPoint &spectrum_width,		// In Hz
	   const SPoint &origin_ppm,			// Index origin in ppm
	   const List &axis_labels);
  virtual ~NMR_Data();

  Stringy path() const;
  int dimension() const;
  IPoint size() const;
  IRegion data_region() const;
  IPoint block_size() const;
  SPoint spectrometer_frequency() const;	// MHz
  SPoint spectrum_width() const;		// Hz
  //
  // There is no provision to record the sweep width.  The spectrum width
  // will be smaller than the sweep width if the data is a subregion of the
  // full acquired spectrum.
  //
  SPoint origin_ppm() const;
  Stringy axis_label(int axis) const;
  const List &axis_label_list() const;

  virtual bool read_value(const IPoint &index, float *value) = 0;
  virtual bool read_values(const IRegion &index_region, float *values) = 0;

  void print_header(std::ostream &out);

 private:
  Stringy data_path;
  IPoint data_size, bsize;
  SPoint sfreq, swidth, oppm;
  List axis_labels;
};

Stringy standard_nucleus_name(const Stringy &nucleus);

//
// Ownership of the block file is passed to this routine.
//
NMR_Data *block_file_nmr_data(Block_File *bf,
			      const SPoint &spectrometer_freq,
			      const SPoint &spectrum_width,
			      const SPoint &origin_ppm,
			      const List &axis_labels);

//
// Data is all zeros.
//
NMR_Data *dataless_nmr_data(const Stringy &path,
			    const IPoint &size,
			    const IPoint &block_size,
			    const SPoint &spectrometer_freq,
			    const SPoint &spectrum_width,
			    const SPoint &origin_ppm,
			    const List &axis_labels);

//
// The routines below produce derived NMR_Data instances.
//
// Ownership of the passed in NMR_Data instance is passed to the new
// NMR_Data instance, ie when the returned NMR_Data instance is deleted
// it deletes the passed in NMR_Data instance.
//
NMR_Data *permute_nmr_axes(NMR_Data *nmr_data, const IPoint &new_order);
NMR_Data *nmr_data_region(NMR_Data *nmr_data, const IRegion &region);
NMR_Data *thresholded_nmr_data(NMR_Data *nmr_data,
			       float neg_threshold, float pos_threshold);
NMR_Data *squished_nmr_data(NMR_Data *nmr_data, const IPoint &cell_size);
NMR_Data *projected_nmr_data(NMR_Data *nmr_data, int axis);
NMR_Data *extract_nmr_data_plane(NMR_Data *nmr_data, int axis, int plane);

#endif
