// ----------------------------------------------------------------------------
// Rectangles in the plane.
//

#include <iostream>		// use ostream
#include <math.h>		// Use fabs()

#include "list.h"		// use List
#include "rectangle.h"
#include "num.h"		// Use min(), segment_distanc()

// ----------------------------------------------------------------------------
//
static double overlap_area(const Rectangle &rect, const List &rectangles);
static void shifted_overlap(const Rectangle &rect, double dx, double dy,
			    const List &rectangles,
			    double *best_dx, double *best_dy,
			    double *least_overlap);

// ----------------------------------------------------------------------------
//
Rectangle::Rectangle(double x1, double y1, double x2, double y2)
{
  if (x1 < x2)	{ xmin = x1; xmax = x2; }
  else		{ xmin = x2; xmax = x1; }

  if (y1 < y2)	{ ymin = y1; ymax = y2; }
  else		{ ymin = y2; ymax = y1; }
}

// ----------------------------------------------------------------------------
//
Rectangle::Rectangle(const Rectangle &r)
{
  this->xmin = r.min(X);
  this->xmax = r.max(X);
  this->ymin = r.min(Y);
  this->ymax = r.max(Y);
}

// ----------------------------------------------------------------------------
//
Rectangle::Rectangle()
{
  xmin = ymin = 0;
  xmax = ymax = 0;
}

// ----------------------------------------------------------------------------
//
double Rectangle::min(Axis a) const { return (a == X ? xmin : ymin); }
double Rectangle::max(Axis a) const { return (a == X ? xmax : ymax); }
double Rectangle::size(Axis a) const
  { return (a == X ? xmax - xmin : ymax - ymin); }
double Rectangle::center(Axis a) const
  { return (a == X ? (xmax + xmin) / 2 : (ymax + ymin) / 2); }
double Rectangle::aspect() const { return (double) size(X) / size(Y); }
double Rectangle::area() const
  { return (no_interior() ? 0 : size(X) * size(Y)); }

// ----------------------------------------------------------------------------
// Return the distance from the point to the perimeter of the rectangle.
//
double Rectangle::edge_distance(double x, double y) const
{
  double d;

  d = segment_distance(min(X), max(X), x, y - min(Y));
  d = ::min(d, segment_distance(min(X), max(X), x, y - max(Y)));
  d = ::min(d, segment_distance(min(Y), max(Y), y, x - min(X)));
  d = ::min(d, segment_distance(min(Y), max(Y), y, x - max(X)));

  return d;
}

// ----------------------------------------------------------------------------
//
bool Rectangle::operator==(const Rectangle &r) const
{
  return (min(X) == r.min(X) && max(X) == r.max(X) &&
	  min(Y) == r.min(Y) && max(Y) == r.max(Y));
}

// ----------------------------------------------------------------------------
//
bool Rectangle::no_interior() const
{
  return xmin >= xmax || ymin >= ymax;
}

// ----------------------------------------------------------------------------
//
bool Rectangle::contains(double x, double y) const
{
  return (x >= min(X) && x <= max(X) && y >= min(Y) && y <= max(Y));
}

// ----------------------------------------------------------------------------
//
bool Rectangle::contains(const Rectangle &r) const
{
  return (r.min(X) >= min(X) && r.max(X) <= max(X) &&
	  r.min(Y) >= min(Y) && r.max(Y) <= max(Y));
}

// ----------------------------------------------------------------------------
//
bool Rectangle::intersects(const Rectangle &r) const
{
  bool disjoint = (max(X) < r.min(X) || r.max(X) < min(X) ||
		   max(Y) < r.min(Y) || r.max(Y) < min(Y));

  return !disjoint;
}

// ----------------------------------------------------------------------------
//
bool Rectangle::intersects_interior(const Rectangle &r) const
{
  bool disjoint = (max(X) <= r.min(X) || r.max(X) <= min(X) ||
		   max(Y) <= r.min(Y) || r.max(Y) <= min(Y));

  return !disjoint;
}

// ----------------------------------------------------------------------------
//
#define TRANSLATION_TOLERANCE 1e-6
bool Rectangle::is_translation(const Rectangle &r) const
{
  return (fabs(size(X) - r.size(X)) < size(X) * TRANSLATION_TOLERANCE &&
	  fabs(size(Y) - r.size(Y)) < size(Y) * TRANSLATION_TOLERANCE);
}

// ----------------------------------------------------------------------------
//
void Rectangle::closest_point(double x, double y, double *cx, double *cy) const
{
  if (x < min(X))	*cx = min(X);
  else if (x > max(X))	*cx = max(X);
  else			*cx = x;

  if (y < min(Y))	*cy = min(Y);
  else if (y > max(Y))	*cy = max(Y);
  else			*cy = y;
}

// ----------------------------------------------------------------------------
//
void Rectangle::operator=(const Rectangle &r)
{
  xmin = r.min(X);
  xmax = r.max(X);
  ymin = r.min(Y);
  ymax = r.max(Y);
}

// ----------------------------------------------------------------------------
//
void Rectangle::scale(double factor)
{
  scale(center(X), center(Y), factor);
}

// ----------------------------------------------------------------------------
//
void Rectangle::scale(double cx, double cy, double factor)
{
  xmin = cx + factor * (xmin - cx);
  xmax = cx + factor * (xmax - cx);
  ymin = cy + factor * (ymin - cy);
  ymax = cy + factor * (ymax - cy);
}

// ----------------------------------------------------------------------------
//
void Rectangle::scale(Axis a, double factor)
{
  double c = center(a);
  double s = size(a);
  double pmin = c - fabs(factor) * s / 2; 
  double pmax = c + fabs(factor) * s / 2; 

  if (a == X)
    { xmin = pmin; xmax = pmax; }
  else if (a == Y)
    { ymin = pmin; ymax = pmax; }
}

// ----------------------------------------------------------------------------
//
void Rectangle::clip(const Rectangle &r)
{
  if (r.max(X) <= min(X) || r.min(X) >= max(X) ||
      r.max(Y) <= min(Y) || r.min(Y) >= max(Y))
    xmin = xmax = ymin = ymax = 0;

  if (r.min(X) > min(X)) xmin = r.min(X);
  if (r.max(X) < max(X)) xmax = r.max(X);
  if (r.min(Y) > min(Y)) ymin = r.min(Y);
  if (r.max(Y) < max(Y)) ymax = r.max(Y);
}

// ----------------------------------------------------------------------------
//
void Rectangle::encompass(double x, double y)
{
  if (x < xmin) xmin = x;
  if (x > xmax) xmax = x;
  if (y < ymin) ymin = y;
  if (y > ymax) ymax = y;
}

// ----------------------------------------------------------------------------
//
void Rectangle::encompass(const Rectangle &r)
{
  if (r.min(X) < min(X)) xmin = r.min(X);
  if (r.max(X) > max(X)) xmax = r.max(X);
  if (r.min(Y) < min(Y)) ymin = r.min(Y);
  if (r.max(Y) > max(Y)) ymax = r.max(Y);
}

// ----------------------------------------------------------------------------
//
void Rectangle::translate(double dx, double dy)
{
  xmin += dx;
  xmax += dx;
  ymin += dy;
  ymax += dy;
}

// ----------------------------------------------------------------------------
//
void Rectangle::translate(double delta, Axis a)
{
  if (a == X)
    { xmin += delta; xmax += delta; }
  else if (a == Y)
    { ymin += delta; ymax += delta; }
}

// ----------------------------------------------------------------------------
//
void Rectangle::translate_to(double x, double y)
{
  xmax = x + size(X);
  ymax = y + size(Y);
  xmin = x;
  ymin = y;
}

// ----------------------------------------------------------------------------
//
void Rectangle::recenter(double cx, double cy)
{
  translate(cx - center(X), cy - center(Y));
}

// ----------------------------------------------------------------------------
//
void Rectangle::transpose()
{
  swap(&xmin, &ymin);
  swap(&xmax, &ymax);
}

// ----------------------------------------------------------------------------
//
void Rectangle::match_aspect(double aspect)
{
  if (aspect > this->aspect())
    scale(X, aspect / this->aspect());
  else
    scale(Y, this->aspect() / aspect);
}

// ----------------------------------------------------------------------------
//
void Rectangle::match_aspect(const Rectangle &r)
{
  if (r.aspect() > aspect())
    scale(X, r.aspect() / aspect());
  else
    scale(Y, aspect() / r.aspect());
}

// ----------------------------------------------------------------------------
//
Axis other_axis(Axis a)
{ 
  if (a == X)
    return Y;
  else if (a == Y)
    return X;

  return NOT_XY;
}

// ----------------------------------------------------------------------------
//
std::ostream &operator<<(std::ostream &s, const Rectangle &r)
{
  return s << "[" << r.min(X) << "," << r.max(X) << "]"
    << " by " << "[" << r.min(Y) << "," << r.max(Y) << "]";
}

// ----------------------------------------------------------------------------
//
void free_rectangle_list_entries(const List &rectangles)
{
  for (int ri = 0 ; ri < rectangles.size() ; ++ri)
    delete (Rectangle *) rectangles[ri];
}

// ----------------------------------------------------------------------------
// Move the movable rectangles so that they overlap with other fixed and
// movable rectangles minimally.  Do this by displacing them in steps along
// the x and y axes for several iterations reducing the overlap area.
// The rectangles in the movable list are shifted.
//
void unoverlap_rectangles(const List &movable, const List &fixed,
			  double step_fraction, int max_iterations)
{
  List all = movable;
  all.append(fixed);

  int overlapped = movable.size();
  for (int i = 0 ; i < max_iterations && overlapped > 0 ; ++i)
    {
      overlapped = 0;
      for (int mi = 0 ; mi < movable.size() ; ++mi)
	{
	  Rectangle *m = (Rectangle *) movable[mi];
	  double overlap = overlap_area(*m, all);
	  if (overlap > m->area())
	    {
	      overlapped += 1;
	      double best_dx = 0;
	      double best_dy = 0;
	      double reduced_self_overlap = min(1.0, step_fraction)*m->area();
	      double least_overlap = overlap - reduced_self_overlap;
	      double dx = step_fraction * m->size(X);
	      double dy = step_fraction * m->size(Y);
	      shifted_overlap(*m, dx, 0, all,
			      &best_dx, &best_dy, &least_overlap);
	      shifted_overlap(*m, -dx, 0, all,
			      &best_dx, &best_dy, &least_overlap);
	      shifted_overlap(*m, 0, dy, all,
			      &best_dx, &best_dy, &least_overlap);
	      shifted_overlap(*m, 0, -dy, all,
			      &best_dx, &best_dy, &least_overlap);
	      m->translate(best_dx, best_dy);
	    }
	}
    }
}

// ----------------------------------------------------------------------------
//
static double overlap_area(const Rectangle &rect, const List &rectangles)
{
  double area = 0;
  for (int ri = 0 ; ri < rectangles.size() ; ++ri)
    {
      Rectangle *r = (Rectangle *) rectangles[ri];
      Rectangle i = rect;
      i.clip(*r);
      area += i.area();
    }
  return area;
}

// ----------------------------------------------------------------------------
//
static void shifted_overlap(const Rectangle &rect, double dx, double dy,
			    const List &rectangles,
			    double *best_dx, double *best_dy,
			    double *least_overlap)
{
  Rectangle r = rect;
  r.translate(dx, dy);
  double overlap = overlap_area(r, rectangles);
  if (overlap < *least_overlap)
    {
      *best_dx = dx;
      *best_dy = dy;
      *least_overlap = overlap;
    }
}
