// ----------------------------------------------------------------------------
// A resonance is the frequency associated with a specific atom of a molecule.
// The frequency depends on the type of atom (1H, 13C, 15N), and it's local
// environment.  Important factors effecting the local environment include
// what types covalent bonds the given atom has, type of solvent, temperature,
// pH, other nearby atoms, ....
//
// So a resonance corresponds to a specific atom of the molecule under study
// and specific experimental condition (temperature, pH).  The frequency can
// be observed as the position of an assigned peak in spectrum.  Because one
// typically has many assigned peaks in several spectra for one resonance
// Sparky averages the positions of all such peaks.  The average weights
// every assigned peak axis equally.
//

#ifndef _Resonance_h
#define _Resonance_h

#include "list.h"		// use List

class Atom;
class Condition;
class CrossPeak;
class Group;
class Stringy;

class Resonance
{
public:

  Resonance(Condition *, Atom *);
  virtual		~Resonance();

  Stringy		name() const;
  double		frequency();
  void			set_frequency(double freq_ppm);  // only if unassigned
  double		frequencySigma() const;

  Condition *		condition() const;
  Atom *		atom() const;
  Group *		group() const;

  void			add_assignment(CrossPeak *x);
  void			remove_assignment(CrossPeak *x);
  void			peak_moved(const CrossPeak *xp);
  const List &		crosspeaks() const;
  int			assignment_count() const;

private:
  Atom		*mAtom;			// to Atom
  Condition	*mCondition;		// in this Condition

  List		mCrossPeaks;

  double	freq_ppm;		// cached computed value
  bool		freq_valid;		// 1 if freq is valid

  void		InvalidateFrequency(void);
  void		mValidateCache();
};

Resonance *closest_resonance(const List &res, double freq_ppm);
int compare_resonance_names(const void *res1, const void *res2);
int compare_resonance_frequencies(const void *a, const void *b);
void sort_aliased_resonances(List &rlist, double alias_min, double alias_max);
List closest_n_resonances(const List &rlist,
			  double alias_min, double alias_max,
			  double pos_ppm, int n);

#endif
