/*
 * Spectrum.C - Implemenation of the Spectrum class 
 *
 * A Spectrum object encompasses all aspects of the Spectrum, including
 * information about the NMR data that is displayed, the list of peaks
 * that are found in the spectrum, relationships between peak assignments
 * due to the type of spectrum acquired, and saving and restoring methods.
 *
 * Most of the Spectrum's NMR data information is stored in the separate
 * SpectrumData class, one of the superclasses of this class.
 *
 */

#include "assigncopy.h"
#include "atom.h"		// Use Atom
#include "condition.h"		// Use Condition
#include "constant.h"
#include "grid.h"
#include "group.h"
#include "integrate.h"
#include "label.h"
#include "line.h"
#include "list.h"		// Use List
#include "molecule.h"
#include "nmrdata.h"		// use NMR_Data
#include "num.h"
#include "peak.h"
#include "peakgp.h"
#include "resonance.h"
#include "session.h"		// use Session
#include "spectrum.h"
#include "spoint.h"		// Use SPoint
#include "system.h"		// use file_path()
#include "uipeak.h"		// Use show_peak_list_dialog()
#include "notifier.h"		// Use dialog_update_spectrum_deleted()

static bool peak_has_assignment(CrossPeak *xp, const List &resonances);
static void MoveAssignments(Spectrum *sp, Condition *tocp);
static void shift_ornaments(const List &olist, const SPoint &delta);

/* #define DEBUG_RES_DATA */

/*
 * When guessing the assignment for a peak we combine the differences in
 * frequency between the peak position and the actual resonances, using
 * all guessing axes (recall there is something called multi-axis guessing).
 *
 * Since 1H frequencies are generally more accurate than 13C or 15N frequencies
 * we weight the frequency differences to reflect this accuracy. To that end,
 * the following rules are applied:
 *
 *	1) when the guess involves only 1H atoms, all frequency differences
 *	   are weighted with a weight of 1.0
 *	2) when the guess involves no 1H atoms, all frequency differences
 *	   are weighted with a weight of 1.0
 *	3) any other guess uses a weight of 1.0 for the 1H atoms and a
 *	   weight of HETERO_GUESS_WEIGHT for the 13C and 15N atoms.
 */
#define HETERO_GUESS_WEIGHT	0.1

static char *_assignFormat[DIM + 1] = {
	NULL,
	"%a1-%a2",
	"%a1-%a2-%a3",
	"%a1-%a2-%a3-%a4",
};

/*
 * Construct a Spectrum in Condition <cond>, with Spectrum ID <id>, of
 * dimensionality <dim>, which is use to initialize the SpectrumData.
 */
Spectrum::Spectrum(Session &s, const Stringy &name, Condition *cond,
		   const Stringy &save_path, NMR_Data *nmr_data)
	: SpectrumData(nmr_data),
	  mIntegrate(this),
	  mPeakPick(nmr_data->dimension()),
	  ses(s),
	  mAssignGuessOptions(nmr_data->dimension()),
	  mSaveFile(s, SAVEFILE_SPECTRUM, true)
{
	mName = name;
	mCondition = cond;
	mSaveFile.set_path(save_path);

	/*
	 * Ornament initialization.
	 */
	mSelectsize = scale(.5, 0, INDEX, PPM);
	mPointersize = scale(.5, 0, INDEX, PPM);
	mLineendsize = scale(.5, 0, INDEX, PPM);

	mSizes[peak] = scale(1.2, 0, INDEX, PPM);
	mSizes[peakgroup] = scale(.8, 0, INDEX, PPM);
	mSizes[label] = scale(2.5, 0, INDEX, PPM);
	mSizes[line] = scale(.4, 0, INDEX, PPM);
	mSizes[grid] = scale(.4, 0, INDEX, PPM);

	/*
	 * Set the assignment format
	 */
	mAssignFormat = GetDefaultAssignmentFormat();

	mAssignCopyThreshold = SPoint(dimension());
	for (int a = 0 ; a < dimension() ; ++a)
	  mAssignCopyThreshold[a] = (nucleus_type(a) == "1H" ? .02 : .1);

	notifier().send_notice(nt_created_spectrum, this);
}

/*
 * Destruct a spectrum
 */
Spectrum::~Spectrum()
{
  notifier().send_notice(nt_will_delete_spectrum, this);

  unselect_ornaments();

  save_file().remove_backup();

  List olist = ornaments();

  //
  // erase ornament and peak list so they are not erased one by one
  // as each ornament is deleted taking O(n^2) time.
  //
  mOrnaments.erase();
  mPeaks.erase();

  for (int oi = 0 ; oi < olist.size() ; ++oi)
    delete (Ornament *) olist[oi];
}

Condition *Spectrum::condition() const
  { return mCondition; }
Molecule *Spectrum::molecule() const
  { return condition()->molecule(); }

//
// Add a crosspeak to the spectrum's set of crosspeaks
//
void Spectrum::add(CrossPeak &x)
{
	mPeaks.append(&x);
}


//
// Remove a composite crosspeak from the spectrum's set of crosspeaks
//
void Spectrum::remove(PeakGp &x)
{
	x.removeAllPeaks();	// bring all the components back into the set

	mPeaks.erase(&x);
}


//
// Remove a simple crosspeak from the spectrum's set of crosspeaks
//
void Spectrum::remove(Peak &x)
{
	mPeaks.erase(&x);
}

// ----------------------------------------------------------------------------
//
CrossPeak *Spectrum::find_assigned_peak(const Stringy &assignment)
{
  const List &plist = crosspeaks();
  for (int xi = 0 ; xi < plist.size() ; ++xi)
    {
      CrossPeak *xp = (CrossPeak *) plist[xi];
      if (xp->assignment_name() == assignment)
	return xp;
    }
  return NULL;
}

// ----------------------------------------------------------------------------
//
CrossPeak *Spectrum::find_assigned_peak(const List &resonances)
{
  int a = 0;
  while (a < resonances.size() && resonances[a] == NULL)
    a += 1;
    
  const List &plist = (a < resonances.size() ?
		       ((Resonance *) resonances[a])->crosspeaks() :
		       crosspeaks());
  for (int xi = 0 ; xi < plist.size() ; ++xi)
    {
      CrossPeak *xp = (CrossPeak *) plist[xi];
      if (xp->spectrum() == this && peak_has_assignment(xp, resonances))
	return xp;
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
static bool peak_has_assignment(CrossPeak *xp, const List &resonances)
{
  int dim = xp->dimension();
  for (int a = 0 ; a < dim ; ++a)
    if (xp->resonance(a) != (Resonance *) resonances[a])
      return false;
  return true;
}

// ----------------------------------------------------------------------------
//
Stringy Spectrum::name() const
  { return mName; }
void Spectrum::set_name(const Stringy &name)
{
  mName = name;
  notifier().send_notice(nt_renamed_spectrum, this);
}

// ----------------------------------------------------------------------------
//
Session &Spectrum::session()
  { return ses; }
Notifier &Spectrum::notifier()
  { return ses.notifier(); }

//
// Return the fully qualified name based on the molecule and condition names
//
Stringy Spectrum::fullname() const
{
  Stringy condname = condition()->fullname();
  if (condname.is_empty())
    return name();
  return condname + " / " + name();
}

//
// Return the array of ornaments of the specified type.
//
const List &Spectrum::ornaments() const
{
	return mOrnaments;
}

// ----------------------------------------------------------------------------
//
List Spectrum::ornaments(Ornament_Type t) const
{
  List list;

  const List &olist = ornaments();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (op->type() == t)
	list.append(op);
    }

  return list;
}

// ----------------------------------------------------------------------------
//
Ornament *Spectrum::nth_ornament(Ornament_Type t, int n) const
{
  int count = 0;
  const List &olist = ornaments();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (op->type() == t && count++ == n)
	return op;
    }

  return NULL;
}

// ----------------------------------------------------------------------------
//
void Spectrum::unselect_ornaments()
{
  List copy = selected_ornaments();
  for (int oi = 0 ; oi < copy.size() ; ++oi)
    ((Ornament *) copy[oi])->select(false);
}

/*
 * Add Ornament <op> to this Spectrum.
 */
void Spectrum::AddOrnament(Ornament *op)
{
  mOrnaments.append(op);
  save_file().NeedsSaving(true);
}

/*
 * Remove Ornament <op> from this Spectrum.
 */
void Spectrum::RemoveOrnament(Ornament *op)
{
  save_file().NeedsSaving(true);
  mOrnaments.erase(op);
}

// ----------------------------------------------------------------------------
//
Stringy Spectrum::GetDefaultAssignmentFormat()
  { return _assignFormat[dimension() - 1]; }

// ----------------------------------------------------------------------------
//
Stringy Spectrum::assignment_format() const
  { return mAssignFormat; }

/*
 * Set the assignment label format
 */
void Spectrum::set_assignment_format(const Stringy &format)
{
	/*
	 * If the assignment format has changed all peaks will need
	 * to have their label recalculated.
	 */
	if (mAssignFormat != format) {
	  mAssignFormat = format;

	  /*
	   * Update each peak that is showing a label that
	   * is an assignment label.
	   */
	  const List &plist = crosspeaks();
	  for (int xi = 0 ; xi < plist.size() ; ++xi) {
	    CrossPeak *x = (CrossPeak *) plist[xi];
	    Label *lp = x->label();

	    if (lp && lp->is_assignment())
	      lp->AssignmentChanged();
	  }
	}
}

// ----------------------------------------------------------------------------
//
const Guess_Assignments &Spectrum::GetAssignGuessOptions() const
  { return mAssignGuessOptions; }

/*
 * Set this Spectrum's Assignment Tool Options to <o>.
 */
void Spectrum::SetAssignGuessOptions(const Guess_Assignments &ga)
{
	mAssignGuessOptions = ga;
}

// ----------------------------------------------------------------------------
//
SPoint Spectrum::assignment_copy_threshold() const
  { return mAssignCopyThreshold; }
void Spectrum::set_assignment_copy_threshold(const SPoint &thresh)
  { mAssignCopyThreshold = thresh; }

// ----------------------------------------------------------------------------
//
void Spectrum::set_ornament_size(Ornament_Type type, double size)
{
  Notifier &n = notifier();
  List olist = ornaments(type);
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    n.send_notice(nt_will_change_ornament, olist[oi]);

  mSizes[type] = size;

  for (int oi = 0 ; oi < olist.size() ; ++oi)
    n.send_notice(nt_changed_ornament, olist[oi]);

  if (type == label)
    n.send_notice(nt_resized_spectrum_labels, this);
}

// ----------------------------------------------------------------------------
// This list does not contain peaklets ie. peaks that are part of peak groups.
//
const List &Spectrum::crosspeaks() const
{
  return mPeaks;
}

// ----------------------------------------------------------------------------
//
List Spectrum::selected_crosspeaks() const
{
  List slist;

  List plist = crosspeaks();
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      if (xp->IsSelected())
	slist.append(xp);
    }

  return slist;
}

// ----------------------------------------------------------------------------
//
List Spectrum::nearby_crosspeaks(const SPoint &p) const
{
  List nearby;

  SPoint limits = assignment_copy_threshold();
  List plist = crosspeaks();
  for (int pi = 0 ; pi < plist.size() ; ++pi)
    {
      CrossPeak *xp = (CrossPeak *) plist[pi];
      if (points_are_close(xp->position(), p, limits))
	nearby.append(xp);
    }

  return nearby;
}

// ----------------------------------------------------------------------------
//
List Spectrum::peaklets() const
{
  List rlist;

  const List &olist = ornaments();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (op->type() == peak)
	rlist.append((Peak *) op);
    }

  return rlist;
}

// ----------------------------------------------------------------------------
//
List Spectrum::selected_peaklets() const
{
  List slist;

  const List &olist = ornaments();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (op->type() == peak && op->IsSelected())
	slist.append((Peak *)op);
    }

  return slist;
}

// ----------------------------------------------------------------------------
//
List Spectrum::selected_ornaments() const
{
  List selected;

  const List &olist = ornaments();
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (op->IsSelected())
	selected.append(op);
    }

  return selected;
}

// ----------------------------------------------------------------------------
//
SaveFile &Spectrum::save_file()
  { return mSaveFile; }

// ----------------------------------------------------------------------------
//
void Spectrum::set_save_file(const SaveFile &sf)
  { mSaveFile = sf; }

/*
 * Move <sp> from its previous Condition to this Condition.
 */
void Spectrum::set_condition(Condition *c)
{
	if (c != condition()) {

		mCondition = c;
		save_file().NeedsSaving(true);
		MoveAssignments(this, c);
		notifier().send_notice(nt_changed_spectrum_condition, this);
	}
}

/*
 * Move all assignments to the new condition.
 */
static void MoveAssignments(Spectrum *sp, Condition *tocp)
{
	const List &plist = sp->crosspeaks();
	if (plist.size() == 0)
		return;

	int	dim = sp->dimension();
	for (int xi = 0 ; xi < plist.size() ; ++xi) {
		CrossPeak *x = (CrossPeak *) plist[xi];

		for (int a = 0; a < dim; ++a) {
		  Resonance *rp = x->resonance(a);
		  if (rp)
		    {
		      Atom *ap = rp->atom();
		      rp = tocp->define_resonance(ap->group()->name(),
						  ap->name(),
						  ap->nucleus());
		      x->SetResonance(a, rp);
		    }
		}
	}
}

double Spectrum::ornament_size(Ornament_Type type) const
  { return mSizes[type]; }

double Spectrum::select_size() const
  { return mSelectsize; }
double Spectrum::pointer_size() const
  { return mPointersize; }
double Spectrum::line_end_size() const
  { return mLineendsize; }

// ----------------------------------------------------------------------------
//
AttachedData &Spectrum::saved_values()
  { return attached_data; }

// ----------------------------------------------------------------------------
// Set a new origin of the ppm scale for a spectrum.
// All ornament positions are shifted so that they stay in
// the same position relative to spectrum contours.
//
void Spectrum::set_ppm_zero(const SPoint &zero_position)
{
  SPoint delta = -zero_position;
  if (delta.norm() > 1e-5)
    {
      set_ppm_shift(ppm_shift() + delta);
      shift_ornaments(ornaments(), delta);
      save_file().NeedsSaving(true);
      notifier().send_notice(nt_shifted_spectrum, this);
    }
}

// ----------------------------------------------------------------------------
//
static void shift_ornaments(const List &olist, const SPoint &delta)
{
  for (int oi = 0 ; oi < olist.size() ; ++oi)
    {
      Ornament *op = (Ornament *) olist[oi];
      if (op->type() == peak)
	{
	  Peak *pk = (Peak *) op;
	  pk->set_position(pk->position() + delta); // Preserves integration
	}
      else
	op->IncrementLocation(delta);
    }
}
