// ----------------------------------------------------------------------------
// Operating system routines common across all platforms.
// This file is included by the platform specific system-PLATFORM.cc
//

#include <ctype.h>		// use isalpha()
#include <signal.h>		// Use sigaction()
#include <stdio.h>		// use fseek(), fseeko()
#include <stdlib.h>		// use atexit()
#include <string.h>		// Use strchr(), strcat(), strncmp(), strrchr()
#include <time.h>		// Use time()

#include <sys/stat.h>		// Use stat()
#include <sys/types.h>		// Use struct stat

#include "list.h"		// use List
#include "memalloc.h"		// use new()
#include "stringc.h"		// Use Stringy
#include "system.h"

// ----------------------------------------------------------------------------
//
static Stringy driveless_file_directory(const Stringy &path);
static void split_drive(const Stringy &path,
			Stringy *drive, Stringy *driveless_path);
static bool split_path(const Stringy &path, Stringy *dir, Stringy *file);
static Stringy dedotify_driveless_path(const Stringy &path);
extern "C" void remove_files();

// ----------------------------------------------------------------------------
// These routines are defined by the platform specific system-PLATFORM.cc
// which includes this file.
//
static Stringy user_home_directory();
static Stringy user_home_directory(const Stringy &username);

// ----------------------------------------------------------------------------
//
Stringy file_name(const Stringy &path)
{
  Stringy dir, file;
  return (split_path(path, &dir, &file) ? file : path);
}

// ----------------------------------------------------------------------------
//
Stringy file_directory(const Stringy &path)
{
  Stringy drive, driveless_path;
  split_drive(path, &drive, &driveless_path);
  return drive + driveless_file_directory(driveless_path);
}

// ----------------------------------------------------------------------------
//
Stringy file_path(const Stringy &directory, const Stringy &file)
{
  return (file.is_empty() ? directory : directory + path_separator() + file);
}

// ----------------------------------------------------------------------------
//
static Stringy driveless_file_directory(const Stringy &path)
{
  Stringy dir, file;
  return (split_path(path, &dir, &file) ?
	  (dir.is_empty() ? path_separator() : dir) :
	  Stringy("."));
}

// ----------------------------------------------------------------------------
// Handle drive letters eg. c:/user/smith
//
static void split_drive(const Stringy &path,
			Stringy *drive, Stringy *driveless_path)
{
  bool has_drive = (path.length() >= 2 && path[1] == ':' && isalpha(path[0]));
  *drive = (has_drive ? path.substring(0,2) : Stringy(""));
  *driveless_path = (has_drive ? path.substring(2) : path);
}

// ----------------------------------------------------------------------------
//
bool is_absolute_path(const Stringy &path)
{
  Stringy drive, driveless_path;
  split_drive(path, &drive, &driveless_path);
  char separator = path_separator()[0];
  return (driveless_path.is_empty() || driveless_path[0] == separator);
}

// ----------------------------------------------------------------------------
//
static bool split_path(const Stringy &path, Stringy *dir, Stringy *file)
{
  const char *start = path.cstring();
  char separator = path_separator()[0];
  const char *delim = strrchr(start, separator);

  if (delim)
    {
      *dir = substring(start, delim);
      *file = Stringy(delim+1);
      return true;
    }

  return false;
}

// ----------------------------------------------------------------------------
//
Stringy parent_directory(const Stringy &directory)
{
  return file_path(directory, "..");
}

// ----------------------------------------------------------------------------
//
bool is_directory(const Stringy &path)
{
  struct stat	sbuf;

  return stat(path.cstring(), &sbuf) == 0 && S_ISDIR(sbuf.st_mode);
}

// ----------------------------------------------------------------------------
//
Stringy current_directory()
{
  char path[4096];
  if (getcwd(path, sizeof(path)) == NULL)
    return "";
  return path;
}

// ----------------------------------------------------------------------------
//
offset_type file_size(const Stringy &path)
{
  struct stat f_stat;
  return (stat(path.cstring(), &f_stat) == EOF ?
	  (offset_type) 0 :
	  f_stat.st_size);
}

/* ----------------------------------------------------------------------------
*  Seek to beginning of file.
*/
bool beginning_of_file(FILE *fp)
{
  return seek_position(fp, 0, 0);
}

/* ----------------------------------------------------------------------------
 * Not using POSIX lseek() because this does not work with stdio.h FILE*
 * because it keeps track of its seek position independent of the file
 * descriptor.  That is needed so FILE* can block read data and save it for
 * subsequent fread() calls.
 *
 * Not using fseek() because it takes an offset of type long which is often
 * only 32 bits.  That prevents accessing files > 2 Gb.  The lseek() routine
 * takes an offset of type off_t which is typically 64 bits.
 */
bool seek_position(FILE *fp, offset_type offset, int origin)
{
#ifdef SPARKY_NO_FSEEKO
  return fseek(fp, static_cast<long>(offset), origin) != -1;
#else
  return fseeko(fp, static_cast<off_t>(offset), origin) != -1;
#endif
}

/*
 * Return true if the file exists.
 */
bool file_exists(const Stringy &path)
{
  struct stat sbuf;
  return stat(path.cstring(), &sbuf) == 0;
}

/*
 * Return true if file <nm> can be written.
 */
bool file_is_writable(const Stringy &path)
{
	/*
	 * Check for writability by opening in append mode so that the
	 * contents are not destroyed (right now).
	 */
	FILE	*fp = fopen(path.cstring(), "a");	// File writable check.
	if (fp) {
		fclose(fp);
		return true;
	}
	return false;
}

/*
 * Map
 *	~[/dir] 	-> my_home[/dir]
 *	~user[/dir]	-> user_home[/dir]
 */
Stringy tilde_expand(const Stringy &path)
{
  if (path.is_empty())
    return path;

  char separator = path_separator()[0];
  if (path[0] == '~')
    if (path.length() == 1 || path[1] == separator)
      {
	Stringy p = user_home_directory();
	return (p.is_empty() ? path : p + path.substring(1));
      }
    else
      {
	const char *user_end = strchr(path.cstring(), separator);
	Stringy user = substring(path.cstring() + 1, user_end);
	Stringy p = user_home_directory(user);
	return (p.is_empty() ? path : p + user_end);
      }
  return path;
}

// ----------------------------------------------------------------------------
//
Stringy dedotify_path(const Stringy &path)
{
  Stringy drive, driveless_path;
  split_drive(path, &drive, &driveless_path);
  return drive + dedotify_driveless_path(driveless_path);
}

// ----------------------------------------------------------------------------
//
static Stringy dedotify_driveless_path(const Stringy &path)
{
  char separator = path_separator()[0];
  if (path[0] != separator)
    return path;		// Handle only absolute paths

  if (path == path_separator())
    return path;

  Stringy dir = dedotify_driveless_path(driveless_file_directory(path));
  Stringy last = file_name(path);
  if (last == ".")
    return dir;
  else if (last == "..")
    return driveless_file_directory(dir);
  return (dir == path_separator() ?
	  path_separator() + last :
	  file_path(dir, last));
}

// ----------------------------------------------------------------------------
//
time_t file_modification_time(const Stringy &path)
{
  struct stat	f_stat;
  return (stat(path.cstring(), &f_stat) == EOF ? 0 : f_stat.st_mtime);
}

// ----------------------------------------------------------------------------
//
Stringy temporary_file_name()
  { return tmpnam(NULL); }

// ----------------------------------------------------------------------------
//
static List files_to_remove;			// state variable
void remove_file_at_exit(const Stringy &path)
{
  for (int f = 0 ; f < files_to_remove.size() ; ++f)
    if (*(Stringy *) files_to_remove[f] == path)
      return;

  if (files_to_remove.empty())
    atexit(remove_files);

  files_to_remove.append(new Stringy(path));
}

// ----------------------------------------------------------------------------
//
extern "C" void remove_files()
{
  for (int f = 0 ; f < files_to_remove.size() ; ++f)
    {
      Stringy *path = (Stringy *) files_to_remove[f];
      remove_file(*path);
      delete path;
    }
  files_to_remove.erase();
}

// ----------------------------------------------------------------------------
//
bool run_command(const Stringy &command)
{
  return system(command.cstring()) == 0;
}
