// ----------------------------------------------------------------------------
//
#include <string.h>		// use strcmp()

#include "integrate.h"		// Use Integration_Method
#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "uicomponents.h"	// Use Spectrum_Menu
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uiview.h"		// Use View
#include "winsystem.h"		// use WinSys

static const char *integration_method(Integration_Method type);
static Integration_Method integration_type(const char *method);

// ----------------------------------------------------------------------------
//
class integration_dialog : public Dialog
{
public:
  integration_dialog(Session &);
  ~integration_dialog();

  static integration_dialog *the(Session &);

  void show(Spectrum *sp);
  void update(Spectrum *sp);

private:
  Widget method, switches;
  Widget ranges, maxstep, tolerance;
  Spectrum_Menu *spectrum_menu;
  Spectrum *sp;

  static void will_delete_spectrum_cb(void *idialog, void *spectrum);
  static void spectrum_menu_cb(Widget, CB_Data, CB_Data);

  void apply();
};

// ----------------------------------------------------------------------------
//
void show_integration_dialog(Session &s, Spectrum *sp)
  { integration_dialog::the(s)->show(sp); }

// ----------------------------------------------------------------------------
//
integration_dialog::integration_dialog(Session &s)
  : Dialog(s, "integrationDialog")
{
  this->sp = NULL;

  spectrum_menu = new Spectrum_Menu(s, dialog, "spectrumMenu");
  ws.option_callback(spectrum_menu->option_menu(), spectrum_menu_cb, this);

  const char *methods[] = { integration_method(INTEGRATE_GAUSSIAN),
			    integration_method(INTEGRATE_LORENTZIAN),
			    integration_method(INTEGRATE_BOX),
			    integration_method(INTEGRATE_ELLIPSE),
			    NULL };
  method = ws.option_menu(dialog, "method", methods);

  const char *swnames[] = {"allowMotion", "adjustLinewidths",
			   "baselineFit", "subtractPeaks",
			   "contouredData", "rectangleData",
			   "contourGroup", "distanceGroup", NULL};
  switches = ws.switches(dialog, "switches", swnames);

  Table_Entry label = TABLE_LABEL;
  Table_Entry value = TABLE_TEXT_FIELD;
  ranges = ws.widget_table(dialog, "ranges", 5, 5,
			label, label, label, label, label,
			label, value, value, value, value,
			label, value, value, value, value,
			label, value, value, value, value,
			label, value, value, value, value);

  maxstep = ws.edit_field(dialog, "maxStep");
  tolerance = ws.edit_field(dialog, "tolerance");

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "ok", ok_cb, this,
			       "apply", apply_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(separator, spectrum_menu->option_menu(),
			method, switches, ranges, maxstep, tolerance,
			separator, controls, END_OF_WIDGETS);

  Notifier &n = session.notifier();
  n.notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);
}

// ----------------------------------------------------------------------------
//
integration_dialog::~integration_dialog()
{
  session.dialog_table().delete_dialog("integration_dialog", this);

  Notifier &n = session.notifier();
  n.dont_notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);

  delete spectrum_menu;
}

// ----------------------------------------------------------------------------
// The default integration_dialog instance.
//
integration_dialog *integration_dialog::the(Session &s)
{
  Stringy name = "integration_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new integration_dialog(s));
  return (integration_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void integration_dialog::will_delete_spectrum_cb(void *idialog, void *spectrum)
{
  integration_dialog *id = (integration_dialog *) idialog;
  Spectrum *sp = (Spectrum *) spectrum;

  if (id->sp == sp)
    id->update(NULL);
}

// ----------------------------------------------------------------------------
//
void integration_dialog::spectrum_menu_cb(Widget, CB_Data client_data, CB_Data)
{
  integration_dialog *id = (integration_dialog *) client_data;
  Spectrum *spect = id->spectrum_menu->selected_spectrum();

  if (spect)
    id->update(spect);
}

// ----------------------------------------------------------------------------
//
static const char *integration_method(Integration_Method type)
{
  const char *mname;

  switch (type)
    {
    case INTEGRATE_GAUSSIAN:	 mname = "Gaussian fit"; break;
    case INTEGRATE_LORENTZIAN:	 mname = "Lorentzian fit"; break;
    case INTEGRATE_BOX:		 mname = "Sum over box"; break;
    case INTEGRATE_ELLIPSE:	 mname = "Sum over ellipse"; break;
    default:
      mname = "Gaussian fit";
    }

  return mname;
}

// ----------------------------------------------------------------------------
//
static Integration_Method integration_type(const char *method)
{
  if (strcmp(method, integration_method(INTEGRATE_GAUSSIAN)) == 0)
    return INTEGRATE_GAUSSIAN;
  else if (strcmp(method, integration_method(INTEGRATE_LORENTZIAN)) == 0)
    return INTEGRATE_LORENTZIAN;
  else if (strcmp(method, integration_method(INTEGRATE_BOX)) == 0)
    return INTEGRATE_BOX;
  else if (strcmp(method, integration_method(INTEGRATE_ELLIPSE)) == 0)
    return INTEGRATE_ELLIPSE;

  return INTEGRATE_NONE;
}

// ----------------------------------------------------------------------------
//
void integration_dialog::show(Spectrum *sp)
{
  update(sp);
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void integration_dialog::update(Spectrum *sp)
{
  this->sp = sp;

  Stringy dialog_title = (sp == NULL ? Stringy("Integration Settings") :
			  Stringy("Integration ") + sp->name());
  ws.set_dialog_title(dialog, dialog_title);

  spectrum_menu->set_spectrum_choice(sp);

  int dim = (sp == NULL ? 2 : sp->dimension());
  Integration_Parameters param = (sp == NULL ? Integration_Parameters(NULL) :
				  sp->mIntegrate);
  ws.set_option(method, integration_method(param.integration_method));
  ws.set_switch(switches, "allowMotion", param.allow_motion);
  ws.set_switch(switches, "adjustLinewidths", param.adjust_linewidths);
  ws.set_switch(switches, "baselineFit", param.fit_baseline);
  ws.set_switch(switches, "subtractPeaks", param.subtract_peaks);
  ws.set_switch(switches, "contouredData", param.contoured_data);
  ws.set_switch(switches, "rectangleData", param.rectangle_data);
  ws.set_switch(switches, "contourGroup", param.contour_grouping);
  ws.set_switch(switches, "distanceGroup", param.distance_grouping);

  for (int a = 0 ; a < dim ; ++a)
    {
      ws.set_label_text(ws.table_element(ranges, 0, a+1),
			table_axis_label(sp, a, ""));

      ws.set_numeric_text_field(ws.table_element(ranges, 1, a+1),
			     "%.3f", param.motion_range[a]);

      double min_lw_ppm = param.linewidth_range.min[a];
      double min_lw_hz = (sp == NULL ? 0 : sp->scale(min_lw_ppm, a, PPM, HZ));
      ws.set_numeric_text_field(ws.table_element(ranges, 2, a+1),
				"%.1f", min_lw_hz);

      double max_lw_ppm = param.linewidth_range.max[a];
      double max_lw_hz = (sp == NULL ? 1000 :
			  sp->scale(max_lw_ppm, a, PPM, HZ));
      ws.set_numeric_text_field(ws.table_element(ranges, 3, a+1),
				"%.1f", max_lw_hz);

      double grange = (sp == NULL ? 30 :
		       sp->scale(param.grouping_dist[a], a, PPM, HZ));
      ws.set_numeric_text_field(ws.table_element(ranges, 4, a+1), "%.1f", grange);
    }
  ws.manage_table_children(ranges, 5, dim + 1);

  ws.set_numeric_edit_value(maxstep, "%.0f", param.maxiterations);
  ws.set_numeric_edit_value(tolerance, "%.3g", 100 * param.tolerance);
}

// ----------------------------------------------------------------------------
//
void integration_dialog::apply()
{
  if (sp == NULL)
    return;

  Integration_Parameters &param = sp->mIntegrate;

  Stringy method_name = ws.option_selected(method);
  param.integration_method = integration_type(method_name.cstring());

  param.allow_motion = ws.switch_state(switches, "allowMotion");
  param.adjust_linewidths = ws.switch_state(switches,
						  "adjustLinewidths");
  param.fit_baseline = ws.switch_state(switches, "baselineFit");
  param.subtract_peaks = ws.switch_state(switches, "subtractPeaks");
  param.contoured_data = ws.switch_state(switches, "contouredData");
  param.rectangle_data = ws.switch_state(switches, "rectangleData");
  param.contour_grouping = ws.switch_state(switches, "contourGroup");
  param.distance_grouping = ws.switch_state(switches, "distanceGroup");

  int dim = sp->dimension();
  SPoint pos_range(dim), lw_min_hz(dim), lw_max_hz(dim), g_range(dim);
  for (int a = 0 ; a < dim ; ++a)
    {
      pos_range[a] = ws.numeric_text_field(ws.table_element(ranges, 1, a+1));
      lw_min_hz[a] = ws.numeric_text_field(ws.table_element(ranges, 2, a+1));
      lw_max_hz[a] = ws.numeric_text_field(ws.table_element(ranges, 3, a+1));
      g_range[a] = ws.numeric_text_field(ws.table_element(ranges, 4, a+1));
    }
  SRegion lw_range(sp->scale(lw_min_hz, HZ, PPM),
		   sp->scale(lw_max_hz, HZ, PPM));
  param.motion_range = pos_range;
  param.linewidth_range = lw_range;
  param.grouping_dist = sp->scale(g_range, HZ, PPM);

  param.maxiterations = (int) ws.numeric_edit_value(maxstep);
  param.tolerance = ws.numeric_edit_value(tolerance) / 100;
}
