// ----------------------------------------------------------------------------
//
#include <string.h>		// Use strcmp()

#include "crosspeak.h"
#include "label.h"		// use Label
#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "peak.h"		// use peak
#include "project.h"		// Use Project
#include "session.h"		// use Session
#include "spectrum.h"
#include "spoint.h"
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "winsystem.h"		// use WinSys

// ----------------------------------------------------------------------------
//
class label_dialog : public Dialog
{
public:
  label_dialog(Session &);
  ~label_dialog();

  static label_dialog *the(Session &);

  void show(Spectrum *sp, const SPoint &position);
  void show(Label *label);
  void show(CrossPeak *xp);
  void update(Label *label);
  void update(CrossPeak *xp);

private:
  Widget type_menu, label_text;
  Spectrum *spectrum;
  SPoint position;
  CrossPeak *xp;
  Label *lbl;		// only valid if xp is null

  static void ornament_selection_cb(void *ldialog, void *);
  static void will_delete_crosspeak_cb(void *ldialog, void *crosspeak);
  static void will_delete_label_cb(void *ldialog, void *label);
  static void will_delete_spectrum_cb(void *ldialog, void *spectrum);
  static void label_type_cb(Widget, CB_Data, CB_Data);

  Label *label() const;
  void apply();
};

static Label *single_label_selected(Project &);

// ----------------------------------------------------------------------------
//
void show_label_dialog(Session &s, Ornament *peak_or_label)
{
  label_dialog *ld = label_dialog::the(s);
  if (peak_or_label == NULL)
    ld->show((Label *) NULL);
  else if (peak_or_label->type() == label)
    ld->show((Label *) peak_or_label);
  else if (is_crosspeak(peak_or_label))
    {
      CrossPeak *xp = (is_peaklet(peak_or_label) ?
		       (CrossPeak *) ((Peak *) peak_or_label)->peakgp() :
		       (CrossPeak *) peak_or_label);
      ld->show(xp);
    }
}

// ----------------------------------------------------------------------------
//
void show_label_dialog(Session &s, Spectrum *sp, const SPoint &position)
  { label_dialog::the(s)->show(sp, position); }

// ----------------------------------------------------------------------------
//
label_dialog::label_dialog(Session &s) : Dialog(s, "labelDialog")
{
  this->spectrum = NULL;
  this->xp = NULL;
  this->lbl = NULL;

  const char *types[] = {"user", "assignment", NULL};
  type_menu = ws.option_menu(dialog, "type", types);
  ws.option_callback(type_menu, label_type_cb, this);

  label_text = ws.edit_field(dialog, "label");

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "ok", ok_cb, this,
			       "apply", apply_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(separator, type_menu, label_text,
		     separator, controls, END_OF_WIDGETS);

  Notifier &n = session.notifier();
  n.notify_me(nt_selected_ornaments_changed, ornament_selection_cb, this);
  n.notify_me(nt_will_delete_crosspeak, will_delete_crosspeak_cb, this);
  n.notify_me(nt_will_delete_label, will_delete_label_cb, this);
  n.notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);
}

// ----------------------------------------------------------------------------
//
label_dialog::~label_dialog()
{
  session.dialog_table().delete_dialog("label_dialog", this);

  Notifier &n = session.notifier();
  n.dont_notify_me(nt_selected_ornaments_changed, ornament_selection_cb, this);
  n.dont_notify_me(nt_will_delete_crosspeak, will_delete_crosspeak_cb, this);
  n.dont_notify_me(nt_will_delete_label, will_delete_label_cb, this);
  n.dont_notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);
}

// ----------------------------------------------------------------------------
// The default label_dialog instance.
//
label_dialog *label_dialog::the(Session &s)
{
  Stringy name = "label_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new label_dialog(s));
  return (label_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void label_dialog::ornament_selection_cb(void *ldialog, void *)
{
  label_dialog *ld = (label_dialog *) ldialog;

  if (ld->shown())
    {
      Project &proj = ld->session.project();
      CrossPeak *xp = proj.single_peak_selected();
      Label *label = single_label_selected(proj);
      if (xp)
	ld->update(xp);
      else if (label)
	ld->update(label);
    }
}

// ----------------------------------------------------------------------------
//
static Label *single_label_selected(Project &proj)
{
  Label *label = NULL;
  List selected = proj.selected_ornaments();
  if (selected.size() == 1)
    {
      Ornament *op = (Ornament *) selected[0];
      if (op->type() == ::label)
	label = (Label *) op;
    }
  return label;
}

// ----------------------------------------------------------------------------
//
void label_dialog::will_delete_crosspeak_cb(void *ldialog, void *xpeak)
{
  label_dialog *ld = (label_dialog *) ldialog;
  CrossPeak *xp = (CrossPeak *) xpeak;

  if (ld->xp == xp)
    ld->update((CrossPeak *) NULL);
}

// ----------------------------------------------------------------------------
//
void label_dialog::will_delete_label_cb(void *ldialog, void *label)
{
  label_dialog *ld = (label_dialog *) ldialog;
  Label *lab = (Label *) label;

  if (ld->label() == lab)
    ld->update(ld->xp);
}

// ----------------------------------------------------------------------------
//
void label_dialog::will_delete_spectrum_cb(void *ldialog, void *spectrum)
{
  label_dialog *ld = (label_dialog *) ldialog;
  Spectrum *sp = (Spectrum *) spectrum;

  if (ld->spectrum == sp)
    ld->update((Label *) NULL);
}

// ----------------------------------------------------------------------------
//
void label_dialog::label_type_cb(Widget w, CB_Data data, CB_Data)
{
  label_dialog *ld = (label_dialog *) data;

  Stringy type = ld->ws.option_selected(w);
  if (type == "user")
    {
      ld->ws.edit_field_editable(ld->label_text, true);
      Label *lab = ld->label();
      Stringy text = (lab && !lab->is_assignment() ?
		      lab->GetString() : Stringy(""));
      ld->ws.set_edit_value(ld->label_text, text);
    }
  else if (type == "assignment")
    {
      ld->ws.edit_field_editable(ld->label_text, true);
      ld->ws.set_edit_value(ld->label_text,
		     (ld->xp ? ld->xp->assignment_name() : Stringy("")));
      ld->ws.edit_field_editable(ld->label_text, false);
    }
}

// ----------------------------------------------------------------------------
//
void label_dialog::show(Spectrum *sp, const SPoint &position)
{
  this->spectrum = sp;
  this->position = position;
  this->xp = NULL;
  this->lbl = NULL;

  ws.sensitize_option(type_menu, "assignment", false);
  ws.set_option(type_menu, "user");
  ws.edit_field_editable(label_text, true);
  ws.set_edit_value(label_text, "");

  ws.show_dialog(dialog);
}

// ----------------------------------------------------------------------------
//
void label_dialog::show(Label *label)
{
  update(label);
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void label_dialog::update(Label *label)
{
  this->spectrum = (label == NULL ? NULL : label->spectrum());
  this->xp = (label == NULL ? NULL : label->attached());
  this->lbl = label;

  ws.sensitize_option(type_menu, "assignment", (xp != NULL));
  bool assign = (label == NULL ? false : label->is_assignment());
  Stringy label_type = (assign ? "assignment" : "user");
  ws.set_option(type_menu, label_type);
  ws.edit_field_editable(label_text, true);
  Stringy text = (label == NULL ? "" : label->GetString());
  ws.set_edit_value(label_text, text);
  ws.edit_field_editable(label_text, !assign);
}

// ----------------------------------------------------------------------------
//
void label_dialog::show(CrossPeak *xp)
{
  update(xp);
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void label_dialog::update(CrossPeak *xp)
{
  this->spectrum = (xp == NULL ? NULL : xp->spectrum());
  this->xp = xp;
  this->lbl = NULL;

  ws.sensitize_option(type_menu, "assignment", true);
  Label *lab = label();
  bool assign = ((lab && lab->is_assignment()) ||
		 (!lab && xp && xp->IsAssigned()));

  ws.set_option(type_menu, (assign ? "assignment" : "user"));
  ws.edit_field_editable(label_text, true);
  Stringy text = (assign ?
		  (xp ? xp->assignment_name() : Stringy("")) :
		  (lab ? lab->GetString() : Stringy("")));
  ws.set_edit_value(label_text, text);
  ws.edit_field_editable(label_text, !assign);
}

// ----------------------------------------------------------------------------
//
Label *label_dialog::label() const
  { return (xp ? xp->label() : lbl); }

// ----------------------------------------------------------------------------
//
void label_dialog::apply()
{
  Stringy text = ws.edit_value(label_text);
  bool assignment = (ws.option_selected(type_menu) == "assignment");

  Label *lab = label();
  if (lab)
    {
      lab->assignment(assignment);
      if (!assignment)
	lab->SetString(text);
    }
  else if (xp)
    if (assignment)
      (void) new Label(xp);
    else
      (void) new Label(xp, text);
  else if (spectrum)
    (void) new Label(spectrum, text, position);
}
