// ----------------------------------------------------------------------------
//
#include <stdarg.h>	        // Use ...
#include <stdio.h>		// Use vsprintf()

#include "command.h"		// Use parse_key()
#include "memalloc.h"		// use new
#include "session.h"		// use Session
#include "stringc.h"		// use Stringy
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uimain.h"
#include "uimenu.h"		// Use create_main_menubar()
#include "version.h"		// Use SPARKY_VERSION
#include "winsystem.h"		// use WinSys,
				//   process_events_until_unmapped()

// ----------------------------------------------------------------------------
//
class main_dialog : public Dialog
{
public:
  main_dialog(Session &);
  ~main_dialog();

  static main_dialog *the(Session &);

  void show();
  void show_message(const Stringy &msg);
  void show_on_status_line(const Stringy &text);

private:
  Widget menubar, workwin, text, statusline;

  static void destroy_query_cb(Widget, CB_Data, CB_Data);
  static void parse_key_cb(Widget, CB_Data, CB_Data);
};

// ----------------------------------------------------------------------------
//
void show_main_window(Session &s)
  { main_dialog::the(s)->show(); }
void show_message(Session &s, const Stringy &msg)
  { main_dialog::the(s)->show_message(msg); }
void delete_main_dialog(Session &s)
  { delete main_dialog::the(s); }

// ----------------------------------------------------------------------------
//
main_dialog::main_dialog(Session &s)
  : Dialog(s.window_system().main_widget(), s)
{
  Stringy title = formatted_string("Sparky %d.%d",
				   SPARKY_VERSION, SPARKY_RELEASE);
  ws.set_dialog_title(dialog, title);
  ws.add_event_callback(destroy_query_event, dialog, destroy_query_cb, this);

  menubar = create_main_menubar(s, dialog);
  workwin = ws.create_scrolled_text(dialog, "text", &text);
  ws.add_event_callback(key_press_event, text, parse_key_cb, this);
  statusline = ws.create_label(dialog, "statusLine");

  ws.set_main_window_areas(dialog, menubar, workwin, statusline);
}

// ----------------------------------------------------------------------------
//
main_dialog::~main_dialog()
{
  session.dialog_table().delete_dialog("main_dialog", this);

  ws.remove_event_callback(destroy_query_event, dialog, destroy_query_cb, this);
  ws.remove_event_callback(key_press_event, text, parse_key_cb, this);

  // Don't delete the main widget.
  // That is done during window system cleanup.

  ws.delete_widget(menubar);
  ws.delete_widget(workwin);
  ws.delete_widget(statusline);
}

// ----------------------------------------------------------------------------
// The default main_dialog instance.
//
main_dialog *main_dialog::the(Session &s)
{
  Stringy name = "main_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new main_dialog(s));
  return (main_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void main_dialog::destroy_query_cb(Widget, CB_Data mdialog, CB_Data)
{
  main_dialog *md = (main_dialog *) mdialog;
  Session &s = md->session;
  if (s.ok_to_end_session())
    s.end_session();
}

// ----------------------------------------------------------------------------
//
void main_dialog::parse_key_cb(Widget, CB_Data mdlg, CB_Data event)
{
  main_dialog *m = (main_dialog *) mdlg;
  m->session.command_dispatcher().parse_key_event(event);
}

// ----------------------------------------------------------------------------
//
void main_dialog::show()
{
  ws.show_dialog(dialog);
}

// ----------------------------------------------------------------------------
//
void main_dialog::show_message(const Stringy &msg)
{
  ws.append_text(text, msg);  // Append message to main window output.
}

// ----------------------------------------------------------------------------
//
void main_dialog::show_on_status_line(const Stringy &text)
{
  ws.set_label_text(statusline, text);
  ws.repaint_widget(statusline);
}

// ----------------------------------------------------------------------------
//
#define MAX_MESSAGE_SIZE 4096
void status_line(Session &s, const char *format, ...)
{
  char msg[MAX_MESSAGE_SIZE];
  va_list args;

  va_start(args, format);
  vsprintf(msg, format, args);
  va_end(args);

  main_dialog::the(s)->show_on_status_line(msg);
}

// ----------------------------------------------------------------------------
//
class Query_Dialog
{
public:
  Query_Dialog(WinSys &ws, const Stringy &name, const Stringy &question,
	       const Stringy &reply1);
  Query_Dialog(WinSys &ws, const Stringy &name, const Stringy &question,
	       const Stringy &reply1, const Stringy &reply2);
  Query_Dialog(WinSys &ws, const Stringy &name, const Stringy &question,
	       const Stringy &reply1, const Stringy &reply2,
	       const Stringy &reply3);
  ~Query_Dialog();

  int wait_for_reply();

private:
  WinSys &ws;
  Widget dialog;
  int choice;
  bool replied;

  static void choice_1_cb(Widget, CB_Data, CB_Data);
  static void choice_2_cb(Widget, CB_Data, CB_Data);
  static void choice_3_cb(Widget, CB_Data, CB_Data);
};

// ----------------------------------------------------------------------------
//
Query_Dialog::Query_Dialog(WinSys &winsys, const Stringy &name,
			   const Stringy &question, const Stringy &reply1)
  : ws(winsys)
{
  this->dialog = ws.choice_dialog(name, true, question.cstring(),
				  reply1.cstring(), choice_1_cb, this,
				  NULL);
}

// ----------------------------------------------------------------------------
//
Query_Dialog::Query_Dialog(WinSys &winsys, const Stringy &name,
			   const Stringy &question, const Stringy &reply1,
			   const Stringy &reply2)
  : ws(winsys)
{
  this->dialog = ws.choice_dialog(name, true, question.cstring(),
				  reply1.cstring(), choice_1_cb, this,
				  reply2.cstring(), choice_2_cb, this,
				  NULL);
}

// ----------------------------------------------------------------------------
//
Query_Dialog::Query_Dialog(WinSys &winsys, const Stringy &name,
			   const Stringy &question, const Stringy &reply1,
			   const Stringy &reply2, const Stringy &reply3)
  : ws(winsys)
{
  this->dialog = ws.choice_dialog(name, true, question.cstring(),
				  reply1.cstring(), choice_1_cb, this,
				  reply2.cstring(), choice_2_cb, this,
				  reply3.cstring(), choice_3_cb, this,
				  NULL);
}

// ----------------------------------------------------------------------------
//
Query_Dialog::~Query_Dialog()
{
  ws.delete_choice_dialog(dialog);
}

// ----------------------------------------------------------------------------
//
int Query_Dialog::wait_for_reply()
{
  this->choice = 0;
  this->replied = false;
  ws.show_dialog(dialog);
  ws.process_until_modal_reply(dialog, &replied);
  return choice;
}

// ----------------------------------------------------------------------------
//
void Query_Dialog::choice_1_cb(Widget, CB_Data qdialog, CB_Data)
{
  Query_Dialog *qd = (Query_Dialog *) qdialog;
  qd->choice = 1;
  qd->replied = true;
}

// ----------------------------------------------------------------------------
//
void Query_Dialog::choice_2_cb(Widget, CB_Data qdialog, CB_Data)
{
  Query_Dialog *qd = (Query_Dialog *) qdialog;
  qd->choice = 2;
  qd->replied = true;
}

// ----------------------------------------------------------------------------
//
void Query_Dialog::choice_3_cb(Widget, CB_Data qdialog, CB_Data)
{
  Query_Dialog *qd = (Query_Dialog *) qdialog;
  qd->choice = 3;
  qd->replied = true;
}

// ----------------------------------------------------------------------------
//
void show_warning(Session &s, const Stringy &text)
{
  Query_Dialog qd(s.window_system(), "warningDialog", text, "OK");
  qd.wait_for_reply();
}

// ----------------------------------------------------------------------------
//
int query(Session &s, const Stringy &question,
	  const Stringy &choice1, const Stringy &choice2)
{
  Query_Dialog qd(s.window_system(), "queryDialog", question,
		  choice1, choice2);
  return qd.wait_for_reply();
}

// ----------------------------------------------------------------------------
//
int query(Session &s, const Stringy &question, const Stringy &choice1,
	  const Stringy &choice2, const Stringy &choice3)
{
  Query_Dialog qd(s.window_system(), "queryDialog", question,
		  choice1, choice2, choice3);
  return qd.wait_for_reply();
}
