// ----------------------------------------------------------------------------
//

#include "condition.h"		// Use Condition
#include "memalloc.h"		// use new()
#include "molecule.h"		// Use Molecule
#include "notifier.h"		// use Notifier
#include "project.h"		// use Project
#include "session.h"		// use Session
#include "spectrum.h"		// Use Spectrum
#include "uicomponents.h"	// Use Molecule_Field, Condition_Field
#include "uidialogs.h"		// use help_cb()
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uiview.h"		// Use View
#include "winsystem.h"			// use WinSys, ...

// ----------------------------------------------------------------------------
//
class spectrum_dialog : public Dialog
{
public:
  spectrum_dialog(Session &);
  ~spectrum_dialog();

  static spectrum_dialog *the(Session &);

  void show(Spectrum *sp);
  void update(Spectrum *sp);

private:
  Widget rename, file, noise, noise_sample, shifts;
  Molecule_Field *molecule;
  Condition_Field *condition;
  Spectrum_Menu *spectrum_menu;
  Spectrum *spectrum;

  static void will_delete_spectrum_cb(void *idialog, void *spectrum);
  static void spectrum_menu_cb(Widget, CB_Data, CB_Data);
  static void recompute_noise_cb(Widget, CB_Data, CB_Data);

  void recompute_noise();
  void apply();
};

// ----------------------------------------------------------------------------
//
void show_spectrum_dialog(Session &s, Spectrum *sp)
  { spectrum_dialog::the(s)->show(sp); }

// ----------------------------------------------------------------------------
//
spectrum_dialog::spectrum_dialog(Session &s) : Dialog(s, "spectrumDialog")
{
  this->spectrum = NULL;

  spectrum_menu = new Spectrum_Menu(s, dialog, "spectrumMenu");
  ws.option_callback(spectrum_menu->option_menu(), spectrum_menu_cb, this);

  rename = ws.edit_field(dialog, "rename");
  file = ws.create_label(dialog, "file");

  molecule = new Molecule_Field(s, dialog, "molecule");
  condition = new Condition_Field(s, dialog, "condition");

  noise = ws.edit_field(dialog, "noise");

  Widget rf = ws.create_frame(dialog, "recompute");
  Widget rbutton = ws.push_button(rf, "noisebutton", recompute_noise_cb, this);
  noise_sample = ws.edit_field(rf, "noisecount");
  ws.row_attachments(noise_sample, rbutton, noise_sample, END_OF_WIDGETS);
  Widget rlabel = ws.create_label(dialog, "recomplabel");

  Table_Entry tf = TABLE_TEXT_FIELD;
  Table_Entry tl = TABLE_LABEL;
  shifts = ws.widget_table(dialog, "shifts", 5, 4,
			   tl, tl, tl, tl,
			   tl, tf, tf, tl,
			   tl, tf, tf, tl,
			   tl, tf, tf, tl,
			   tl, tf, tf, tl);
  ws.set_label_text(ws.table_element(shifts,0,1), "shift\n(ppm)");
  ws.set_label_text(ws.table_element(shifts,0,2), "sweepwidth\n(Hz)");
  ws.set_label_text(ws.table_element(shifts,0,3), "spectrum\nwidth (Hz)");

  Widget separator = ws.create_separator(dialog, "separator");

  Widget controls = ws.button_row(dialog, "controls",
			       "ok", ok_cb, this,
			       "apply", apply_cb, this,
			       "close", close_cb, this,
			       "help", help_cb, &s,
			       NULL);

  ws.column_attachments(separator, spectrum_menu->option_menu(), rename, file,
			molecule->edit_field(), condition->edit_field(),
			noise, rf, rlabel, shifts,
			separator, controls, END_OF_WIDGETS);

  Notifier &n = session.notifier();
  n.notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);
}

// ----------------------------------------------------------------------------
//
spectrum_dialog::~spectrum_dialog()
{
  session.dialog_table().delete_dialog("spectrum_dialog", this);

  Notifier &n = session.notifier();
  n.dont_notify_me(nt_will_delete_spectrum, will_delete_spectrum_cb, this);

  delete spectrum_menu;
  delete molecule;
  delete condition;
}

// ----------------------------------------------------------------------------
// The default spectrum_dialog instance.
//
spectrum_dialog *spectrum_dialog::the(Session &s)
{
  Stringy name = "spectrum_dialog";
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog(name) == NULL)
    dt.set_dialog(name, new spectrum_dialog(s));
  return (spectrum_dialog *) dt.get_dialog(name);
}

// ----------------------------------------------------------------------------
//
void spectrum_dialog::will_delete_spectrum_cb(void *sdialog, void *spectrum)
{
  spectrum_dialog *sd = (spectrum_dialog *) sdialog;
  Spectrum *sp = (Spectrum *) spectrum;

  if (sd->spectrum == sp)
    sd->update(NULL);
}

// ----------------------------------------------------------------------------
//
void spectrum_dialog::spectrum_menu_cb(Widget, CB_Data sdialog, CB_Data)
{
  spectrum_dialog *sd = (spectrum_dialog *) sdialog;
  Spectrum *spect = sd->spectrum_menu->selected_spectrum();

  if (spect)
    sd->update(spect);
}

// ----------------------------------------------------------------------------
//
void spectrum_dialog::recompute_noise_cb(Widget, CB_Data sdialog, CB_Data)
{
  spectrum_dialog *sd = (spectrum_dialog *) sdialog;
  sd->recompute_noise();
}

// ----------------------------------------------------------------------------
//
void spectrum_dialog::recompute_noise()
{
  if (spectrum == NULL)
    return;

  int sample_count = (int) ws.numeric_edit_value(noise_sample);
  if (sample_count > 0 && sample_count <= 10000)
    {
      double noise_level = sample_spectrum_noise(spectrum, sample_count);
      ws.set_numeric_edit_value(noise, "%.3g", noise_level);
    }
}

// ----------------------------------------------------------------------------
//
void spectrum_dialog::show(Spectrum *sp)
{
  update(sp);
  ws.show_dialog(dialog);
  ws.raise_widget(dialog);
}

// ----------------------------------------------------------------------------
//
void spectrum_dialog::update(Spectrum *sp)
{
  this->spectrum = sp;

  Stringy dialog_title = (spectrum == NULL ? Stringy("Spectrum Settings") :
			  Stringy("Spectrum ") + spectrum->name());
  ws.set_dialog_title(dialog, dialog_title);

  spectrum_menu->set_spectrum_choice(spectrum);

  ws.set_edit_value(rename, (spectrum == NULL ? "" : spectrum->name()));

  Stringy data_path = (spectrum == NULL ? "" : spectrum->data_path());
  ws.set_label_text(file, Stringy("File: ") + data_path);

  ws.set_edit_value(molecule->edit_field(),
		    (spectrum == NULL ? "" : spectrum->molecule()->name()));
  ws.set_edit_value(condition->edit_field(),
		    (spectrum == NULL ? "" : spectrum->condition()->name()));

  if (spectrum && spectrum->have_noise_level())
    ws.set_numeric_edit_value(noise, "%.3g", spectrum->noise_level());
  else
    ws.set_edit_value(noise, "");

  int dim = (spectrum == NULL ? 2 : spectrum->dimension());
  ws.manage_table_children(shifts, dim+1, 4);

  SPoint ppm_shift = (spectrum == NULL ? SPoint(dim) : spectrum->ppm_shift());
  SPoint sweep_width = (spectrum == NULL ? SPoint(dim) :
			spectrum->scale(spectrum->ppm_sweep_width(), PPM, HZ));
  SPoint spectrum_width = 
    (spectrum == NULL ? SPoint(dim) :
     spectrum->scale(spectrum->ppm_spectrum_width(), PPM, HZ));
  for (int a = 0 ; a < dim ; ++a)
    {
      Stringy label = table_axis_label(spectrum, a, "");
      ws.set_label_text(ws.table_element(shifts,a+1,0), label);
      ws.set_numeric_text_field(ws.table_element(shifts,a+1,1),
				"%.5f", ppm_shift[a]);
      ws.set_numeric_text_field(ws.table_element(shifts,a+1,2),
				"%.3f", sweep_width[a]);
      ws.set_label_text(ws.table_element(shifts,a+1,3),
			formatted_string("%.3f", spectrum_width[a]));
    }
}

// ----------------------------------------------------------------------------
//
void spectrum_dialog::apply()
{
  if (spectrum == NULL)
    return;

  Stringy spname = ws.edit_value(rename);
  spname = trim_white(spname);
  if (spname != spectrum->name() && !spname.is_empty())
    {
      Project &proj = spectrum->session().project();
      Stringy uspname = proj.unique_spectrum_name(spname);
      spectrum->set_name(uspname);
      List vlist = proj.view_list(spectrum);
      for (int vi = 0 ; vi < vlist.size() ; ++vi)
	{
	  View *v = (View *) vlist[vi];
	  v->set_name(proj.unique_view_name(uspname, v->is_top_level_window()));
	}
    }

  Stringy molname = ws.edit_value(molecule->edit_field());
  Stringy condname = ws.edit_value(condition->edit_field());

  if (molname != spectrum->molecule()->name() ||
      condname != spectrum->condition()->name())
    {
      Molecule *m = session.project().define_molecule(molname);
      Condition *c = m->define_condition(condname);

      spectrum->set_condition(c);
    }

  double noise_level = ws.numeric_edit_value(noise);
  if (noise_level > 0)
    spectrum->set_noise_level(noise_level);

  int dim = spectrum->dimension();
  SPoint ppm_shift(dim);
  for (int a = 0 ; a < dim ; ++a)
    ppm_shift[a] = ws.numeric_text_field(ws.table_element(shifts,a+1,1));
  SPoint new_ppm_zero = spectrum->ppm_shift() - ppm_shift;
  spectrum->set_ppm_zero(new_ppm_zero);

  // Move ppm scale, not contours, in spectrum views
  List vlist = session.project().view_list(spectrum);
  shift_views(vlist, -new_ppm_zero);

  SPoint sweep_width = spectrum->ppm_sweep_width();
  for (int a = 0 ; a < dim ; ++a)
    {
      double sw = spectrum->scale(sweep_width[a], a, PPM, HZ);
      Stringy previous = formatted_string("%.3f", sw);
      Widget tf = ws.table_element(shifts,a+1,2);
      Stringy current = ws.text_field_string(tf);
      if (current != previous)
	{
	  double sw_hz = ws.numeric_text_field(tf);
	  double sw_ppm = spectrum->scale(sw_hz, a, HZ, PPM);
	  if (sw_ppm > 0)
	    sweep_width[a] = sw_ppm;
	}
    }
  spectrum->set_ppm_sweep_width(sweep_width);
  update_resonance_panels(vlist);
}
