# -----------------------------------------------------------------------------
# Routines read CCPN XML nmr data files into Sparky, and to write this format
# from Sparky.
#
import string

# -----------------------------------------------------------------------------
#
def write_study_info(s, out):

  out.write('%s version %s\n' % (s.dataModelName, s.dataModelVersion))
  if s.details:
    out.write('Details: %s\n' % s.details)
  if s.archiveDetails:
    out.write('Archive details: %s\n' % s.archiveDetails)

  for m in s.molecules:
    write_molecule_info(m, out)

  for eg in s.expGroups:
    write_experiment_group_info(eg, out)

# -----------------------------------------------------------------------------
#
def write_molecule_info(m, out):
    
  out.write('Molecule %d, %s, %s\n' % (m.serial, m.name, m.sourceInfo))

# -----------------------------------------------------------------------------
#
def write_experiment_group_info(eg, out):

  out.write('Experiment group %d\n' % eg.serial)
  for e in eg.experiments:
    write_experiment_info(e, out)

# -----------------------------------------------------------------------------
#
def write_experiment_info(e, out):

  for ds in e.dataSources:
    size = data_source_size(ds)
    size_string = tuple_string(size, '%d')
    nuclei = data_source_nuclei(ds)
    nuc_string = '(' + string.join(nuclei, ',') + ')'
    out.write('Experiment %d, %s, %d-D, %s, %s' %
              (e.serial, e.name, e.ndim, nuc_string, size_string))
    if len(ds.xpks) > 0:
      out.write(', %d peaks' % len(ds.xpks))
    if e.type:
      out.write(', %s' % e.type)
    if e.details:
      out.write(', %s' % e.details)
    if ds.name != e.name:
      out.write(', data source %d %s' % (ds.serial, ds.name))
    if ds.filename:
      out.write(', file %s' % ds.filename)
    out.write('\n')

    sf = data_source_spectrometer_freq(ds)
    sw = data_source_sweepwidth(ds)
    origin = data_source_ppm_origin(ds)
    bs = data_source_block_size(ds)
    out.write('SF %s, SW %s, Origin %s BS %s\n' %
              (tuple_string(sf, '%.2f'),
               tuple_string(sw, '%.2f'),
               tuple_string(origin, '%.2f'),
               tuple_string(bs, '%.2f')))
    
    for xp in ds.xpks:
      write_crosspeak_info(xp, out)
  
# -----------------------------------------------------------------------------
#
def write_crosspeak_info(xp, out):

  ndim = xp.dataSource.ndim
  pos = crosspeak_position(xp)
  pos_string = tuple_string(pos, '%.2f')
  out.write('  Cross peak %d, (%s)' % (xp.serial, pos_string))

  for assignment in crosspeak_assignments(xp):
    acomps = map(lambda ga: ga[0]+ga[1], assignment)
    astring = string.join(acomps, ', ')
    out.write(', (%s)' % astring)

  out.write('\n')

# -----------------------------------------------------------------------------
# Number of points along each axis of a spectrum.
#
def data_source_size(ds):

  ndim = ds.experiment.ndim
  size = [0] * ndim
  for dd in ds.dataDims:
    size[dd.dim-1] = dd.npoints
  return size

# -----------------------------------------------------------------------------
#
def data_source_block_size(ds):

  ndim = ds.experiment.ndim
  size = [0] * ndim
  for dd in ds.dataDims:
    size[dd.dim-1] = dd.npointsBlock
  return size

# -----------------------------------------------------------------------------
# Returns list of atomic symbols like 1H, 13C, 15N for spectrum dimensions.
#
def data_source_nuclei(ds):

  ndim = ds.experiment.ndim
  nuclei = ['?'] * ndim
  for dd in ds.dataDims:
    ed = dd.expDim
    if len(ed.expDimRefs) > 0:
      edr = ed.expDimRefs[0]    # Not clear why there can be more than one
      if len(edr.isotopes) > 0:
        iso = edr.isotopes[0]   # Maybe this handles 12C and 13C both present
        nuclei[dd.dim-1] = '%d%s' % (iso.massNumber, iso.nucleus.symbol)
  return nuclei

# -----------------------------------------------------------------------------
#
def data_source_spectrometer_freq(ds):

  ndim = ds.experiment.ndim
  sf = [0] * ndim
  for dd in ds.dataDims:
    ed = dd.expDim
    if len(ed.expDimRefs) > 0:
      edr = ed.expDimRefs[0]    # Not clear why there can be more than one
      sf[dd.dim-1] = edr.SF
  return sf

# -----------------------------------------------------------------------------
#
def data_source_sweepwidth(ds):

  ndim = ds.experiment.ndim
  sw = [0] * ndim
  for dd in ds.dataDims:
    if len(dd.dataDimRefs) > 0:
      ddr = dd.dataDimRefs[0]      # Take first of alternative referencings
      sw[dd.dim-1] = ddr.valuePerPoint * dd.npointsOrig
    else:
      pass      # could try getting sweepwidth from ExpDimRef
  return sw

# -----------------------------------------------------------------------------
#
def data_source_ppm_origin(ds):

  ndim = ds.experiment.ndim
  origin = [0] * ndim
  for dd in ds.dataDims:
    if len(dd.dataDimRefs) > 0:
      ddr = dd.dataDimRefs[0]      # Take first of alternative referencings
      origin[dd.dim-1] = ddr.refValue + ddr.refPoint * ddr.valuePerPoint
      # Not clear if refValue is always in ppm.  May use unit from ExpDimRef.
  return origin

# -----------------------------------------------------------------------------
#
def crosspeak_position(xp):

  ndim = xp.dataSource.ndim
  pos = [None] * ndim
  for xpd in xp.xpkDims:
    pos[xpd.dim-1] = xpd.positionPoints
  return pos

# -----------------------------------------------------------------------------
# Each assignment is a tuple of (group_name, atom_name).
#
def crosspeak_assignments(xp):

  from CcpnNmrData import SingleAssignment

  assignments = []
  ndim = xp.dataSource.ndim
  for xpc in xp.xpkContribs:
    assignment = [('','?')] * ndim
    for xpcd in xpc.xpkContribDims:
      a = xpcd.assignment
      if isinstance(a, SingleAssignment):
	r = a.uniqueResidue.molResidue
	rname = '%s%d' % (r.chemComponent.oneLetterCode, r.seqID)
	ag = a.uniqueAtomGroup
	if ag:
	  aname = ag.name
	else:
	  aname = '?'
        assignment[xpcd.dim-1] = (rname, aname)
    assignments.append(assignment)

  return assignments
  
# -----------------------------------------------------------------------------
#
def tuple_string(values, format):

  value_strings = map(lambda v, fmt=format: fmt % v, values)
  return '(' + string.join(value_strings, ', ') + ')'

# -----------------------------------------------------------------------------
#
def load_study(path):

  load_reference_data()
  from CcpnData2XML import importXML
  study = importXML(path)
  return study

# -----------------------------------------------------------------------------
#
loaded_reference_data = 0
ref_path = '/usr/local/src/sparky/other/ccpn/Reference.xml'

# -----------------------------------------------------------------------------
#
def load_reference_data():

  global loaded_reference_data
  if not loaded_reference_data:
    from CcpnData2XML import importXML
    ref = importXML(ref_path)
    loaded_reference_data = 1

# -----------------------------------------------------------------------------
#
def sparky_load_study(path):

  study = load_study(path)
  import sparky
  session = sparky.session_list[0]
  proj = session.project
  
  for eg in study.expGroups:
    for e in eg.experiments:
      for ds in e.dataSources:
	if len(ds.xpks) > 0:
	  sp = create_sparky_spectrum(ds, proj)
	  create_sparky_peaks(ds.xpks, sp)
	  v = create_sparky_view(sp)
	  v.is_shown = 0

# -----------------------------------------------------------------------------
#
def create_sparky_spectrum(ds, proj):

  name = ds.name
  size = data_source_size(ds)
  bsize = data_source_block_size(ds)
  sfreq = data_source_spectrometer_freq(ds)
  swidth_ppm = data_source_sweepwidth(ds)
  swidth = map(lambda f, w: f*w, sfreq, swidth_ppm)	# Hz
  origin_ppm = data_source_ppm_origin(ds)
  axis_labels = data_source_nuclei(ds)
  sp = proj.create_dataless_spectrum(name, size, bsize,
				     sfreq, swidth, origin_ppm,
				     axis_labels)
  return sp

# -----------------------------------------------------------------------------
#
def create_sparky_peaks(xpks, sp):
  
  ppm_per_index = map(lambda w,s: float(w)/s, sp.spectrum_width, sp.data_size)
  origin_ppm = sp.region[1]

  plist = []
  for xpk in xpks:
    pos_index = crosspeak_position(xpk)
    pos_ppm = map(lambda i, o, ppi: o - i * ppi,
		  pos_index, origin_ppm, ppm_per_index)
    for assignment in crosspeak_assignments(xpk):
      #
      # For each assignment create a separate peak marker since Sparky
      # cannot handle multiple assignments on one peak.
      #
      p = sp.place_peak(pos_ppm)
      for a in range(len(assignment)):
	group_name, atom_name = assignment[a]
	p.assign(a, group_name, atom_name)
      p.show_assignment_label()
      plist.append(p)

  return plist

# -----------------------------------------------------------------------------
#
def create_sparky_view(sp):

  v = sp.session.create_view(None, sp)
  v.show_scrollbars = 1
  v.show_scales = 1
  return v

# -----------------------------------------------------------------------------
#
#study_path = '/usr/local/src/sparky/other/ccpn/M31Csmall.xml'
study_path = '/usr/local/src/sparky/other/ccpn/M31C.xml'
study = load_study(study_path)

#import sys
#write_study_info(study, sys.stdout)

#sparky_load_study(study_path)
