/*
 * A CrossPeak object is a generic "peak" in an NMR spectrum.
 *
 * The CrossPeak class is subclassed into "Peak" and "PeakGp" (peak group)
 * classes. The Peak class is used for a single extremum peaks that arise in
 * most NMR spectra. The COESY and related spectra are different because a
 * crosspeak is characterised by a pattern of nodal and anti-nodal extrema
 * arrayed about a central point. The central point is the actual NMR "peak"
 * location. This cluster of extrema is handled as individual Peaks grouped
 * together into a PeakGp (peak group) object.
 *
 * CrossPeaks handle generic operations such as: labeling of the crosspeak,
 * integrating the crosspeak, making assignments, returning the crosspeak
 * frequency, and returning the crosspeak location (which may differ from
 * the frequency due to "aliasing" of the frequency axes).
 *
 */
#ifndef _CrossPeak_h
#define _CrossPeak_h

class CrossPeak;

#include "ornament.h"
#include "spoint.h"		// Use SPoint
#include "stringc.h"		// use Stringy

class Label;
class List;
class Rectangle;
class Resonance;

// ----------------------------------------------------------------------------
//
class CrossPeak : public Ornament
{
public:
  virtual Ornament_Type	type() const;
  virtual const char	*type_name() const;

  SPoint		position() const;
  virtual void		set_position(const SPoint &);

  SPoint		alias() const;
  virtual void		set_alias(const SPoint &);
  bool			IsAliased() const;
  bool			IsAliased(int a) const;

  SPoint		frequency() const;	// freq = position + alias
  double		frequency(int a) const;

  virtual bool		volume(double *vol = 0, Stringy *method = 0) const;
  bool			volume_error(double *fraction = 0,
				     Stringy *method = 0) const;
  void			set_volume_error(double fraction,
					 const Stringy &method);

  virtual double	SignalToNoise(void) const = 0;

  virtual Resonance *	resonance(int a) const;
  virtual void		SetResonance(int a, Resonance *);
  Resonance *		ChangeAssignment(int a,
					 const Stringy &atom,
					 const Stringy &group);
  void			DeleteAssignment();
  bool			IsAssigned() const;
  bool			is_fully_assigned() const;
  void			update_resonances() const;

  Stringy		assignment_name() const;	// Full name
  virtual Stringy	assignment_name(const Stringy &format) const;

  Label			*label() const;
  void			label(Label *lbl);
  Stringy		user_name() const;	// Non-assignment label text

  virtual void		display(ODraw &) const;
  virtual SRegion	erasure_region(ODraw &) const;
  virtual SRegion	selection_region(ODraw &) const;
  Rectangle		marker_square(ODraw &) const;

protected:

  CrossPeak(Spectrum *sp, const SPoint &position);
  virtual	~CrossPeak();

private:

  Resonance		*mResonances[DIM];	// the assignment
  SPoint		mPosition;		// the peak position
  SPoint		mFrequency;		// the peak frequency
  SPoint		mAlias;			// the alias

  Label			*mLabel;
  double		mVolumeError;		// fraction
  Stringy		mVolumeErrorMethod;

  void			update_frequency();
};

bool is_crosspeak(Ornament *op);
bool is_peaklet(Ornament *op);
int compare_assignments(const CrossPeak *xp1, const CrossPeak *xp2, int axis);

#endif
