// ----------------------------------------------------------------------------
//
#include <stdio.h>		// use FILE
#include <string.h>		// use strncmp()

#include "atom.h"		// use Atom
#include "group.h"		// use Group
#include "list.h"		// use List
#include "memalloc.h"		// use new()
#include "model.h"		// use Model
#include "spoint.h"		// use SPoint
#include "stringc.h"		// use Stringy
#include "table.h"		// use Table

// ----------------------------------------------------------------------------
// Fields from PDB file ATOM line.
//
class Atom_Entry
{
public:
  static bool equal(const void *ae1, const void *ae2);
  static unsigned long hash(const void *ae);

  int atom_serial_number;
  char atom_name[5];
  char alt_loc;
  char residue_name[5];
  char residue_chain;
  int seq_number;
  char seq_insert;
  double xyz[3];
  double occupancy;
  double temp_factor;
  int footnote_num;

  Stringy name;
};

// ----------------------------------------------------------------------------
//
static bool parse_pdb_atom(const char *line, Atom_Entry *ae);

// ----------------------------------------------------------------------------
//
Model::Model(const Stringy &path, const List &atom_entries)
  : atom_table(Atom_Entry::equal, Atom_Entry::hash)
{
  this->path = path;

  //
  // Ownership of elements of atom_entries taken by Model.
  //
  for (int ei = 0 ; ei < atom_entries.size() ; ++ei)
    atom_table.insert((TableKey) atom_entries[ei],
		      (TableData) atom_entries[ei]);
}

// ----------------------------------------------------------------------------
//
Model::~Model()
{
  List values = atom_table.values();
  for (int vi = 0 ; vi < values.size() ; ++vi)
    delete (Atom_Entry *) values[vi];
  atom_table.erase();
}

// ----------------------------------------------------------------------------
//
Stringy Model::file_path() const
{
  return path;
}

// ----------------------------------------------------------------------------
//
bool Model::coordinates(const Atom &a, SPoint *xyz) const
{
  Atom_Entry ae;

  ae.name = a.name();
  ae.seq_number = a.group()->number();
  TableData te;
  if (atom_table.find((TableKey) &ae, &te))
    {
      *xyz = SPoint(3, ((Atom_Entry *) te)->xyz);
      return true;
    }
  return false;
}

// ----------------------------------------------------------------------------
//
bool Atom_Entry::equal(const void *ae1, const void *ae2)
{
  Atom_Entry *e1 = (Atom_Entry *) ae1;
  Atom_Entry *e2 = (Atom_Entry *) ae2;
  return (e1->seq_number == e2->seq_number && e1->name == e2->name);
}

// ----------------------------------------------------------------------------
//
unsigned long Atom_Entry::hash(const void *ae)
{
  Atom_Entry *e = (Atom_Entry *) ae;
  return hash_combine((unsigned long) e->seq_number, hash_string(&e->name));
}

// ----------------------------------------------------------------------------
//
#define MAX_LINE_LENGTH 4096
Model *read_pdb_model(const Stringy &pdb_path)
{
  FILE *pdb_fp = fopen(pdb_path.cstring(), "r");
  if (!pdb_fp)
    return NULL;

  List entries;
  Atom_Entry ae;
  char line[MAX_LINE_LENGTH];
  while (fgets(line, sizeof line, pdb_fp))
    if (parse_pdb_atom(line, &ae))
      entries.append(new Atom_Entry(ae));

  fclose(pdb_fp);

  if (entries.size() == 0)
    return NULL;

  Model *m = new Model(pdb_path, entries);
  return m;
}

// ----------------------------------------------------------------------------
//
static bool parse_pdb_atom(const char *line, Atom_Entry *ae)
{
  if (strncmp(line, "ATOM", 4) == 0 &&
      sscanf(line, "%*6c %5d %4c%c%4c%c%4d%c   %8lf%8lf%8lf%6lf%6lf %3d",
	     &ae->atom_serial_number, ae->atom_name, &ae->alt_loc,
	     ae->residue_name, &ae->residue_chain, &ae->seq_number,
	     &ae->seq_insert, &ae->xyz[0], &ae->xyz[1], &ae->xyz[2],
	     &ae->occupancy, &ae->temp_factor, &ae->footnote_num) >= 10)
    {
      ae->atom_name[4] = '\0';
      ae->residue_name[4] = '\0';
      ae->name = trim_white(ae->atom_name);
      return true;
    }
  return false;
}
