/*
 * SaveFile.C	- Implementation of the SaveFile class
 *
 * A SaveFile is the primitive means by which the Sparky database
 * is saved and restored from disk. The three kinds of SaveFiles
 * are: Project files, Spectrum Annotation files, and Static
 * Resonance files.
 *
 */
#include <fstream>		// Use ofstream
#include <iostream>		// use ostream
#include <string.h>		// use strstr()

#include "memalloc.h"		// use new()
#include "project.h"		// use Project
#include "reporter.h"		// use Reporter
#include "savefile.h"
#include "session.h"		// use Session
#include "system.h"		// use file_directory(), file_exists(), ...
#include "uimain.h"		// Use query()
#include "version.h"		// Use SPARKY_VERSION

#define MAX_LINE_LENGTH		4096

static void put_prolog(FILE *fp, SAVEFILE_TYPE t);
static void put_prolog(std::ostream &s, SAVEFILE_TYPE t);
static bool read_prolog(FILE *fp, SAVEFILE_TYPE t,
			int *versionp, int *releasep);
static bool confirm_overwrite(Session &, SAVEFILE_TYPE t, const Stringy &path);
static Stringy strip_directory(const Stringy &dir, const Stringy &path);
static bool file_save_previous(const Stringy &path, Reporter &);
static bool file_revert_to_previous(const Stringy &path, Reporter &);

/*
 * Map between the SAVEFILE_TYPE <t> and its string name.
 */
static const char *type_name(SAVEFILE_TYPE t)
{
  switch (t)
    {
    case SAVEFILE_PROJECT: return "project";
    case SAVEFILE_SPECTRUM: return "save";
    }
  return "unknown";
}

// ----------------------------------------------------------------------------
//
Stringy SaveFile::directory() const
  { return file_directory(path()); }

// ----------------------------------------------------------------------------
//
bool SaveFile::NeedsSaving() const
  { return mNeedsSaving || !mBackupInUse.is_empty(); }
void SaveFile::NeedsSaving(bool state)
  { mNeedsSaving = state; mNeedsBackup = state; }

/*
 * Return true if it's OK to overwrite the SaveFile named <nm> of
 * SaveFile type <t>. <t> is used only for the diagnostic message.
 */
static bool confirm_overwrite(Session &s, SAVEFILE_TYPE t, const Stringy &nm)
{
	/*
	 * If the user wants checking of overwriting, and if <nm> exists, then
	 * warn it will be overwritten and get confirmation to overwrite it.
	 */
  Preferences &pref = s.project().preferences;
	if (pref.prompt_before_overwrite && file_exists(nm)) {
	  Stringy msg = formatted_string("%s file %s exists.\n"
					 "Overwrite file?",
					 type_name(t), nm.cstring());
	  int choice = query(s, msg, "OK", "OK and don't ask again", "Cancel");
	  if (choice == 2)
	    pref.prompt_before_overwrite = false;

	  return (choice == 1 || choice == 2);
	}
	return true;
}

/*
 * Construct a SaveFile of type <t>
 */
SaveFile::SaveFile(Session &s, SAVEFILE_TYPE t, bool save_previous)
  : session(s)
{
	mType = t;
	mSavePrevious = save_previous;
	mNeedsSaving = false;
	mNeedsBackup = false;
}

// ----------------------------------------------------------------------------
//
SaveFile &SaveFile::operator=(const SaveFile &sf)
{
  mType = sf.mType;
  mPath = sf.mPath;
  mSavePrevious = sf.mSavePrevious;
  mNeedsSaving = sf.mNeedsSaving;
  mNeedsBackup = sf.mNeedsBackup;
  mBackupInUse = sf.mBackupInUse;

  return *this;
}

// ----------------------------------------------------------------------------
//
Stringy SaveFile::path() const
  { return mPath; }

/*
 * Set the SaveFile's on-disk location to <nm>
 */
void SaveFile::set_path(const Stringy &path)
{
  Stringy p = tilde_expand(path.cstring());
  if (!is_absolute_path(p))
    p = file_path(current_directory(), p);
  p = dedotify_path(p);
  if (!same_file(p, mPath))
    {
      mPath = p;
      NeedsSaving(true);
    }
}

// ----------------------------------------------------------------------------
//
Stringy SaveFile::ShortPath() const
{
  switch (mType)
    {
    case SAVEFILE_SPECTRUM:
      return strip_directory(session.project().save_path(""), path());
    case SAVEFILE_PROJECT:
      return strip_directory(session.project().project_path(""), path());
    }

  return path();
}

// ----------------------------------------------------------------------------
//
static Stringy strip_directory(const Stringy &dir, const Stringy &path)
{
  Stringy d = dir + path_separator();
  if (strstr(path.cstring(), d.cstring()) == path.cstring())
    return path.cstring() + d.length();
  else
    return path;
}

// ----------------------------------------------------------------------------
//
Stringy SaveFile::auto_backup_path()
  { return path() + ".backup"; }

// ----------------------------------------------------------------------------
//
static void add_string(const Stringy &s, List &strings)
  {strings.append(new Stringy(s)); }
static bool find_string(const Stringy &s, List &strings)
{
  for (int si = 0 ; si < strings.size() ; ++si)
    if (*((Stringy *) strings[si]) == s)
      return true;

  return false;
}
static void remove_string(const Stringy &s, List &strings)
{
  for (int si = 0 ; si < strings.size() ; ++si)
    {
      Stringy *string = (Stringy *) strings[si];
      if (*string == s)
	{
	  strings.erase(si);
	  delete string;
	  return;
	}
    }
}

// ----------------------------------------------------------------------------
//
bool SaveFile::save_path(bool backup, Stringy *spath)
{
  Reporter &rr = session.reporter();
  if (backup)
    {
      if (!mNeedsBackup)
	return false;

      if (old_backup_exists() || !user_owns_file(path()))
	return false;

      if (!find_string(auto_backup_path(), backup_paths))
	add_string(auto_backup_path(), backup_paths);

      rr.message("Backing up %s\n", ShortPath().cstring());

      *spath = auto_backup_path();
    }
  else
    {
      if (!NeedsSaving())
	return false;

      if (!confirm_overwrite(session, mType, path()))
	return false;

      if (mSavePrevious && !file_save_previous(path(), rr))
	return false;

      rr.message("Saving %s\n", ShortPath().cstring());

      *spath = path();
    }

  return true;
}

/*
 * Open a SaveFile for saving, returning a non-NULL FILE * if success
 */
FILE *SaveFile::OpenForSaving(bool backup)
{
  Stringy path;
  if (!save_path(backup, &path))
    return NULL;

  // Write text file in binary mode so I don't get CRLF line termination
  FILE *fp = fopen(path.cstring(), "wb");
  if (fp == NULL)
    {
      Reporter &rr = session.reporter();
      rr.warning("Can't open %s file %s", type_name(mType), path.cstring());
      return NULL;
    }

  put_prolog(fp, mType);

  return fp;
}

// ----------------------------------------------------------------------------
//
std::ostream *SaveFile::OpenForWriting(bool backup)
{
  Stringy path;
  if (!save_path(backup, &path))
    return NULL;

  std::ostream *s = new std::ofstream(path.cstring(), std::ios::out);
  if (!s->good())
    {
      delete s;
      Reporter &rr = session.reporter();
      rr.warning("Can't open %s file %s", type_name(mType), path.cstring());
      return NULL;
    }

  put_prolog(*s, mType);

  return s;
}

// ----------------------------------------------------------------------------
//
List SaveFile::backup_paths;	// state variable

// ----------------------------------------------------------------------------
//
bool SaveFile::old_backup_exists()
{
  return (!find_string(auto_backup_path(), backup_paths) &&
	  file_exists(auto_backup_path()));
}

// ----------------------------------------------------------------------------
//
void SaveFile::remove_backup()
{
  Stringy path = auto_backup_path();
  if (find_string(path, backup_paths))
    {
      remove_file(path);
      remove_string(path, backup_paths);
    }
}

// ----------------------------------------------------------------------------
//
void SaveFile::remove_auto_backups()
{
  for (int bfi = 0 ; bfi < backup_paths.size() ; ++bfi)
    {
      Stringy *path = (Stringy *) backup_paths[bfi];
      remove_file(*path);
      delete path;
    }
  backup_paths.erase();
}

// ----------------------------------------------------------------------------
//
Stringy SaveFile::load_path()
{
  if (mBackupInUse.is_empty() && old_backup_exists() &&
      session.project().use_backup_file(auto_backup_path()))
    mBackupInUse = auto_backup_path();

  return mBackupInUse.is_empty() ? path() : mBackupInUse;
}

// ----------------------------------------------------------------------------
//
FILE *SaveFile::OpenForReading()
{
  FILE *fp = fopen(load_path().cstring(), "r");	// Sparky data
  if (fp)
    {
      int ver, rel;
      if (!read_prolog(fp, mType, &ver, &rel))
	{
	  fclose(fp);
	  fp = NULL;
	}
    }
  return fp;
}

/*
 * Close the SaveFile, returning true if successful
 */
bool SaveFile::EndRead(FILE *fp)
{
  bool success = (ferror(fp) == 0);
  if (fclose(fp))
    success = false;

  return success;
}

// ----------------------------------------------------------------------------
//
bool SaveFile::EndSave(FILE *fp, bool backup, bool success)
{
  success = success && (ferror(fp) == 0);
  if (fclose(fp))
    success = false;
  end_of_write(backup, success);

  return success;
}

// ----------------------------------------------------------------------------
//
void SaveFile::end_of_write(bool backup, bool success)
{
  if (success && backup)
      mNeedsBackup = false;

  if (success && !backup)
    {
      NeedsSaving(false);
      if (!mBackupInUse.is_empty() ||
	  find_string(auto_backup_path(), backup_paths))
	{
	  remove_string(mBackupInUse, backup_paths);
	  remove_file(mBackupInUse);
	  mBackupInUse = "";
	}
    }

  if (!success && !backup)
    file_revert_to_previous(mPath.cstring(), session.reporter());
}

// ----------------------------------------------------------------------------
//
bool SaveFile::EndWrite(std::ostream *s, bool backup, bool success)
{
  success = success && s->good();
  delete s;
  end_of_write(backup, success);

  return success;
}

/*
 * Put the file identifier for this SaveFile to <fp>
 */
static void put_prolog(FILE *fp, SAVEFILE_TYPE t)
{
  fprintf(fp, "<sparky %s file>\n", type_name(t));
  fprintf(fp, "<version %d.%d>\n", SPARKY_VERSION, SPARKY_RELEASE);
}

/*
 * Put the file identifier for this SaveFile to <fp>
 */
static void put_prolog(std::ostream &s, SAVEFILE_TYPE t)
{
  s	<< "<sparky " << type_name(t) << " db>\n";
  s	<< "<version " << SPARKY_VERSION << "." << SPARKY_RELEASE << ">\n";
}

/*
 * Get the file identifier from <fp>, assuming it is of type <t>.
 * If <fp> does point to a file of type <t>, return true and fill
 * in <versionp> and <releasep> if non-NULL.
 */
static bool read_prolog(FILE *fp, SAVEFILE_TYPE t,
			int *versionp, int *releasep)
{
	char	buf[MAX_LINE_LENGTH], type[32];
	int	ig;

	/*
	 * Point <versionp> to <releasep> to <ig>nore if they are NULL.
	 */
	if (versionp == NULL)
		versionp = &ig;
	if (releasep == NULL)
		releasep = &ig;

	/*
	 * Get the prolog.
	 */
	return (fgets(buf, sizeof buf, fp) &&
		sscanf(buf, "<sparky %s", type) == 1 &&
		strcmp(type, type_name(t)) == 0 &&
		fgets(buf, sizeof buf, fp) &&
		sscanf(buf, "<version %d.%d>", versionp, releasep) == 2);
}

/*
 * Rename a file from <filename> to <filename>.BAK
 */
static bool file_save_previous(const Stringy &path, Reporter &rr)
{
	if (file_exists(path)) {
		Stringy previous = path + ".BAK";
		remove_file(previous);
		if (!move_file(path, previous)) {
			rr.warning("Can't rename\n%s to\n%s",
				   path.cstring(), previous.cstring());
			return false;
		}
	}
	return true;
}

/*
 * Rename a file from <filename>.BAK to <filename>
 */
static bool file_revert_to_previous(const Stringy &path, Reporter &rr)
{
	Stringy previous = path + ".BAK";

	if (file_exists(previous)) {
	  remove_file(path);
	  if (!move_file(previous, path)) {
	    rr.warning("Can't rename\n%s to\n%s",
		       previous.cstring(), path.cstring());
	    return false;
	  }
	}
	return true;
}

// ----------------------------------------------------------------------------
//
bool check_file_type(const Stringy &path, SAVEFILE_TYPE t,
		     int *version, int *release)
{
  FILE *fp = fopen(path.cstring(), "r");		// File type check
  if (fp)
    {
      bool success = read_prolog(fp, t, version, release);
      fclose(fp);
      return success;
    }
  return false;
}
