// ----------------------------------------------------------------------------
// Define the main window menus and popup view menu.
//

#include "command.h"		// use Command_Dispatcher
#include "memalloc.h"		// use new()
#include "notifier.h"		// use Notifier
#include "project.h"		// use Project
#include "session.h"		// use Session
#include "uidialog.h"		// use Dialog, Dialog_Table
#include "uimenu.h"
#include "uiview.h"		// Use View
#include "winsystem.h"		// use WinSys

// ----------------------------------------------------------------------------
//
class Menu_Entry
{
public:
  Menu_Entry(Command_Dispatcher &, const Stringy &);

  Command_Dispatcher &command_dispatcher;
  Stringy accelerator;
};

// ----------------------------------------------------------------------------
//
static void create_file_menu(Session &, Widget menubar);
static void create_view_menu(Session &, Widget menubar);
static void create_peak_menu(Session &, Widget menubar);
static void create_windows_menu(Session &, Widget menubar);
static void create_help_menu(Session &, Widget menubar);

static void view_added_cb(void *view_pane, void *view);
static void view_deleted_cb(void *view_pane, void *view);
static void view_renamed_cb(void *view_pane, void *view);
static void update_view_menu(Widget pane, Session &s);
static void show_view_cb(Widget, CB_Data view, CB_Data);

static void add_menu_path(Session &s, Widget pane,
			  const Stringy &path, const Stringy &accelerator);
static void delete_menu_path(Session &, Widget pane, const Stringy &path);
static void menu_entry(Session &, Widget pane,
		       const Stringy &name, const Stringy &accelerator);
static void destroy_menu_callback_cb(Widget, CB_Data accel, CB_Data);
static void accelerator_cb(Widget, CB_Data, CB_Data);

// ----------------------------------------------------------------------------
//
Widget create_main_menubar(Session &s, Widget main_window)
{
  WinSys &ws = s.window_system();
  Widget menubar = ws.create_menu_bar(main_window, "menuBar");

  create_file_menu(s, menubar);
  create_view_menu(s, menubar);
  create_peak_menu(s, menubar);
  create_windows_menu(s, menubar);

  Widget emenu = ws.add_menu(menubar, "extensionMenuButton", "extensionMenu");
  s.dialog_table().set_dialog("main_extension_menu", emenu);

  create_help_menu(s, menubar);

  return menubar;
}

// ----------------------------------------------------------------------------
//
static void create_file_menu(Session &s, Widget menubar)
{
  WinSys &ws = s.window_system();
  Widget pane = ws.add_menu(menubar, "fileMenuButton", "cmdMenu");

  menu_entry(s, pane, "open", "fo");
  menu_entry(s, pane, "save", "fs");
  menu_entry(s, pane, "saveAs", "fa");

  ws.menu_separator(pane, "separator");

  ws.menu_label(pane, "projHeading");

  menu_entry(s, pane, "projOpen", "jo");
  menu_entry(s, pane, "projSave", "js");
  menu_entry(s, pane, "projSaveAs", "ja");
  menu_entry(s, pane, "projClose", "jc");

  ws.menu_separator(pane, "separator");

  menu_entry(s, pane, "print", "pt");
  menu_entry(s, pane, "preferences", "pf");

  ws.menu_separator(pane, "separator");

  menu_entry(s, pane, "iconify", "ic");
  menu_entry(s, pane, "quit", "qt");
}

// ----------------------------------------------------------------------------
//
static void create_view_menu(Session &s, Widget menubar)
{
  WinSys &ws = s.window_system();
  Widget pane = ws.add_menu(menubar, "viewMenuButton", "cmdMenu");

  menu_entry(s, pane, "dupview", "vd");
  menu_entry(s, pane, "hideview", "vh");
  menu_entry(s, pane, "deleteview", "vD");
  menu_entry(s, pane, "projectviews", "pv");
  menu_entry(s, pane, "refresh", "vr");
  menu_entry(s, pane, "view", "vt");
  menu_entry(s, pane, "pointerModes", "pm");

  ws.menu_separator(pane, "separator");

  menu_entry(s, pane, "contourscale", "vC");
  menu_entry(s, pane, "resonance", "vR");
  menu_entry(s, pane, "slice", "vS");
  menu_entry(s, pane, "peakinfo", "vp");
  menu_entry(s, pane, "axisscale", "vs");

  ws.menu_separator(pane, "separator");

  Widget axes_pane = ws.cascade_pane(pane, "axesCascade");
  menu_entry(s, axes_pane, "ppm", "xp");
  menu_entry(s, axes_pane, "hz", "xh");
  menu_entry(s, axes_pane, "data", "xd");
  menu_entry(s, axes_pane, "atoms", "xa");
  menu_entry(s, axes_pane, "roll", "xr");
  menu_entry(s, axes_pane, "transpose", "xx");
  menu_entry(s, axes_pane, "origin", "xO");
  menu_entry(s, axes_pane, "shiftaxes", "st");

  Widget crosshair_pane = ws.cascade_pane(pane, "crosshairsCascade");
  menu_entry(s, crosshair_pane, "xhair", "hh");
  menu_entry(s, crosshair_pane, "transxhair", "ht");
  menu_entry(s, crosshair_pane, "otherxhair", "ho");

  Widget edge_pane = ws.cascade_pane(pane, "edgePanelsCascade");
  menu_entry(s, edge_pane, "resonance", "vR");
  menu_entry(s, edge_pane, "resfilter", "rf");
  menu_entry(s, edge_pane, "slice", "vS");
  menu_entry(s, edge_pane, "slicescale", "sa");
  menu_entry(s, edge_pane, "slicesubtract", "sf");
  menu_entry(s, edge_pane, "peakinfo", "vp");
  menu_entry(s, edge_pane, "contourscale", "vC");
  menu_entry(s, edge_pane, "axisscale", "vs");
  menu_entry(s, edge_pane, "scrollbars", "vb");

  Widget goto_pane = ws.cascade_pane(pane, "gotoCascade");
  menu_entry(s, goto_pane, "centerview", "vc");
  menu_entry(s, goto_pane, "goto", "rgX");
  menu_entry(s, goto_pane, "region", "rt");

  Widget object_pane = ws.cascade_pane(pane, "showCascade");
  menu_entry(s, object_pane, "showall", "os");
  menu_entry(s, object_pane, "showpeak", "ps");
  menu_entry(s, object_pane, "showpeakgp", "xs");
  menu_entry(s, object_pane, "showlabel", "ls");
  menu_entry(s, object_pane, "showline", "is");
  menu_entry(s, object_pane, "showgrid", "gs");

  Widget zoom_pane = ws.cascade_pane(pane, "zoomCascade");
  menu_entry(s, zoom_pane, "zoomin", "zi");
  menu_entry(s, zoom_pane, "zoomout", "zo");
  menu_entry(s, zoom_pane, "zoomdown", "zd");
  menu_entry(s, zoom_pane, "zoomup", "zu");
  menu_entry(s, zoom_pane, "zoomfull", "zf");
  menu_entry(s, zoom_pane, "zoomprev", "zp");
  menu_entry(s, zoom_pane, "zoomnext", "zn");
  menu_entry(s, zoom_pane, "zoomremember", "zr");

  Widget more_pane = ws.cascade_pane(pane, "moreCascade");
  menu_entry(s, more_pane, "contour", "ct");
  menu_entry(s, more_pane, "midas", "mi");
  menu_entry(s, more_pane, "ortho", "ov");
  menu_entry(s, more_pane, "overlay", "ol");
  menu_entry(s, more_pane, "renumber", "rv");
  menu_entry(s, more_pane, "rightlabels", "lr");
  menu_entry(s, more_pane, "spectrum", "st");
  menu_entry(s, more_pane, "subtractfit", "vf");
  menu_entry(s, more_pane, "synchronize", "yt");
  menu_entry(s, more_pane, "unoverlap", "lu");
  menu_entry(s, more_pane, "viewdepth", "vz");
}

// ----------------------------------------------------------------------------
//
static void create_peak_menu(Session &s, Widget menubar)
{
  WinSys &ws = s.window_system();
  Widget pane = ws.add_menu(menubar, "peakMenuButton", "cmdMenu");

  menu_entry(s, pane, "assign", "at");
  menu_entry(s, pane, "integrate", "it");
  menu_entry(s, pane, "peaklist", "lt");
  menu_entry(s, pane, "peakpick", "kt");

  ws.menu_separator(pane, "separator");

  Widget alias_pane = ws.cascade_pane(pane, "aliasCascade");
  menu_entry(s, alias_pane, "aliasaddw1", "a1");
  menu_entry(s, alias_pane, "aliasaddw2", "a2");
  menu_entry(s, alias_pane, "aliasaddw3", "a3");
  menu_entry(s, alias_pane, "aliasaddw4", "a4");
  menu_entry(s, alias_pane, "aliassubw1", "A1");
  menu_entry(s, alias_pane, "aliassubw2", "A2");
  menu_entry(s, alias_pane, "aliassubw3", "A3");
  menu_entry(s, alias_pane, "aliassubw4", "A4");
  menu_entry(s, alias_pane, "aliasdownw1", "d1");
  menu_entry(s, alias_pane, "aliasdownw2", "d2");
  menu_entry(s, alias_pane, "aliasdownw3", "d3");
  menu_entry(s, alias_pane, "aliasdownw4", "d4");
  menu_entry(s, alias_pane, "aliasupw1", "u1");
  menu_entry(s, alias_pane, "aliasupw2", "u2");
  menu_entry(s, alias_pane, "aliasupw3", "u3");
  menu_entry(s, alias_pane, "aliasupw4", "u4");
  menu_entry(s, alias_pane, "unalias", "ua");

  Widget assignment_pane = ws.cascade_pane(pane, "assignmentCascade");
  menu_entry(s, assignment_pane, "assign", "at");
  menu_entry(s, assignment_pane, "assignguess", "ag");
  menu_entry(s, assignment_pane, "assigntable", "tb");
  menu_entry(s, assignment_pane, "resassign", "ra");
  menu_entry(s, assignment_pane, "assigncopy", "ac");
  menu_entry(s, assignment_pane, "assignpaste", "aP");
  menu_entry(s, assignment_pane, "assignpastelabel", "ap");
  menu_entry(s, assignment_pane, "xdiagassigncopy", "aX");
  menu_entry(s, assignment_pane, "acopymode", "co");
  menu_entry(s, assignment_pane, "assigndelete", "aD");
  menu_entry(s, assignment_pane, "labelpeaks", "lb");
  menu_entry(s, assignment_pane, "unlabelpeaks", "lB");

  Widget copy_pane = ws.cascade_pane(pane, "copyCascade");
  menu_entry(s, copy_pane, "ocopy", "oc");
  menu_entry(s, copy_pane, "opaste", "op");
  menu_entry(s, copy_pane, "xocopy", "oX");
  menu_entry(s, copy_pane, "assigncopy", "ac");
  menu_entry(s, copy_pane, "assignpaste", "aP");
  menu_entry(s, copy_pane, "assignpastelabel", "ap");
  menu_entry(s, copy_pane, "xdiagassigncopy", "aX");
  menu_entry(s, copy_pane, "acopymode", "co");

  Widget integrate_pane = ws.cascade_pane(pane, "integrateCascade");
  menu_entry(s, integrate_pane, "integrate", "it");
  menu_entry(s, integrate_pane, "box", "ib");
  menu_entry(s, integrate_pane, "ellipse", "ie");
  menu_entry(s, integrate_pane, "gaussian", "ig");
  menu_entry(s, integrate_pane, "lorentzian", "il");
  menu_entry(s, integrate_pane, "integpeak", "pi");
  menu_entry(s, integrate_pane, "unintegrate", "pU");

  Widget res_pane = ws.cascade_pane(pane, "resonanceCascade");
  menu_entry(s, res_pane, "reslist", "rl");
  menu_entry(s, res_pane, "resassign", "ra");
  menu_entry(s, res_pane, "renamereson", "rr");
  menu_entry(s, res_pane, "deleteres", "dr");
  menu_entry(s, res_pane, "resonancew1", "r1");
  menu_entry(s, res_pane, "resonancew2", "r2");
  menu_entry(s, res_pane, "resonancew3", "r3");
  menu_entry(s, res_pane, "resonancew4", "r4");
  menu_entry(s, res_pane, "predefined", "pd");

  Widget sel_pane = ws.cascade_pane(pane, "selectCascade");
  menu_entry(s, sel_pane, "selectall", "pa");
  menu_entry(s, sel_pane, "selectfassign", "pF");
  menu_entry(s, sel_pane, "selectpassign", "pP");
  menu_entry(s, sel_pane, "selectuassign", "pN");
  menu_entry(s, sel_pane, "selectcolor", "pC");
  menu_entry(s, sel_pane, "selectinvert", "pI");
  menu_entry(s, sel_pane, "selectaliased", "pA");
  menu_entry(s, sel_pane, "selectintra", "pR");
  menu_entry(s, sel_pane, "selectseq", "pS");
  menu_entry(s, sel_pane, "selectmedium", "pM");
  menu_entry(s, sel_pane, "selectlong", "pL");

  Widget more_pane = ws.cascade_pane(pane, "moreCascade");
  menu_entry(s, more_pane, "center", "pc");
  menu_entry(s, more_pane, "color", "cr");
  menu_entry(s, more_pane, "linewidth", "pe");
  menu_entry(s, more_pane, "group", "pg");
  menu_entry(s, more_pane, "labelpeak", "pl");
  menu_entry(s, more_pane, "listselected", "sl");
  menu_entry(s, more_pane, "lock", "pk");
  menu_entry(s, more_pane, "note", "nt");
  menu_entry(s, more_pane, "newlist", "nl");
  menu_entry(s, more_pane, "ornament", "ot");
  menu_entry(s, more_pane, "osize", "oz");
  menu_entry(s, more_pane, "unlock", "pu");
  menu_entry(s, more_pane, "undo", "eu");
  menu_entry(s, more_pane, "height", "ph");
}

// ----------------------------------------------------------------------------
//
static void create_windows_menu(Session &s, Widget menubar)
{
  WinSys &ws = s.window_system();
  Widget pane = ws.add_menu(menubar, "windowsMenuButton", "cmdMenu");

  //
  // For adding entries for spectrum view windows
  //
  Notifier &n = s.notifier();
  n.notify_me(nt_added_view_to_project, view_added_cb, pane);
  n.notify_me(nt_removed_view_from_project, view_deleted_cb, pane);
  n.notify_me(nt_renamed_view, view_renamed_cb, pane);
}

// ----------------------------------------------------------------------------
//
static void view_added_cb(void *view_pane, void *view)
{
  View *v = (View *) view;
  if (v->is_top_level_window())
    {
      WinSys &ws = v->session().window_system();
      ws.add_menu_button_at_top((Widget) view_pane, v->name(),
				show_view_cb, v);
    }
}

// ----------------------------------------------------------------------------
//
static void view_deleted_cb(void *view_pane, void *view)
{
  View *v = (View *) view;
  if (v->is_top_level_window())
    {
      WinSys &ws = v->session().window_system();
      ws.delete_menu_button((Widget) view_pane, v->name());
    }
}

// ----------------------------------------------------------------------------
//
static void view_renamed_cb(void *view_pane, void *view)
{
  View *v = (View *) view;
  if (v->is_top_level_window())
    update_view_menu((Widget) view_pane, v->session());
}

// ----------------------------------------------------------------------------
//
static void update_view_menu(Widget pane, Session &s)
{
  WinSys &ws = s.window_system();
  ws.delete_menu_buttons(pane);
  List vlist = s.project().top_level_views();
  for (int vi = 0 ; vi < vlist.size() ; ++vi)
    {
      View *v = (View *) vlist[vi];
      ws.add_menu_button(pane, v->name(), show_view_cb, v);
    }
}

// ----------------------------------------------------------------------------
//
static void show_view_cb(Widget, CB_Data view, CB_Data)
  { ((View *) view)->show_view(true); }

// ----------------------------------------------------------------------------
//
static void create_help_menu(Session &s, Widget menubar)
{
  WinSys &ws = s.window_system();
  Widget pane = ws.add_menu(menubar, "helpMenuButton", "helpMenu");

  menu_entry(s, pane, "manual", "he");
  menu_entry(s, pane, "intro", "hi");
  menu_entry(s, pane, "files", "hf");
  menu_entry(s, pane, "spectra", "hs");
  menu_entry(s, pane, "peaks", "hp");
  menu_entry(s, pane, "extensions", "hx");
}

// ----------------------------------------------------------------------------
//
void add_extension_menu_entry(Session &s,
			      const Stringy &menu_text, const Stringy &accel)
{
  Dialog_Table &dt = s.dialog_table();
  Widget mext_menu = dt.get_dialog("main_extension_menu");
  add_menu_path(s, mext_menu, menu_text, accel);

  popup_view_menu(s);	// Make sure view popup menu has been created.
  Widget vext_menu = dt.get_dialog("view_extension_menu");
  add_menu_path(s, vext_menu, menu_text, accel);
}

// ----------------------------------------------------------------------------
//
void remove_extension_menu_entry(Session &s, const Stringy &menu_text)
{
  Dialog_Table &dt = s.dialog_table();
  Widget mext_menu = dt.get_dialog("main_extension_menu");
  delete_menu_path(s, mext_menu, menu_text);

  popup_view_menu(s);	// Make sure view popup menu has been created.
  Widget vext_menu = dt.get_dialog("view_extension_menu");
  delete_menu_path(s, vext_menu, menu_text);
}

// ----------------------------------------------------------------------------
// Make a menu entry at the end of a chain of cascaded menus.
// The path has the form cascade1/cascade2/.../buttonname.
// The cascaded menus are created if they do not exist.
//
static void add_menu_path(Session &s, Widget pane,
			  const Stringy &path, const Stringy &accelerator)
{
  Stringy rest;
  Stringy name = first_word(path, "/", &rest);
  if (rest.is_empty())
    menu_entry(s, pane, name, accelerator);
  else
    {
      WinSys &ws = s.window_system();
      Widget p = ws.find_cascade_pane(pane, name);
      if (p == NULL)
	p = ws.cascade_pane(pane, name);
      add_menu_path(s, p, rest, accelerator);
    }
}

// ----------------------------------------------------------------------------
// The path has the form cascade1/cascade2/.../buttonname.
// The cascaded menus are deleted if they are empty.
//
static void delete_menu_path(Session &s, Widget pane, const Stringy &path)
{
  WinSys &ws = s.window_system();
  Stringy rest;
  Stringy name = first_word(path, "/", &rest);
  if (rest.is_empty())
    ws.delete_menu_button(pane, name);
  else
    {
      Widget p = ws.find_cascade_pane(pane, name);
      if (p == NULL)
	return;
      delete_menu_path(s, p, rest);
      if (ws.is_pane_empty(p))
	ws.delete_cascade(pane, name);
    }
}

// ----------------------------------------------------------------------------
//
static void menu_entry(Session &s, Widget pane,
		       const Stringy &name, const Stringy &accelerator)
{
  Menu_Entry *me = new Menu_Entry(s.command_dispatcher(), accelerator);
  WinSys &ws = s.window_system();
  ws.add_menu_button(pane, name, accelerator, accelerator_cb, me);
  ws.add_event_callback(destroy_event, pane, destroy_menu_callback_cb, me);
}

// ----------------------------------------------------------------------------
//
Menu_Entry::Menu_Entry(Command_Dispatcher &cd, const Stringy &accel)
  : command_dispatcher(cd)
{
  this->accelerator = accel;
}

// ----------------------------------------------------------------------------
//
static void destroy_menu_callback_cb(Widget, CB_Data me, CB_Data)
{
  delete (Menu_Entry *) me;
}

// ----------------------------------------------------------------------------
//
Widget popup_view_menu(Session &s)
{
  Dialog_Table &dt = s.dialog_table();
  if (dt.get_dialog("popup_view_menu") == NULL)
    {
      WinSys &ws = s.window_system();
      Widget menu = ws.create_menu_pane(ws.main_widget(), "cmdMenu");
      dt.set_dialog("popup_view_menu", menu);

      ws.menu_label(menu, "popupHeading");

      ws.menu_separator(menu, "separator");

      Widget assignment_pane = ws.cascade_pane(menu, "assignmentCascade");
      menu_entry(s, assignment_pane, "assigncopy", "ac");
      menu_entry(s, assignment_pane, "assigndelete", "aD");
      menu_entry(s, assignment_pane, "assignpaste", "aP");
      menu_entry(s, assignment_pane, "assignpastelabel", "ap");
      menu_entry(s, assignment_pane, "xdiagassigncopy", "aX");
      menu_entry(s, assignment_pane, "assign", "at");
      menu_entry(s, assignment_pane, "resassign", "ra");
      menu_entry(s, assignment_pane, "labelpeaks", "lb");
      menu_entry(s, assignment_pane, "unlabelpeaks", "lB");

      Widget alias_pane = ws.cascade_pane(menu, "aliasCascade");
      menu_entry(s, alias_pane, "aliasaddw1", "a1");
      menu_entry(s, alias_pane, "aliasaddw2", "a2");
      menu_entry(s, alias_pane, "aliasaddw3", "a3");
      menu_entry(s, alias_pane, "aliasaddw4", "a4");
      menu_entry(s, alias_pane, "aliassubw1", "A1");
      menu_entry(s, alias_pane, "aliassubw2", "A2");
      menu_entry(s, alias_pane, "aliassubw3", "A3");
      menu_entry(s, alias_pane, "aliassubw4", "A4");
      menu_entry(s, alias_pane, "aliasdownw1", "d1");
      menu_entry(s, alias_pane, "aliasdownw2", "d2");
      menu_entry(s, alias_pane, "aliasdownw3", "d3");
      menu_entry(s, alias_pane, "aliasdownw4", "d4");
      menu_entry(s, alias_pane, "aliasupw1", "u1");
      menu_entry(s, alias_pane, "aliasupw2", "u2");
      menu_entry(s, alias_pane, "aliasupw3", "u3");
      menu_entry(s, alias_pane, "aliasupw4", "u4");
      menu_entry(s, alias_pane, "unalias", "ua");

      Widget axes_pane = ws.cascade_pane(menu, "axesCascade");
      menu_entry(s, axes_pane, "ppm", "xp");
      menu_entry(s, axes_pane, "hz", "xh");
      menu_entry(s, axes_pane, "data", "xd");
      menu_entry(s, axes_pane, "origin", "xO");
      menu_entry(s, axes_pane, "roll", "xr");
      menu_entry(s, axes_pane, "transpose", "xx");
      menu_entry(s, axes_pane, "atoms", "xa");
      menu_entry(s, axes_pane, "shiftaxes", "st");

      Widget crosshair_pane = ws.cascade_pane(menu, "crosshairsCascade");
      menu_entry(s, crosshair_pane, "xhair", "hh");
      menu_entry(s, crosshair_pane, "transxhair", "ht");
      menu_entry(s, crosshair_pane, "otherxhair", "ho");

      Widget edit_pane = ws.cascade_pane(menu, "editCascade");
      menu_entry(s, edit_pane, "ocopy", "oc");
      menu_entry(s, edit_pane, "opaste", "op");
      menu_entry(s, edit_pane, "xocopy", "oX");
      menu_entry(s, edit_pane, "undo", "eu");

      Widget ext_menu = ws.cascade_pane(menu, "extensionsCascade");
      dt.set_dialog("view_extension_menu", ext_menu);

      Widget integrate_pane = ws.cascade_pane(menu, "integrateCascade");
      menu_entry(s, integrate_pane, "integrate", "it");
      menu_entry(s, integrate_pane, "box", "ib");
      menu_entry(s, integrate_pane, "ellipse", "ie");
      menu_entry(s, integrate_pane, "gaussian", "ig");
      menu_entry(s, integrate_pane, "lorentzian", "il");
      menu_entry(s, integrate_pane, "integpeak", "pi");

      Widget peak_pane = ws.cascade_pane(menu, "peakCascade");
      menu_entry(s, peak_pane, "center", "pc");
      menu_entry(s, peak_pane, "color", "cr");
      menu_entry(s, peak_pane, "linewidth", "pe");
      menu_entry(s, peak_pane, "group", "pg");
      menu_entry(s, peak_pane, "integpeak", "pi");
      menu_entry(s, peak_pane, "selectinvert", "pI");
      menu_entry(s, peak_pane, "labelpeak", "pl");
      menu_entry(s, peak_pane, "listselected", "sl");
      menu_entry(s, peak_pane, "lock", "pk");
      menu_entry(s, peak_pane, "osize", "oz");
      menu_entry(s, peak_pane, "peaklist", "lt");
      menu_entry(s, peak_pane, "resassign", "ra");
      menu_entry(s, peak_pane, "selectaliased", "pA");
      menu_entry(s, peak_pane, "selectall", "pa");
      menu_entry(s, peak_pane, "selectfassign", "pF");
      menu_entry(s, peak_pane, "selectpassign", "pP");
      menu_entry(s, peak_pane, "selectuassign", "pN");
      menu_entry(s, peak_pane, "selectcolor", "pC");
      menu_entry(s, peak_pane, "selectintra", "pR");
      menu_entry(s, peak_pane, "selectseq", "pS");
      menu_entry(s, peak_pane, "selectmedium", "pM");
      menu_entry(s, peak_pane, "selectlong", "pL");
      menu_entry(s, peak_pane, "unintegrate", "pU");
      menu_entry(s, peak_pane, "unlock", "pu");
      menu_entry(s, peak_pane, "height", "ph");

      Widget region_pane = ws.cascade_pane(menu, "regionCascade");
      menu_entry(s, region_pane, "goto", "rgX");
      menu_entry(s, region_pane, "region", "rt");

      Widget resonance_pane = ws.cascade_pane(menu, "resonanceCascade");
      menu_entry(s, resonance_pane, "reslist", "rl");
      menu_entry(s, resonance_pane, "renamereson", "rr");
      menu_entry(s, resonance_pane, "deleteres", "dr");
      menu_entry(s, resonance_pane, "resassign", "ra");
      menu_entry(s, resonance_pane, "resonancew1", "r1");
      menu_entry(s, resonance_pane, "resonancew2", "r2");
      menu_entry(s, resonance_pane, "resonancew3", "r3");
      menu_entry(s, resonance_pane, "resonancew4", "r4");
      menu_entry(s, resonance_pane, "predefined", "pd");

      Widget show_pane = ws.cascade_pane(menu, "showCascade");
      menu_entry(s, show_pane, "showall", "os");
      menu_entry(s, show_pane, "showpeak", "ps");
      menu_entry(s, show_pane, "showpeakgp", "xs");
      menu_entry(s, show_pane, "showlabel", "ls");
      menu_entry(s, show_pane, "showline", "is");
      menu_entry(s, show_pane, "showgrid", "gs");

      Widget tools_pane = ws.cascade_pane(menu, "toolsCascade");
      menu_entry(s, tools_pane, "assign", "at");
      menu_entry(s, tools_pane, "assigncopy", "co");
      menu_entry(s, tools_pane, "assignguess", "ag");
      menu_entry(s, tools_pane, "assigntable", "tb");
      menu_entry(s, tools_pane, "contour", "ct");
      menu_entry(s, tools_pane, "crosshair", "tt");
      menu_entry(s, tools_pane, "integrate", "it");
      menu_entry(s, tools_pane, "midas", "mi");
      menu_entry(s, tools_pane, "region", "rt");
      menu_entry(s, tools_pane, "newlist", "nl");
      menu_entry(s, tools_pane, "note", "nt");
      menu_entry(s, tools_pane, "ornament", "ot");
      menu_entry(s, tools_pane, "overlay", "ol");
      menu_entry(s, tools_pane, "peaklist", "lt");
      menu_entry(s, tools_pane, "peakpick", "kt");
      menu_entry(s, tools_pane, "predefined", "pd");
      menu_entry(s, tools_pane, "Preferences", "pf");
      menu_entry(s, tools_pane, "Print", "pt");
      menu_entry(s, tools_pane, "renamereson", "rr");
      menu_entry(s, tools_pane, "reslist", "rl");
      menu_entry(s, tools_pane, "spectrum", "st");
      menu_entry(s, tools_pane, "synchronize", "yt");
      menu_entry(s, tools_pane, "projectviews", "pv");
      menu_entry(s, tools_pane, "view", "vt");

      Widget view_pane = ws.cascade_pane(menu, "viewCascade");
      menu_entry(s, view_pane, "axisscale", "vs");
      menu_entry(s, view_pane, "centerview", "vc");
      menu_entry(s, view_pane, "contourscale", "vC");
      menu_entry(s, view_pane, "dupview", "vd");
      menu_entry(s, view_pane, "hideview", "vh");
      menu_entry(s, view_pane, "deleteview", "vD");
      menu_entry(s, view_pane, "ortho", "ov");
      menu_entry(s, view_pane, "peakinfo", "vp");
      menu_entry(s, view_pane, "refresh", "vr");
      menu_entry(s, view_pane, "renumber", "rv");
      menu_entry(s, view_pane, "resonance", "vR");
      menu_entry(s, view_pane, "resfilter", "rf");
      menu_entry(s, view_pane, "rightlabels", "lr");
      menu_entry(s, view_pane, "scrollbars", "vb");
      menu_entry(s, view_pane, "slice", "vS");
      menu_entry(s, view_pane, "slicescale", "sa");
      menu_entry(s, view_pane, "slicesubtract", "sf");
      menu_entry(s, view_pane, "subtractfit", "vf");
      menu_entry(s, view_pane, "unoverlap", "lu");
      menu_entry(s, view_pane, "view", "vt");
      menu_entry(s, view_pane, "viewdepth", "vz");

      Widget zoom_pane = ws.cascade_pane(menu, "zoomCascade");
      menu_entry(s, zoom_pane, "zoomin", "zi");
      menu_entry(s, zoom_pane, "zoomout", "zo");
      menu_entry(s, zoom_pane, "zoomdown", "zd");
      menu_entry(s, zoom_pane, "zoomup", "zu");
      menu_entry(s, zoom_pane, "zoomfull", "zf");
      menu_entry(s, zoom_pane, "zoomprev", "zp");
      menu_entry(s, zoom_pane, "zoomnext", "zn");
      menu_entry(s, zoom_pane, "zoomremember", "zr");
    }

  return (Widget) dt.get_dialog("popup_view_menu");
}

// ----------------------------------------------------------------------------
//
static void accelerator_cb(Widget, CB_Data menu_entry, CB_Data)
{
  Menu_Entry *me = (Menu_Entry *) menu_entry;
  me->command_dispatcher.execute_menu_accelerator(me->accelerator);
}
